﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief ショップページを開くのに利用される型や定数、引数の宣言(内部開発者向け)
 */

#pragma once

#include <nn/account/account_Types.h>
#include <nn/applet/applet.h>
#include <nn/ec/detail/ec_ShowShopPageTypes.h>

namespace nn { namespace ec { namespace detail {

class ShowShopPageArg
{
public:
    /**
     * @brief 情報を格納するために使用するデータのサイズ(バイト)です。
     */
    static const size_t DataSize = 8192;

    /**
     * @brief NULL 終端を含むリクエストできる URL の最大バイト数です。
     */
    static const size_t RequestRelativeUrlMaxLength = 3072;

    /**
     * @brief NULL 終端を含むコールバックとして指定できる URL の最大バイト数です。
     */
    static const size_t CallbackUrlMaxLength = 1024;

    /**
     * @brief NULL 終端を含むコールバック可能として指定できる URL の最大バイト数です。
     */
    static const size_t CallbackableUrlMaxLength = 1024;

    /**
     * @brief  ショップページを表示する為の最低限の情報を持つクラスを作成します。
     *
     * @param[in]   pRequestRelativeUrl 開く際に指定されるルートからの相対パスを指定してください。
     *
     * @details 指定する値は NULL 終端されている必要があります。
     *          RequestRelativeUrlMaxLength -1 文字以下にする必要があります。
     */
    explicit ShowShopPageArg(const char* pRequestRelativeUrl) NN_NOEXCEPT;

    /**
     * @brief コールバックURLを設定します。
     *
     * @param[in]   pCallbackUrl    コールバック URL を指定してください。
     *
     * @details 指定する値は NULL 終端されている必要があります。
     *          CallbackUrlMaxLength -1 文字以下にする必要があります。
     *          二度呼び出した場合、値は上書きされます。
     */
    void SetCallbackUrl(const char* pCallbackUrl) NN_NOEXCEPT;

    /**
     * @brief コールバック可能URLを設定します。
     *
     * @param[in]   pCallbackableUrl    コールバック可能 URL を指定してください。
     *
     * @details 指定する値は NULL 終端されている必要があります。
     *          CallbackableUrlMaxLength -1 文字以下にする必要があります。
     *          二度呼び出した場合、値は上書きされます。
     */
    void SetCallbackableUrl(const char* pCallbackableUrl) NN_NOEXCEPT;

    /**
     * @brief Uidを設定します。
     *
     * @param[in]   uid アカウントの Uid を指定してください。
     *
     * @details 指定する値は、アカウントシステムに登録されているユーザーの Uid または
     *          nn::account::InvalidUid である必要があります。
     *          二度呼び出した場合、値は上書きされます。
     */
    void SetUid(const nn::account::Uid& uid) NN_NOEXCEPT;

    /**
     * @brief ページの背景を設定します。
     *
     * @param[in]   backgroundKind  背景を指定してください。
     *
     * @details 二度呼び出した場合、値は上書きされます。
     */
    void SetBackgroundKind(const ShopBackgroundKind backgroundKind) NN_NOEXCEPT;

    /**
     * @internal
     *
     * @brief RequelstRelativeUrlの格納先アドレスを取得します。
     *
     * @return RequelstRelativeUrlの格納先アドレスを返します。
     *
     * @details 取得したバッファへの書き込みは NULL 終端する必要があります。
     *          RequestRelativeUrlMaxLength -1 文字以下にする必要があります。
     */
    uint8_t* GetRequestRelativeUrlBuffer() NN_NOEXCEPT;

    /**
     * @internal
     *
     * @brief   ショップページを開く為の変数を一つのストレージにまとめ、ライブラリアプレットの入力チャンネルにプッシュします。
     *
     * @param[in]   libraryAppletHandle     対象のライブラリアプレットのハンドルを指定します。
     *
     */
    void PushToInChannel(nn::applet::LibraryAppletHandle handle) const NN_NOEXCEPT;

private:
    NN_ALIGNAS(4) uint8_t m_Data[DataSize];
};

//--------------------------------------------------------------------------
/**
 * @brief ショップページを開いた際の戻り値の情報を構築するクラスです。
 */
class ShopPageReturnValue
{
public:
    /**
     * @brief NULL 終端を含む最後に開いていたページの URL の最大バイト数です。
     */
    static const size_t LastUrlMaxLength = 4096;

    /**
     * @brief 有効な情報を持たない状態でクラスを生成します。
     */
    ShopPageReturnValue() NN_NOEXCEPT;

    /**
     * @brief 終了理由を取得します。
     *
     * @return 終了理由を返します。
     */
    ShopExitReason GetShopExitReason() const NN_NOEXCEPT;

    /**
     * @brief 最後に開いていたページの URL を取得します。
     *
     * @return ページの URL を返します。
     *
     * @pre
     *     GetShopExitReason() == ShopExitReason_CallbackUrlReached
     */
    const char* GetLastUrl() const NN_NOEXCEPT;

    /**
     * @brief 最後に開いていたページの URL のバイト数(NULL 文字分含む)を取得します。
     *
     * @return URL のバイト数を返します。
     *
     * @pre
     *     GetShopExitReason() == ShopExitReason_CallbackUrlReached
     */
    size_t GetLastUrlSize() const NN_NOEXCEPT;

protected:
    uint64_t m_ExitReason;
    char m_LastUrl[LastUrlMaxLength];
    uint64_t m_LastUrlSize;
};

}}} // namespace nn::ec::detail
