﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

#include <nn/dns/parser/dns_ParserDependencies.h>
#include <nn/dns/parser/dns_ParserCommon.h>

/**
 * @file
 *
 * This file contains the definition of the @a QuestionIterator class
 * provided by the Dns Parser library.
 */

/**
 * @namespace nn::dns::parser
 * @brief DNSパーサライブラリの名前空間。
 */
namespace nn { namespace dns { namespace parser {

/**
 * @brief The @a QuestionIterator class is used to iterate over the
 * MessageSectionConstant::Question section of nn::dns::parser::Message
 * and creates nn::dns::parser::Question objects from the specified
 * section.
 */
class QuestionIterator
{
private:
    /**
     * @brief A pointer to the @ref Message.
     */
    const Message* m_pMessage;

    /**
     * @brief A pointer to the section start.
     */
    MemoryBlock m_Section;

    /**
     * @brief A pointer to the current location within the section.
     */
    const uint8_t* m_pCurrent;

    /**
     * @brief The number of records remaining.
     */
    uint8_t m_Count;

    /**
     * @brief This field is the current Question.
     */
    Question m_Question;

public:
    /**
     * @brief Constructor.
     */
    QuestionIterator();

    /**
     * @brief Copy constructor.
     *
     * @param rhs The @a QuestionIterator on the right hand side of
     * the expression.
     */
    QuestionIterator(const QuestionIterator& rhs);

    /**
     * @brief Destructor.
     */
    ~QuestionIterator();

    /**
     * @brief Assignment operator.
     *
     * @param rhs The @a QuestionIterator on the right hand side of
     * the expression.
     */
    QuestionIterator& operator=(const QuestionIterator& rhs);

    /**
     * @brief Initializes a QuestionIterator.
     *
     * @param[in] message The Message that contains the section.
     *
     * @param[in] section This constant defines the section to iterate
     * over. This paramater should always be
     * nn::dns::parser::MessageSectionConstant::Question.
     *
     * @return This function returns 1 on success, 0 if no questions
     * are present, and -1 on error. Errors include:
     *
     * - Message does not have a buffer.
     * - You specify the nn::dns::parser::MessageSectionConstant::Answer,
     * nn::dns::parser::MessageSectionConstant::Authority, or
     * nn::dns::parser::MessageSectionConstant::Additional section.
     */
    int Initialize(const Message& message, MessageSectionConstant section) NN_NOEXCEPT;

    /**
     * @brief This function gets the next record in the section.
     *
     * This function initializes the 'current' field. If there are
     * additional records then it initializes the record from the cursor
     * and sets the 'message' parameter.
     *
     * @return This function returns 0 if there are no records left then
     * it returns 1 if a single record was cached. If an error is
     * encountered such as exceeding the section boundary then
     * it returns -1.
     */
    int GetNext() NN_NOEXCEPT;

    /**
     * @brief This function gets the current cached @ref Question.
     *
     * @return A reference to the question.
     */
    Question& GetCurrent() NN_NOEXCEPT;
};

}}}; //nn::dns::parserr
