﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

#include <nn/dns/parser/dns_ParserDependencies.h>
#include <nn/dns/parser/dns_ParserCommon.h>

/**
 * @file
 *
 * This file contains the definition of the @a Message class provided
 * by the Dns Parser library.
 */

/**
 * @namespace nn::dns::parser
 * @brief DNSパーサライブラリの名前空間。
 */
namespace nn { namespace dns { namespace parser {

/**
 * @brief The @a Message class represents a DNS message as defined by
 * the RFC.
 *
 * @details
 * The @a Message class maps sections (header, question, answer,
 * authority, and additional) to a single contiguous buffer. It is
 * is geared toward reading messages but writing data back to the
 * buffer is also possible with some work. Please note that any
 * changes that affect the size of a record must also take into
 * account changes in downstream @ref Label pointers.
 *
 * To access the sections (question, answer, authority, additional)
 * you must use the @ref QuestionIterator or @ref RecordIterator but
 * header data is cached within the @a Message instance and is
 * accessed without using an iterator.
 */
class Message
{
private:
    /**
     * @brief m_pBuffer A pointer to the buffer that backs the Message.
     */
    const uint8_t* m_pBuffer;

    /**
     * @brief m_Size The size of the buffer that backs the Message.
     */
    size_t m_Size;

    /**
     * @brief Indicates the object size or if it is uninitialized.
     */
    size_t m_DirtySize;

    /**
     * @brief Points to the start and end of the header section in
     * the buffer.
     *
     * There can only be one Dns Header per message. In a FromBuffer
     * case then the 'pBuffer' is the start of this range.
     */
    MemoryBlock m_HeaderSection;

    /**
     * @brief If the question count is greater than 0 then this
     * contains the start and end of the question section.
     */
    MemoryBlock m_QuestionSection;

    /**
     * @brief If the answer count is greater than 0 then this
     * contains the start and end of the answer section.
     */
    MemoryBlock m_AnswerSection;

    /**
     * @brief If the authority count is greater than 0 then this
     * contains the start and end of the authority section.
     */
    MemoryBlock m_AuthoritySection;

    /**
     * @brief If the additional count is greater than 0 then this
     * contains the start and end of the additional section.
     */
    MemoryBlock m_AdditionalSection;

    /**
     * @brief A cached object that simplifies interacting with the
     * fixed header portion of the Message.
     *
     * ref nn::dns::parser::Message::ToBuffer uses this field to
     * populate the header.
     */
    Header m_Header;

public:

    /**
     * @brief Constructor.
     */
    Message();

    /**
     * @brief Copy constructor.
     *
     * @details This also copies the underlying buffer pointer and
     * so changes in the question, answer, authority, or additional
     * sections in one message will be reflected in the copied
     *
     * @param rhs The @a Message on the right hand side of the
     * expression.
     */
    Message(const Message& rhs);

    /**
     * @brief Destructor.
     */
    ~Message();

    /**
     * @brief Assignment operator.
     *
     * @details This also copies the underlying buffer pointer and
     * so changes in the question, answer, authority, or additional
     * sections in one message will be reflected in the copied
     *
     * @param rhs The object on the right hand side of the expression.
     */
    Message& operator=(const Message& rhs);

    /**
     * @brief Returns whether or not two Message objects are equal.
     *
     * @param[in] message The Message object to compare with this object.
     *
     * @return Returns true if both objects are content equivalent.
     */
    bool operator==(const Message& message) const NN_NOEXCEPT;

    /**
     * @brief Getter for m_pBuffer
     * @return The buffer pointer.
     */
    const uint8_t* GetBuffer() const NN_NOEXCEPT;

    /**
     * @brief Getter / setter for m_size.
     *
     * @return The buffer size.
     */
    size_t GetBufferSize() const NN_NOEXCEPT;

    /**
     * @brief Getter / setter for m_HeaderSection.
     *
     * @return A reference to the header section buffer.
     */
    MemoryBlock& GetHeaderSection() NN_NOEXCEPT;

    /**
     * @brief Getter / setter for m_QuestionSection.
     *
     * @return A reference to the question section buffer.
     */
    MemoryBlock& GetQuestionSection() NN_NOEXCEPT;

    /**
     * @brief Getter / setter for m_AnswerSection.
     *
     * @return A reference to the answer section buffer.
     */
    MemoryBlock& GetAnswerSection() NN_NOEXCEPT;

    /**
     * @brief Getter / setter for m_AuthoritySection.
     *
     * @return A reference to the authority section buffer.
     */
    MemoryBlock& GetAuthoritySection() NN_NOEXCEPT;

    /**
     * @brief Getter / setter for m_AdditionalSection
     *
     * @return A reference to the additional section buffer.
     */
    MemoryBlock& GetAdditionalSection() NN_NOEXCEPT;

    /**
     * @brief Getter / setter for m_Header.
     *
     * @return A reference to the cached header object.
     */
    Header& GetHeader() NN_NOEXCEPT;

    /**
     * @brief Initialize a Message.
     */
    void Initialize() NN_NOEXCEPT;

    /**
     * @brief Get the size of the Message for serialization.
     *
     * @return This function returns error or success:
     *          - -1 : An error occurred.
     *          - >0 : The size of the object.
     */
    ssize_t SizeOf() const NN_NOEXCEPT;

    /**
     * @brief Reads a @a Message from the contents of a buffer in the
     * standard RFC format.
     *
     * @details It is the reverse of @ref Message::ToBuffer().
     *
     * @param[in] pBuffer The buffer that contains @a Message bytes.
     *
     * @param[in] size The size in bytes left in the buffer.
     *
     * @return Returns the number of bytes processed or -1 on error.
     *
     * Error cases include a null buffer pointer or parsing error.
     */
    ssize_t FromBuffer(const uint8_t* pBuffer, size_t size) NN_NOEXCEPT;

    /**
     * @brief Writes a @a Message to a buffer in The RFC standard format.
     *
     * @details It is the reverse of @ref Message::FromBuffer().
     *
     * @param[in] pOutBuffer The buffer that receives @a Message bytes.
     *
     * @param[in] size The size of the buffer in bytes.
     *
     * @return Returns the number of bytes written to the buffer or
     * -1 on error.
     *
     * Error cases include null buffer pointer, a buffer of
     * insufficient size, or a parsing error.
     */
    ssize_t ToBuffer(uint8_t* const pOutBuffer, size_t size) const NN_NOEXCEPT;
};

}}}; // nn::dns::parser
