﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

#include <nn/dns/parser/dns_ParserDependencies.h>
#include <nn/dns/parser/dns_ParserCommon.h>

/**
 * @file
 *
 * This file contains the definition of the @a Label class provided
 * by the Dns Parser library.
 */

/**
 * @namespace nn::dns::parser
 * @brief DNSパーサライブラリの名前空間。
 */
namespace nn { namespace dns { namespace parser {

/**
 * @brief The @a Label class represents a DNS label which, as defined
 * in the RFC, is used to declare names.
 *
 * @details
 * The first two bits of the label indicate the type and only two are
 * assigned for use. At the time of this writing there are proposals
 * to extended the type field to to 8 bits in certain cases
 * but they have not been adopted by the Internet Society. Therefore
 * the dns parser only supports the two active types of labels.
 *
 * The first type of label is a normal label where the two most
 * significant bits of the first octet are both zero, the remaining 6
 * bits are treated as an unsigned length parameter, and an equal
 * number of subsequent bytes are treated as characters. The label
 * with a length of zero is the null label which represents the root
 * zone and when encountered the character sequence stops.
 * For example:
 *
 * <pre>
 *  0    1    2    3    4
 * ----+----+----+----+----
 * 0x03| 'w'| 'w'| 'w'|0x00
 * </pre>
 *
 * The second type of label is one where the two most significant bits
 * of the first octet are both one (decimal 3), the remaining 6 bits
 * are concatenated with the next octet and treated as an offset into
 * the message buffer. These kind of labels are called "pointer
 * labels" or "pointers" (which has nothing to do with a PTR record).
 * For example:
 *
 * <pre>
 *  0   1
 * ----+----
 * 0xC0|0x0C
 * </pre>
 *
 * In this case the normal label at byte offset 12 into the original
 * message contains the normal label bytes with its leading length
 * parameter.
 *
 * Combining these two types gives rise to its final form: a
 * sequence of characters that is assembled from both normal and
 * compressed labels that refer to various parts of the DNS message.
 * Consider a label that starts at offset 0XE0 and then points at a
 * different character sequence:
 *
 *
 * <pre>
 *  E0  E1    E2   E3   E4   E5
 * ----+----+----+----+----+----
 * 0x03| 'w'| 'w'| 'w'|0xC0|0x0C
 * </pre>
 *
 * <pre>
 *  0C   0D   0E   0F   10   11   12   13   14   15   16   17   18   19
 * ----+----+----+----+----+----+----+----+----+----+----+----+----+----
 * 0x08| 'n'| 'i'| 'n'| 't'| 'e'| 'n'| 'd'| 'o'|0x03| 'c'| 'o'| 'm'|0x00
 *
 * In this case the label character sequence is www.nintendo.com.
 * </pre>
 *
 */
class Label
{
private:
    /**
     * @brief A pointer to the @ref Message that contains this object.
     */
    const Message* m_pMessage;

    /**
     * @brief The block of memory that refers to the buffer
     * that created the @a Label with @ref Label::FromBuffer().
     */
    MemoryBlock m_Range;

    /**
     * @brief Indicates the object size or if it is uninitialized.
     */
    size_t m_DirtySize;

    /**
     * @brief The @a Label data.
     */
    const uint8_t* m_pData;

public:

    /**
     * @brief Constructor.
     */
    Label();

    /**
     * @brief Copy constructor.
     *
     * @param rhs The object on the right hand side of the expression.
     */
    Label(const Label& rhs);

    /**
     * @brief Destructor.
     */
    ~Label();

    /**
     * @brief Assignment operator.
     *
     * @param rhs The @a Label on the right hand side of the
     * expression.
     */
    Label& operator=(const Label& rhs);

    /**
     * @brief Returns true if two @a Label objects are equal.
     *
     * @param[in] rhs The @a Label on the right hand side of
     * the expression.
     *
     * @return Returns true if both objects are content equivalent.
     */
    bool operator==(const Label& rhs) const NN_NOEXCEPT;

    /**
     * @brief Getter for @a m_pMessage.
     *
     * @return A reference to the message pointer.
     */
    const Message* & GetMessage();

    /**
     * @brief Getter for for @a m_pMessage.
     *
     * @return A pointer to the label data.
     */
    const uint8_t* GetData() NN_NOEXCEPT;

    /**
     * @brief Initializes the @a Label.
     *
     * @details
     * Initializes a @a Label with the provided
     * nn::dns::parser::Message, zeroes out all fields, sets dirty
     * flag to zero, and point the @a Label at the the
     * nn::dns::parser::Message.
     *
     * @param[in] pMessage The provided @a Message.
     */
    void Initialize(const Message* pMessage) NN_NOEXCEPT;

    /**
     * @brief Get the size of a @a Label for serialization.
     *
     * @return Returns error or success:
     *          - -1 : An error occurred such a malformed label.
     *          - >0 : The size of the object.
     */
    ssize_t SizeOf() const NN_NOEXCEPT;

    /**
     * @brief Returns the size of a buffer that is able to contain a
     * string representation of the label. This includes the null
     * terminator. It is intended to be used with Label::ToString().
     *
     * @return The size in bytes of the string representation of the
     * label, including the null terminator.
     */
    ssize_t GetStringBufferSize() const NN_NOEXCEPT;

    /**
     * @brief Reads a @a Header from the contents of a buffer in the
     * standard RFC format.
     *
     * @details It is the reverse of @ref Label::ToBuffer().
     *
     * @param[in] pBuffer The buffer that contains @a Label bytes.
     *
     * @param[in] size The size in bytes left in the buffer.
     *
     * @return Returns the number of bytes processed or -1 on error.
     *
     * Error cases include a null buffer pointer or a parsing error.
     */
    ssize_t FromBuffer(const uint8_t* pBuffer, size_t size) NN_NOEXCEPT;

    /**
     * @brief Writes a @a Label to a buffer in standard RFC format.
     *
     * @details It is the reverse of @ref Label::FromBuffer().
     *
     * @param[out] pOutBuffer The buffer that receives @a Label bytes.
     *
     * @param[in] size The size of the buffer in bytes.
     *
     * @return Returns the number of bytes processed or -1 on error.
     *
     * Error cases include null buffer pointer, a buffer of
     * insufficient size, or a parsing error.
     */
    ssize_t ToBuffer(uint8_t* const pOutBuffer, size_t size) const NN_NOEXCEPT;

    /**
     * @brief Creates a Label from a domainname string.
     *
     * @details
     *
     * - The label bytes are copied from the string.
     * - The string is composed in dotted-name format (i.e. "a",
     * "a.bb", "a.bb.cc").
     * - The length of each dotted string segment must be at least 1
     * - character (i.e. "a..bb" is invalid).
     * - The the length of each segment cannot exceed
     * - nn::dns::parser::DnsParserConstant::HostnameMaxStringLength
     * characters.
     * - The total length of the string cannot exceed
     * nn::dns::parser::DnsParserConstant::DomainnameMaxStringLength.
     * - If you use this function and want to create a label from a
     * buffer then you must manually set the 'message' field.
     *
     * @param[out] pOutBuffer The buffer that receives @a Label bytes.
     *
     * @param[in] size The size of the buffer in bytes.
     *
     * @param[in] pString A C-style string such as www.nintendo.com
     * that represents a host or domain name.
     *
     * @return On success returns the number of bytes written.
     * This will always be equal to the string length returned from
     * strlen(pString) + DnsLabelConstant::StringLabelPadding (2). It
     * returns -1 on error.
     */
    static ssize_t ToBufferFromDomainNameString(uint8_t * const pOutBuffer,
                                                size_t size,
                                                const char* pString) NN_NOEXCEPT;

    /**
     * @brief Creates a Label from a hostname string.
     *
     * @details
     * - The label bytes are copied from the string.
     * - The the length of the string cannot exceed
     * nn::dns::parser::LibraryConstant::HostnameMaxStringLength
     * characters.
     * - LDH rules are checked in the following manner:
     *
     * -- 1) Letters (isalpha) and digits (isdigit) are allowed
     *       anywhere in the string.
     * -- 2) Characters that are not a letters or digits are not
     *       allowed anywhere.
     * -- 3) The hyphen is not allowed at the beginning or end.
     *
     * @param[out] pOutBuffer The buffer that receives @a Label bytes.
     *
     * @param[in] size The size of the buffer in bytes.
     *
     * @param[in] pString A C-style hostname string such "foo"
     * that represents a host name.
     *
     * @return On success returns the number of bytes written.
     * This will always be equal to the string length returned from
     * strlen(pString) + DnsLabelConstant::StringLabelPadding (2). It
     * returns -1 on error.
     */
    static ssize_t ToBufferFromHostnameString(uint8_t * const pOutBuffer,
                                              size_t size,
                                              const char* pString) NN_NOEXCEPT;

    /**
     * @brief Creates a C-style string from a Label object.
     *
     * @param[out] pOutString The string buffer that receives a
     * string representation of the label.
     *
     * @param[in] bufferSize The size of the string buffer including
     * the null terminator.
     *
     * @return Returns the number of characters written to the string
     * not including the null terminator.
     */
    ssize_t ToString(char* pOutString, size_t bufferSize) const NN_NOEXCEPT;

    /**
     * @brief Checks if all compressed label pointers are valid.
     * Specifically it checks that there are no cycles in the pointer
     * graph.
     *
     * @return Whether or not the Label is valid.
     */
    bool IsValid() const;
};

}}}; // nn::dns::parser
