﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

/**
 * @file
 *
 * This file contains constants that are provided by the Dns Parser
 * library. It is intentionally not an exhaustive list of all
 * possible DNS constants.
 */


/**
 * @namespace nn::dns::parser
 * @brief DNSパーサライブラリの名前空間。
 */
namespace nn { namespace dns { namespace parser {

/**
 * @enum nn::socket::dns::parser::LibraryConstant
 *
 * @brief This set of constants are used within the context of
 * the parser library.
 */
enum class LibraryConstant : uint32_t
{
    /**
     * @brief Used to create an inaddr-arpa domain name string buffer.
     */
    InaddrArpaStringBufferSize = 29,

    /**
     * @brief Indicates that the object is dirty.
     */
    ObjectIsDirty = 0,

    /**
     * @brief The maximum length of a hostname.
     */
    HostnameMaxStringLength = 63,

    /**
     * @brief The maximum size of a domain name string.
     */
    DomainNameMaxStringLength = 253,
};

/**
 * @enum nn::socket::dns::parser::HeaderConstant
 *
 * @brief Constants used within with the Header class.
 */
enum class HeaderConstant : uint16_t
{
    /**
     * @brief The fixed size of the DNS header.
     */
    Size = 12,
};

/**
 * @enum nn::socket::dns::parser::HeaderFlagsConstant
 *
 * @brief Constants used within with the Header class.
 */
enum class HeaderFlagsConstant : uint16_t
{
    /**
     * @brief Used to test a DnsHeaderFlags bit.
     */
    True = 1,

    /**
     * @brief Used to test a DnsHeaderFlags bit.
     */
    False = 0,

    /**
     * @brief Shift used to get the Response Code from the
     * @ref Header::GetFlags() value with
     * @ref nn::dns::parser::HeaderFlagsConstant::ResponseCodeMask.
     */
    ResponseCodeShift = 0,

    /**
     * @brief Mask used with
     * @ref nn::dns::parser::HeaderFlagsConstant::ResponseCodeShift
     * to get the Response Code from the @ref Header::GetFlags() value.
     */
    ResponseCodeMask = 15,

    /**
     * @brief Shift used with
     * @ref nn::dns::parser::HeaderFlagsConstant::IsNonAuthenticatedAcceptableMask
     * to get the "is non-authenticated data acceptable" bit from the
     * @ref Header::GetFlags() value.
     */
    IsNonAuthenticatedAcceptableShift = 4,

    /**
     * @brief Mask used with
     * @ref nn::dns::parser::HeaderFlagsConstant::IsNonAuthenticatedAcceptableShift
     * to get the "is non-authenticated data acceptable" bit from the
     * @ref Header::GetFlags() value.
     */
    IsNonAuthenticatedAcceptableMask = 1,

    /**
     * @brief Shift used
     * @ref nn::dns::parser::HeaderFlagsConstant::IsResponseAuthenticatedMask
     * to get the "is the response authenticated?" bit from the
     * @ref Header::GetFlags() value.
     */
    IsResponseAuthenticatedShift = 5,

    /**
     * @brief Mask used with
     * @ref nn::dns::parser::HeaderFlagsConstant::IsResponseAuthenticatedShift
     * to get the "is the response authenticated?" bit from the
     * @ref Header::GetFlags() value.
     */
    IsResponseAuthenticatedMask = 1,

    /**
     * @brief Shift used with
     * @ref nn::dns::parser::HeaderFlagsConstant::ZeroMask to get the
     * reserved "zero" bit from the @ref Header::GetFlags() value.
     */
    ZeroShift = 6,

    /**
     * @brief Mask used with
     * @ref nn::dns::parser::HeaderFlagsConstant::ZeroShift to get the
     * reserved "zero" bit from the @ref Header::GetFlags() value.
     */
    ZeroMask = 1,

    /**
     * @brief Shift used with
     * @ref nn::dns::parser::HeaderFlagsConstant::IsResponseRecursionAvailableMask
     * to get the "is recursion available from the server" bit from
     * the @ref Header::GetFlags() value.
     */
    IsResponseRecursionAvailableShift = 7,

    /**
     * @brief Mask used with
     * @ref nn::dns::parser::HeaderFlagsConstant::IsResponseRecursionAvailableShift
     * to get the "is recursion available" bit from the
     * @ref Header::GetFlags() value.
     */
    IsResponseRecursionAvailableMask = 1,

    /**
     * @brief Shift used with
     * @ref nn::dns::parser::HeaderFlagsConstant::IsRecursionDesiredMask
     * to get the "recursion is desired for this request" bit from
     * the @ref Header::GetFlags() value.
     */
    IsRecursionDesiredShift = 8,

    /**
     * @brief Mask used with
     * @ref nn::dns::parser::HeaderFlagsConstant::IsRecursionDesiredShift
     * to get the "recursion is desired for this request" bit from
     * the @ref Header::GetFlags() value.
     */
    IsRecursionDesiredMask = 1,

    /**
     * @brief Shift used with
     * @ref nn::dns::parser::HeaderFlagsConstant::IsMessageTruncatedMask
     * to get the "message is truncated" bit from the
     * @ref Header::GetFlags() value.
     */
    IsMessageTruncatedShift = 9,

    /**
     * @brief Mask used with
     * @ref nn::dns::parser::HeaderFlagsConstant::IsMessageTruncatedShift
     * to get the "message is truncated" bit from the
     * @ref Header::GetFlags() value.
     */
    IsMessageTruncatedMask = 1,

    /**
     * @brief Shift used with
     * @ref nn::dns::parser::HeaderFlagsConstant::IsResponseAuthorativeMask
     * to get the "message is authoritative" bit from the
     * @ref Header::GetFlags() value.
     */
    IsResponseAuthorativeShift = 10,

    /**
     * @brief Mask used with
     * @ref nn::dns::parser::HeaderFlagsConstant::IsResponseAuthorativeMask
     * to get the "message is authoritative" bit from the
     * @ref Header::GetFlags() value.
     */
    IsResponseAuthorativeMask = 1,

    /**
     * @brief Shift used with
     * @ref nn::dns::parser::HeaderFlagsConstant::OperationCodeMask
     * to get the Operation Code from the @ref Header::GetFlags()
     * value.
     */
    OperationCodeShift = 11,

    /**
     * @brief Mask used with
     * @ref nn::dns::parser::HeaderFlagsConstant::OperationCodeShift
     * to get the Operation Code from the @ref Header::GetFlags()
     * value.
     */
    OperationCodeMask = 15,

    /**
     * @brief Shift used with
     * @ref nn::dns::parser::HeaderFlagsConstant::IsResponseMask to
     * get the "this message is a response" bit from the
     * @ref Header::GetFlags() value.
     */
    IsResponseShift = 15,

    /**
     * @brief Mask used with @ref nn::dns::parser::HeaderFlagsConstant::IsResponseShift
     * to get the "this message is a response" bit from the
     * @ref Header::GetFlags() value.
     */
    IsResponseMask = 1,
};

inline HeaderFlagsConstant operator | (HeaderFlagsConstant a, HeaderFlagsConstant b) NN_NOEXCEPT
{
    return static_cast<HeaderFlagsConstant>(static_cast<int>(a) | static_cast<int>(b));
};

inline HeaderFlagsConstant operator |= (HeaderFlagsConstant &a, HeaderFlagsConstant b) NN_NOEXCEPT
{
    return a = a | b;
};

inline HeaderFlagsConstant operator & (HeaderFlagsConstant a, HeaderFlagsConstant b) NN_NOEXCEPT
{
    return static_cast<HeaderFlagsConstant>(static_cast<int>(a) & static_cast<int>(b));
};

inline HeaderFlagsConstant operator &= (HeaderFlagsConstant &a, HeaderFlagsConstant b) NN_NOEXCEPT
{
    return a = a & b;
};

inline HeaderFlagsConstant operator ^ (HeaderFlagsConstant a, HeaderFlagsConstant b) NN_NOEXCEPT
{
    return static_cast<HeaderFlagsConstant>(static_cast<int>(a) ^ static_cast<int>(b));
};

inline HeaderFlagsConstant operator ^= (HeaderFlagsConstant &a, HeaderFlagsConstant b) NN_NOEXCEPT
{
    return a = a ^ b;
};

inline HeaderFlagsConstant operator ~ (HeaderFlagsConstant a) NN_NOEXCEPT
{
    return static_cast<HeaderFlagsConstant>(~static_cast<int>(a));
};

inline HeaderFlagsConstant operator << (HeaderFlagsConstant a, HeaderFlagsConstant b) NN_NOEXCEPT
{
    return static_cast<HeaderFlagsConstant>(static_cast<int>(a) << static_cast<int>(b));
};

inline HeaderFlagsConstant operator >> (HeaderFlagsConstant a, HeaderFlagsConstant b) NN_NOEXCEPT
{
    return static_cast<HeaderFlagsConstant>(static_cast<int>(a) >> static_cast<int>(b));
};

/**
 * @enum nn::socket::dns::parser::OperationCodeConstant
 *
 * @brief This set of constants are used within the context of the
 * nn::dns::parser::Header Operation Code contained within the flags value.
 * For more information see RFC 2929 section 2.2.
 */
enum class OperationCodeConstant : uint32_t
{
    /**
     * @brief Standard queries such as recursive and iterative queries.
     * @details See RFC 1035.
     */
    StandardQuery = 0,

    /**
     * @brief Historical query type that mapped records to domain
     * names. Deprecated in RFC 3425.
     * @details See RFC 1035.
     */
    InverseQuery = 1,

    /**
     * @brief Receive status from a server.
     * @details See RFC 1035.
     */
    ServerStatus = 2,

    /**
     * @brief Notify a server when the zone changes. Used to
     * replicate zones between master and slave servers.
     * @details See RFC 1996.
     */
    Notify = 3,

    /**
     * @brief This operation code is set to query for server status.
     * @details See RFC 2136.
     */
    Update = 5,
};

/**
 * @enum nn::socket::dns::parser::ResponseCodeConstant
 *
 * @brief This set of constants are used within the context of
 * the nn::dns::parser::Header Response Code contained within the
 * flags value. For more information see RFC 2929 section 2.3.
 */
enum class ResponseCodeConstant : uint32_t
{
    /**
     * @brief No error condition.
     * @details See RFC 1035.
     */
    NoError = 0,

    /**
     * @brief The name server was unable to interpret the query.
     * @details See RFC 1035.
     */
    FormatError = 1,

    /**
     * @brief The name server was unable to process the query.
     * @details See RFC 1035.
     */
    ServerFailure = 2,

    /**
     * @brief From the perspective of the name server the domain is
     * Non existent.
     * @details See RFC 1035.
     */
    NonExistentDomain = 3,

    /**
     * @brief The name server does not support the requested kind of
     * query.
     * @details See RFC 1035.
     */
    NotImplemented = 4,

    /**
     * @brief The query was refused by the name server, usually for
     * policy reasons.
     * @details See RFC 1035.
     */
    Refused = 5,

    /**
     * @brief The name exists but it should not.
     * @details See RFC 2136.
     */
    YxDomain = 6,

    /**
     * @brief The Resource Record set exists but it should not.
     * @details See RFC 2136.
     */
    YxRRSet = 7,

    /**
     * @brief The Resource Record set that should exist does not.
     * @details See RFC 2136.
     */
    NxRRSet = 8,

    /**
     * @brief The server is not authorative for the queried zone.
     * @details See RFC 2136.
     */
    NotAuthorative = 9,

    /**
     * @brief The name is not contained in zone available for assignment.
     * @details See RFC 2136.
     */
    NotZone = 10,

    /**
     * @brief Bad OPT version.
     * @details See RFC 2671.
     */
    BadVersion = 16,

    /**
     * @brief TSIG signature failure.
     * @details See RFC 2845.
     */
    BadSignature = 16,

    /**
     * @brief The key is not recognized.
     * @details See RFC 2845.
     */
    BadKey = 17,

    /**
     * @brief The signature was out of the time window.
     * @details See RFC 2845.
     */
    BadTime = 18,

    /**
     * @brief Bad TKEY mode.
     * @details See RFC 2930.
     */
    BadMode = 19,

    /**
     * @brief Duplicate key name.
     * @details See RFC 2930.
     */
    BadName = 20,

    /**
     * @brief Algorithm not supported.
     * @details See RFC 2930.
     */
    BadAlgorithm = 21,
};

/**
 * @enum nn::socket::dns::parser::LabelConstant
 *
 * @brief This set of constants are used within the context of
 * the Label class.
 */
enum class LabelConstant : uint32_t
{
    /**
     * @brief Defines the maximum buffer size size of a single label
     * plus the leading length parameter.
     */
    LabelMaximumBufferSize = 1 + static_cast<int>(LibraryConstant::HostnameMaxStringLength),

    /**
     * @brief This constant is used to convert strings into labels.
     */
    StringLabelPadding = 2,

    /**
     * @brief Defines the maximum buffer size of a sequence of
     * labels in memory plus the leading length parameter and a byte
     * for the root zone / null terminator.
     */
    LabelSequenceMaximumBufferSize = static_cast<int>(LibraryConstant::DomainNameMaxStringLength) + StringLabelPadding,

    /**
     * @brief The first octet of every label determines the type of
     * label. Only the two most significant bits of the first octet
     * are used to determine the type of label. Therefore this
     * constant is used to shift the bits in the octet to the right.
     */
    TypeShift = 6,

    /**
     * @brief The mask to get the type parameter from the first octet
     * after shifting @ref nn::dns::parser::LabelConstant::TypeShift bits to the right.
     */
    TypeMask = 3,

    /**
     * @brief The offset mask to get the length parameter
     * in a normal type label or the first octet of the offset
     * in a compressed type label.
     */
    LengthOffsetMask = 63,

    /**
     * @brief The number of bits to left shift the first octet in a pointer offset.
     */
    OffsetLeftShift = 8,

    /**
     * @brief Used to indicate that the label is a 'normal' label;
     * i.e. it is a string of characters.
     */
    TypeNormal = 0,

    /**
     * @brief Used to indicate that the label is a 'compressed' label
     * or a pointer to a normal label somewhere within the message
     * buffer.
     */
    TypeCompressed = 3,

    /**
     * @brief The size of a compressed label, also known is a pointer,
     * is the same as the size of a uint16.
     */
    TypeCompressedSize = 2,

    /**
     * @brief For the sake of completeness the extended label
     * type is documented, although it is not handled.
     */
    TypeExtended = 1,
};

/**
 * @enum nn::socket::dns::parser::ClassConstant
 *
 * @brief This set of constants are used within the context of the
 * Question and Record "_class" field to determine
 * the class of network being queried.
 */
enum class ClassConstant : uint16_t
{
    /**
     * @brief This constant defines the Internet (IN) class.
     */
    Internet = 1,

    /**
     * @brief This constant defines the Chaos (CH) class.
     */
    Chaos = 3,

    /**
     * @brief This constant defines the Hesiod (HS) class.
     */
    Hesiod = 4,
};

/**
 * @enum nn::socket::dns::parser::TypeConstant
 *
 * @brief This set of constants are used within the context of the
 * Question and Record "type" field to determine the type of object
 * being queried.
 */
enum class TypeConstant : uint16_t
{
    /**
     * @brief Used for IPv4 host address queries.
     * @details See RFC 1035.
     */
    Address = 1,

    /**
     * @brief Delegates a DNS zone to use the given authoritative
     * name servers.
     * @details See RFC 1035.
     */
    Nameserver = 2,

    /**
     * @brief Alias of one name to another.
     * @details See RFC 1035.
     */
    CanonicalName = 5,

    /**
     * @brief Specifies authoritative information about a DNS zone
     * such as the primary name server, contact information,
     * serial number, and refresh rate.
     * @details See RFC 1035 & RFC 2308.
     */
    StartOfAuthority = 6,

    /**
     * @brief Pointer to a canonical name. Most commonly used to
     * perform reverse DNS lookups.
     */
    Pointer = 12,

    /**
     * @brief Mail exchange record. Maps to a list of message transfer
     * agents (MTA) for that domain.
     * @details See RFC 1035 & RFC 7505.
     */
    MailExchange = 15,

    /**
     * @brief Used for human and machine-readable text data.
     * @details See RFC 1035 & RFC 1464.
     */
    Text = 16,

    /**
     * @brief Define the responsible persons for a domain record type.
     * Usually an e-mail address with the '@' replaced by a '.'
     * @details See RFC 1183.
     */
    ResponsiblePersons = 17,

    /**
     * @brief Define the location of AFS server cell record type.
     * @details See RFC 1183.
     */
    AfsDb = 18,

    /**
     * @brief Defines the Signature record type used in
     * @a Signature(0) and @a Tkey.
     * designated RRSIG. Deprecated by RFC 3755.
     * @details See RFC 2535, RFC 2930, RFC 2931, and RFC 3755.
     */
    Signature = 24,

    /**
     * @brief Defines the Key record type. Used only for Signature(0)
     * and Tkey. Deprecated by RFC 3755 & RFC 4025.
     * @details See RFC 2535, RFC 2930, RFC 3445, RFC 3755, and
     * RFC 4025.
     */
    Key = 25,

    /**
     * @brief Defines the record IPv6 host address record type.
     * @details See RFC 3596.
     */
    Address6 = 28,

    /**
     * @brief Defines the Geographic location record type.
     * @details See RFC 1876.
     */
    Location = 29,

    /**
     * @brief Defines the service locator record type.
     * @details See RFC 2782.
     */
    Service = 33,

    /**
     * @brief Defines the Naming Authority Pointer record type.
     * @details See RFC 3403.
     */
    NamingAuthorityPointer = 35,

    /**
     * @brief Defines the Key Exchanger Record type.  Used with some
     * cryptographic systems to identify a key management agent for
     * the associated domain-name.
     * @details See RFC 2230.
     */
    KeyExchangerRecord = 36,

    /**
     * @brief Defines Certificate record; Stores PKIX, SPKI, PGP, etc.
     * @details See RFC 4398.
     */
    Certificate = 37,

    /**
     * @brief Defines the Delegation name record - an alias for a
     * name and its subdomains, unlike a @a CanonicalName which is an
     * alias for a single record.
     * details See RFC 6672.
     */
    DelegationNameRecord = 39,

    /**
     * @brief Defines the "Option" record type. It is a "pseudo DNS
     * record type" that is needed to support EDNS.
     * @details See RFC 6891.
     */
    Option = 41,

    /**
     * @brief Defines the Address Prefix List record, which specifies
     * lists of address ranges for various address families.
     * @details See RFC 3123. Please note this option is experimental.
     */
    AddressPrefixList = 42,

    /**
     * @brief Defines the Delegation signer record, which is used
     * to identify the DNSSEC signing key of a delegated zone.
     * @details See RFC 4034.
     */
    DelegationSigner = 43,

    /**
     * @brief Defines the SSH Public Key Fingerprint record, which is
     * used to publish public host key fingerprints in the DNS to aid
     * authenticating a host.
     * @details See RFC 4255.
     */
    SshPublicKeyFingerprint = 44,

    /**
     * @brief Defines the IPSEC Key record that can be used with IPSec.
     * @details see RFC 4025.
     */
    IpSecKey = 45,

    /**
     * @brief Defines the Signature for a DNSSEC-secured record set.
     * Uses the same format as the @a Signature record.
     * @details See RFC 4034.
     */
    DnsSecSignature = 46,

    /**
     * @brief Defines the Next Secure Record. Part of DNSSEC. Uses the
     * same format as the obsolete Next record.
     * @details See RFC 4034.
     */
    NextSecureRecord = 47,

    /**
     * @brief Defines the key record used in DNSSEC. Uses the same
     * format as the @ref TypeConstant::Key record.
     * @details See RFC 4034.
     */
    DnsKey = 48,

    /**
     * @brief Defines the DHCP identifier option. Used in conjunction
     * with the FQDN option to DHCP.
     * @details See RFC 4701.
     */
    DhcpId = 49,

    /**
     * @brief Defines the Next Secure record (version 3) which is an
     * extension to DNSSEC that allows proof of nonexistence for a
     * name without permitting zonewalking.
     * @details See RFC 5155.
     */
    NextSecureRecordVer3 = 50,

    /**
     * @brief Defines the Next Secure Record (version 3) parameters
     * record for use with @a NextSecureRecordv3.
     * @details See RFC 5155.
     */
    NextSecureRecordVer3Params = 51,

    /**
     * @brief Defines the record type for DNS-based Authentication of
     * Named Entities (DANE) method which is used to associate a
     * TLS server certificate or public key with the domain name where
     * the record is found.
     * @details see RFC 6698.
     */
    TransportLayerSecurityAuthentication = 52,

    /**
     * @brief Defines the Host Identity Protocol, which is a method
     * of separating the end-point identifier and locator roles of IP
     * addresses.
     * @details See RFC 8005.
     */
    HostIdentityProtocol = 55,

    /**
     * @brief Defines the Child Delegation Signer record for transfer
     * to parent.
     * @details See RFC 7344.
     */
    ChildDelegationSigner = 59,

    /**
     * @brief Defines the Child DNS Key record type, which is a child
     * copy of a @a DnsKey record for transfer to the parent.
     * @details See RFC 7344.
     */
    ChildDnsKey = 60,

    /**
     * @brief Defines a DNS-based Authentication of Named Entities
     * (DANE) method for publishing and locating OpenPGP public keys
     * in DNS for a specific e-mail address.
     * @details See RFC 7929.
     */
    OpenPgpKey = 61,

    /**
     * @brief Defines the Transaction Key record type, which is used
     * to provide keying material used with @a TransactionSignature
     * and encrypted under the public key in the accompanying @a Key
     * resource record.
     * @details See RFC 2930.
     */
    TransactionKey = 249,

    /**
     * @brief Defines the Transaction Signature record type which can
     * be used to authenticate dynamic updates as coming from an
     * approved client, or to authenticate responses as coming from an
     * approved recursive name server.
     * See RFC 2845.
     */
    TransactionSignature = 250,

    /**
     * @brief Defines the Incremental Zone Transfer which requests a
     * zone transfer of the given zone but only differences from a
     * previous serial number.
     * @details See RFC 1996.
     */
    IncrementalZoneTransfer = 251,

    /**
     * @brief Defines the Authoritative Zone Transfer record type,
     * which is used to transfer an entire zone from the master name
     * server to secondary name servers.
     * @details See RFC 1035.
     */
    AuthorativeZoneTransfer = 252,

    /**
     * @brief Defines the All cached records record type.
     * @details See RFC 1035.
     */
    AllCachedRecords = 255,

    /**
     * @brief Defines the Uniform Resource Identifier record type
     * which can be used for publishing mappings from hostnames to
     * URIs.
     * @details See RFC 7553.
     */
    UniformResourceIdentifier = 256,

    /**
     * @brief Defines the Certification Authority Authorization record
     * type which is used to constrain acceptable Certificate
     * Authorities for a host or domain.
     * @details See RFC 6844.
     */
    CertificationAuthorityAuthorization = 257,

    /**
     * @brief Defines the DNSSEC Trust Authorities record type, which
     * is part of a deployment proposal for DNSSec without a signed
     * DNS root. Uses the same format as the @a DelegationSigner
     * record.
     * @details See http://www.watson.org/~weiler/INI1999-19.pdf.
     */
    DnsSecTrustAuthorities = 32768,

    /**
     * @brief Defines the DNSSEC trust anchors outside of the DNS
     * delegation chain. Uses the same format as the same format as
     * the @a DelegationSigner record.
     * @details See RFC 4431.
     */
    DnsSecLookasideValidation = 32769,
};

/**
 * @enum nn::socket::dns::parser::MessageSectionConstant
 *
 * @brief These constants are used to refer to specific sections in
 * nn::dns::parser::Message.
 */
enum class MessageSectionConstant : uint8_t
{
    Header     = 0, ///< Header section.
    Question   = 1, ///< Question section.
    Answer     = 2, ///< Answer section.
    Authority  = 3, ///< Authority section.
    Additional = 4, ///< Additional section.
    Max        = 5, ///< Maximum number of sections in a DNS message.
};

/**
 * @enum nn::socket::dns::parser::RecordConstant
 *
 * @brief This enum defines constants used within the context of
 * the Record class.
 */
enum class RecordConstant : uint32_t
{
    /**
     * @brief The dnsparser does not handle all DNS Record
     * type data. Instead it handles specific kinds of data that
     * and the maximum data size that can be is 255 bytes. In the
     * future it might be necessary to more fully develop this
     * handling but it is out of scope when the dnsparser was
     * written.
     */
    DataMaximumSize = 255
};

}}}; //nn::dns::parser
