﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once
#include <nn/dhcps.h>

/**
 * @file
 *
 * @brief This is the public API for the DHCP server.
 */

/**
 * @namespace nn::dhcps
 */
namespace nn { namespace dhcps {


/*
 * Functions that control the DHCP server.
 */


/**
 * @brief Initializes the server.
 *
 * @details It must be called while the server is in the
 * @ref DhcpState::Uninitialized state.
 *
 * @param[in] parameters Contains the configuration parameters
 * necessary to initialize the server.
 *
 * @return Either a success or failure result code.
 */
Result Initialize(const UserConfiguration& parameters) NN_NOEXCEPT;

/**
 * @brief Gets the required memory size for the lease table as
 * initialized.
 *
 * @details It must be called after @ref Initialize() and before
 * @ref SetMemory().
 *
 * @param[out] pOutSize On success this parameter contains the size of
 * memory required for the lease table as initialized. Using nullptr
 * is not allowed.
 *
 * @return Either a success or failure result code.
 */
Result GetMemorySize(size_t* pOutSize) NN_NOEXCEPT;

/**
 * @brief Set the memory for the lease table as initialized.
 *
 * @details The size of the memory must be at least as large as the
 * 'pOutSize' value returned by @ref GetMemorySize().
 *
 * It must be called after @ref GetMemorySize() and before
 * @ref Start().
 *
 * @param[in] pMemory The block of memory allocated for the lease
 * table. Using nullptr is not allowed.
 *
 * @param[in] size The size of the provided memory in bytes.
 *
 * @return Either a success or failure result code.
 */
Result SetMemory(void* pMemory, size_t size) NN_NOEXCEPT;

/**
 * @brief Starts the server.
 *
 * @details It must be called after calling @ref SetMemory() and while
 * the server is in the @ref DhcpState::Stopped state.
 *
 * @return Either a success or failure result code.
 */
Result Start() NN_NOEXCEPT;

/**
 * @brief Stops the server.
 *
 * @details It must be called while the server is in the
 * @ref DhcpState::Running state.
 *
 * @return Either a success or failure result code.
 */
Result Stop() NN_NOEXCEPT;

/**
 * @brief Finalizes the server.
 *
 * @details It must be called while the server is in the
 * @ref DhcpState::Error, @ref DhcpState::Initialized,
 * @ref DhcpState::MemorySet, or @ref DhcpState::Stopped states. You
 * cannot finalize the server when it is in the
 * @ref DhcpState::Running state.
 *
 * @return Either a success or failure result code.
 */
Result Finalize() NN_NOEXCEPT;


/*
 * Query and Informational functions.
 */


/**
 * @brief Gets a copy of the lease for the provided client id hash
 * value.
 *
 * @details
 * - It must be called when the server is in the
 * @ref DhcpState::Running state.
 *
 * - Querying by an unrecognized client id hash returns an error.
 *
 * - Querying for an expired lease returns an error.
 *
 * - It is not legal to query for a client id hash equal to
 * @a ClientIdentifierUnknown (0). In this case this function
 * returns ResultClientIdentifierHashNotFound.
 *
 * - If a client does not provide a client identifier option then
 * the client identifier hash associated with the lease is
 * @a ClientIdentifierUnknown (0) and in this case you should use
 * @ref GetLeaseByEthernetMacAddress() to query by the MAC address
 * instead.
 *
 * @param[out] pOutLease On success this parameter contains a copy of
 * the lease. Using nullptr returns an error.
 *
 * @param[in] clientId This is the client id hash that represents the
 * DHCP client identifier option. Using
 * @a ClientIdentifierUnknown returns an error.
 *
 * @return Either a success or failure result code.
 */
Result GetLeaseByDhcpClientIdentifierHash(DhcpLease* pOutLease,
                                          ClientIdentifierHash clientId) NN_NOEXCEPT;

/**
 * @brief Gets a copy of the @ref DhcpLease for the provided IP address.
 *
 * @details
 * - It must be called when the server is in the
 * @ref DhcpState::Running state.
 *
 * - Querying by an unconfigured IP address returns an error.
 *
 * - Querying for an expired lease returns an error.
 *
 * @param[out] pOutLease On success this parameter contains a copy of
 * the lease. Using nullptr returns an error.
 *
 * @param[in] ipAddress This value is an IP address in network-byte
 * order.
 *
 * @return Either a success or failure result code.
 */
Result GetLeaseByInternetAddress(DhcpLease* pOutLease,
                                 nn::socket::InAddr ipAddress) NN_NOEXCEPT;

/**
 * @brief Gets a copy of the lease for the provided MAC address.
 *
 * @details
 * - It must be called when the server is in the
 * @ref DhcpState::Running state.
 *
 * - Querying by an unrecognized MAC address returns an error.
 *
 * - Querying for an expired lease returns an error.
 *
 * @param[out] pOutLease On success this parameter contains a copy
 * of the lease. Using nullptr returns an error.
 *
 * @param[in] macAddress This value is the MAC address.
 *
 * @return Either a success or failure result code.
 */
Result GetLeaseByEthernetMacAddress(DhcpLease* pOutLease,
                                    const EthernetMacAddress& macAddress) NN_NOEXCEPT;

/**
 * @brief Gets a string representation for the provided
 * DHCP Server @a Event.
 *
 * @param[in] e The provided event.
 *
 * @return A string representation for the provided event.
 */
const char* EventToString(Event e) NN_NOEXCEPT;

/**
 * @brief Gets a string representation for the provided @a State.
 *
 * @param[in] in The provided state.
 *
 * @return A string representation for the provided state.
 */
const char* StateToString(State in) NN_NOEXCEPT;
}};
