﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
 * @file
 * @brief   デバッグモニタ向けの API を宣言するファイルです
 */

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/svc/svc_BaseType.h>
#include <nn/svc/svc_DmntType.h>
#include <nn/svc/svc_Types.h>
#include <nn/ncm/ncm_ProgramId.h>

#include <nn/dbg/dbg_Types.h>
#include <nn/os/os_Types.h>
#include <nn/os/os_Event.h>

namespace nn {
namespace dbg {

//! @name 初期化と終了
//! @{

    Result Initialize() NN_NOEXCEPT;
    Result InitializeForSnapShotDumper() NN_NOEXCEPT;
    Result Finalize() NN_NOEXCEPT;
    Result FinalizeForSnapShotDumper() NN_NOEXCEPT;

//! @}



//! @name デバッグの開始
//! @{

    /**
        @brief  ホスト上のプログラムを起動しデバッグを開始します

        @param[out] pOut    デバッグハンドルを格納するバッファ
        @param[in]

        @retresult
            @handleresult{nn::os::Result,}
        @endretresult

        @pre
            -

        @post
            - ResultSuccess ならば
                -

        @details

    */
    Result DebugNewProcessFromHost(nn::svc::Handle* pOut, const char* pPath, const void* pArgument, size_t size) NN_NOEXCEPT;

    /**
        @brief  対象のプログラムの起動を待機し、起動したらデバッグを開始します

        @param[out] pOut    デバッグハンドルを格納するバッファ
        @param[in]

        @retresult
            @handleresult{nn::os::Result,}
        @endretresult

        @post
            - ResultSuccess ならば
                - *pOut は programId で指定されるプログラムから起動したプロセスに関連づいたデバッグハンドル

        @details
            関数内部で programId で指定されるプログラムの起動を待機し、
            該当するプログラムが起動したらそのデバッグを開始します
    */
    Result DebugNextProcess(nn::svc::Handle* pOut, ncm::ProgramId programId) NN_NOEXCEPT;

    /**
        @brief  キャンセル可能な形で対象のプログラムの起動を待機し、起動したらデバッグを開始します

        @param[out] pOut    デバッグハンドルを格納するバッファ
        @param[in]

        @retresult
            @handleresult{nn::os::Result,}
        @endretresult

        @post
            - ResultSuccess ならば
                - *pOut は programId で指定されるプログラムから起動したプロセスに関連づいたデバッグハンドル

        @details
            関数内部で programId で指定されるプログラムの起動を待機し、
            該当するプログラムが起動したらそのデバッグを開始します

            本関数呼び出し中に別スレッドから pCancelEvent が指す os::Event をシグナルすることで
            本関数内部での待機をキャンセルさせることができます。
    */
    Result DebugNextProcess(nn::svc::Handle* pOut, ncm::ProgramId programId, os::Event* pCancelEvent) NN_NOEXCEPT;

    /**
        @brief  アプリケーションの起動を待機し、起動したらデバッグを開始します

        @param[out] pOut    デバッグハンドルを格納するバッファ

        @retresult
            @handleresult{nn::os::Result,}
        @endretresult

        @post
            - ResultSuccess ならば
                - *pOut はアプリケーションプロセスに関連づいたデバッグハンドル

        @details
            関数内部でアプリケーションの起動を待機し、
            アプリケーションが起動したらそのデバッグを開始します
    */
    Result DebugNextApplicationProcess(nn::svc::Handle* pOut) NN_NOEXCEPT;

    /**
        @brief  キャンセル可能な形でアプリケーションの起動を待機し、起動したらデバッグを開始します

        @param[out] pOut            デバッグハンドルを格納するバッファ
        @param[in]  pCancelEvent    初期化済みの os::Event へのポインタ。

        @retresult
            @handleresult{nn::os::Result,}
        @endretresult

        @pre
            -

        @post
            - ResultSuccess ならば
                - *pOut はアプリケーションプロセスに関連づいたデバッグハンドル

        @details
            関数内部でアプリケーションの起動を待機し、
            アプリケーションが起動したらそのデバッグを開始します

            本関数呼び出し中に別スレッドから pCancelEvent が指す os::Event をシグナルすることで
            本関数内部での待機をキャンセルさせることができます。
    */
    Result DebugNextApplicationProcess(nn::svc::Handle* pOut, os::Event* pCancelEvent) NN_NOEXCEPT;

    /**
        @brief  プロセス ID を指定して実行中のプロセスのデバッグを開始します

        @param[out] pOut        デバッグハンドルを格納するバッファ
        @param[in]  processId   対象のプロセスを指定するプロセス ID

        @retresult
            @handleresult{nn::os::Result,}
        @endretresult

        @post
            - ResultSuccess ならば
                - *pOut は processId で裁定されるプロセスに関連づいたデバッグハンドル
    */
    Result DebugActiveProcess(nn::svc::Handle* pOut, nn::Bit64 processId) NN_NOEXCEPT;

    /**
        @brief  指定したプログラムから作成されたプロセスのデバッグを開始します

        @param[out] pOut        デバッグハンドルを格納するバッファ
        @param[in]  programId   対象のプログラムを指定するプログラム ID

        @retresult
            @handleresult{nn::os::Result,}
        @endretresult

        @post
            - ResultSuccess ならば
                - *pOut は programId で指定されるプログラムから起動したプロセスに関連づいたデバッグハンドル

        @details
            programId で指定されたプログラムが複数存在する場合は
            そのうちのいずかのプロセスのデバッグを開始します
    */
    Result DebugActiveProcess(nn::svc::Handle* pOut, ncm::ProgramId programId) NN_NOEXCEPT;

    /**
        @brief  実行中のアプリケーションプロセスのデバッグを開始します

        @param[out] pOut    デバッグハンドルを格納するバッファ

        @retresult
            @handleresult{nn::os::Result,}
        @endretresult

        @post
            - ResultSuccess ならば
                - *pOut はアプリケーションプロセスに関連づいたデバッグハンドル

        @details
            アプリケーションプロセスが複数存在する場合は
            そのうちのいずかのプロセスのデバッグを開始します
    */
    Result DebugActiveApplication(nn::svc::Handle* pOut) NN_NOEXCEPT;

//! @}



//! @name JIT デバッグ
//! @{

    /**
        @brief  JIT デバッグ待機中のプロセスのリストを取得します

        @param[out] pNumProcesses   pProcessIds に格納されたプロセス ID の個数
        @param[out] pProcessIds     JIT デバッグ待機中のプロセスのプロセス ID を格納するバッファ
        @param[in]  arraySize       pProcessIds で指定される配列の要素数

        @retresult
            @handleresult{nn::os::Result,}
        @endretresult

        @pre
            -

        @post
            - ResultSuccess であれば
                - *pNumProcesses >= 0
                - pProcessIds[i] (0 <= i < *pNumProcesses) は WaitAttach 状態のプロセスのプロセス ID
    */
    Result GetJitDebugProcessList(int32_t* pNumProcesses, os::ProcessId pProcessIds[], int32_t arraySize) NN_NOEXCEPT;

//! @}


//! @name デバッグ対象操作
//! @{

    Result BreakDebugProcess(nn::svc::Handle debug) NN_NOEXCEPT;
    Result TerminateDebugProcess(nn::svc::Handle debug) NN_NOEXCEPT;
    Result GetDebugEvent(nn::svc::DebugEventInfo* pInfo, nn::svc::Handle debug) NN_NOEXCEPT;
    Result ContinueDebugEvent(nn::svc::Handle debug, nn::Bit32 flags, nn::Bit64 threadIds[], nn::Bit32 size) NN_NOEXCEPT;
    Result GetProcessList(int32_t* pNumProcesses, nn::Bit64 pProcessIds[], int32_t arraySize) NN_NOEXCEPT;
    Result GetThreadList(int32_t* pNumThreads, nn::Bit64 pThreadIds[], int32_t arraySize, nn::svc::Handle domain) NN_NOEXCEPT;
    Result GetDebugThreadContext(nn::svc::ThreadContext* pContext, nn::svc::Handle debug, nn::Bit64 threadId, nn::Bit32 controlFlags) NN_NOEXCEPT;
    Result SetDebugThreadContext(nn::svc::Handle debug, nn::Bit64 threadId, const nn::svc::ThreadContext& context, nn::Bit32 controlFlags) NN_NOEXCEPT;
    Result QueryDebugProcessMemory(nn::svc::MemoryInfo* pBlockInfo, nn::svc::PageInfo* pPageInfo, nn::svc::Handle process, uintptr_t addr) NN_NOEXCEPT;
    Result ReadDebugProcessMemory(uintptr_t buf, nn::svc::Handle debug, uintptr_t addr, size_t size) NN_NOEXCEPT;
    Result WriteDebugProcessMemory(nn::svc::Handle debug, uintptr_t buf, uintptr_t addr, size_t size) NN_NOEXCEPT;
    Result SetHardwareBreakPoint(nn::svc::HardwareBreakPointRegisterName regNo, nn::Bit64 control, nn::Bit64 value) NN_NOEXCEPT;
    Result GetDebugThreadParam(nn::Bit64* pOut1, nn::Bit32* pOut2, nn::svc::Handle debug, nn::Bit64 threadId, nn::svc::DebugThreadParam select) NN_NOEXCEPT;
    Result CloseHandle(nn::svc::Handle handle) NN_NOEXCEPT;
    Result WaitSynchronization(int32_t* pOut, const nn::svc::Handle handles[], int32_t numHandles, int64_t ns) NN_NOEXCEPT;
    Result GetCurrentProcessId(nn::Bit64* pOut) NN_NOEXCEPT;
    Result GetProcessId(nn::Bit64* pOut, nn::svc::Handle process) NN_NOEXCEPT;
    Result GetProcessModuleInfo(int* pOutCount, nn::dbg::ModuleInfo* pOutModules, int num, os::ProcessId pid) NN_NOEXCEPT;

//! @}

} // end of namespace dbg
} // end of namespace nn
