﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief Definitions for all types used for the CEC process.
 */

#pragma once

#include <nn/nn_Common.h>

namespace nn { namespace cec {


    /**
     * @brief       Power state values.
     * @details     This lists the possible power states with regards to CEC.
     * @details     Transition states matter only if the device is not immediately fully responsive on CEC/HDMI.
     */
    typedef enum
    {
        PowerState_On,          //!< Device is turned on and responding nominally on CEC.
        PowerState_Standby,     //!< Device is in standby and minimally responsive on CEC.
        PowerState_GoingOn,     //!< Device is transitioning from Standby to On.
        PowerState_GoingStandby //!< Device is transitioning from On to Standby.
    } PowerState;

    /**
     * @brief       Connection status values.
     * @details     This lists the possible connections between the NX, cradle, and TV.
     */
    typedef enum
    {
        ConnectionState_NotConnected,           //!< Not connected to cradle.
        ConnectionState_OnlyCradleConnected,    //!< Connected to cradle, but no HDMI connection from cradle to TV.
        ConnectionState_CradleAndTvConnected    //!< Connected to cradle and TV.
    } ConnectionState;

    /**
     * @brief       Device Menu Control User Control code.
     * @details     Typedef for Device Menu Control User Control codes as defined by CEC.
     */
    typedef uint8_t RemoteControlCommand;

    /**
     * @brief       Device Menu Control User Control codes.
     * @details     CEC defined values for Device Menu Control User Control codes.
     */
    const RemoteControlCommand  RemoteControlVolumeUp   = 0x41;
    const RemoteControlCommand  RemoteControlVolumeDown = 0x42;
    const RemoteControlCommand  RemoteControlVolumeMute = 0x43;

    /**
     * @brief       Out parameter from nn::cec::GetBusEventType().
     * @details     The enumerated values define the various events that the CEC manager can detect and transmit.
     * @details     The difference between BusEventType_ActiveSourceChangedToInactive and BusEventType_GoStandby
     *              is that with the former message the TV and other devices on the CEC bus remain on, but another
     *              input source has been selected, while the latter sends either the target device or all devices on the
     *              CEC bus into standby.
     */
    typedef enum
    {
        BusEventType_Ignore,                        //!< No event data update, just a signal.  For internal use.
        BusEventType_ActiveSourceChangedToActive,   //!< The device is now active source and needs to output valid signal.
        BusEventType_ActiveSourceChangedToInactive, //!< The device is no longer the active source.
        BusEventType_GoStandby,                     //!< The device has been requested to go into standby.
        BusEventType_Suspending,                    //!< Unexpected I/O errors forced suspension of the CEC manager.
        BusEventType_ConnectionChange,              //!< A change in connection status has been detected.
        BusEventType_FeatureAbortOneTouchPlay,      //!< A Feature Abort related to One Touch Play was received.
        BusEventType_FeatureAbortStandby,           //!< A Feature Abort related to Standby was received.
        BusEventType_FeatureAbortSetOnScreenString, //!< A Feature Abort related to Set On Screen String was received.
        BusEventType_FeatureAbortDeviceMenuControl, //!< A Feature Abort related to Device Menu Control was received.
        BusEventType_Started                        //!< The CEC manager has successfully started.
    } BusEventType;

}}
