﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/** @file       cdacm_Lib.h
    @brief      CDACM API declaration
    @details    Provides explanation of each API
*/

#pragma once

namespace nn {
namespace cdacm {

/**
 * @brief Register for notification of a supported USB-to-serial dongle being attached
 *
 * @param[out] pUnitAvailableEventPtr
 *   A multi-wait event will will be signaled when a USB-to-serial dongle is ready to be opened.
 *
 * @param[in] pBuffer
 *   A user supplied buffer to store writes before the USB stack is loaded or a supported USB-to-serial
 *   device is connected.
 *
 * @param[in] bufferLen
 *   Length of the user supplied buffer.
 *
 * @retresult
 *   @handleresult{ResultSuccess, notification successfully configured}
 * @endretresult
 *
 * @pre
 *   Initialize() has not already been called, or if it was then Finalize()
 *   was called. pUnitAvailableEventPtr is not nullptr and it's initialized.
 *
 * @post
 *   Library is ready for operation. The ownership of pUnitAvailableEventPtr is
 *   transferred to CDACM and won't be relinquished until Finalize() is called.
 *
 * @details
 *   Upon registration, the USB stack will notify upon first attachment or existing attachment
 *   of supported devices.  Ownership of devices will occur when opening a handle.
 */
Result Initialize(nn::os::MultiWaitType **pUnitAvailableEventPtr,
                  void                   *pBuffer,
                  uint32_t                bufferLen) NN_NOEXCEPT;

/**
 * @brief Finalize CdAcm class library
 *
 * @retresult
 *   @handleresult{ResultSuccess, Finalization completed}
 * @endretresult
 *
 * @pre
 *   Initialize() has already been called.
 *
 * @post
 *   USB CdAcm class library can no longer be used.
 *
 * @details
 *   Upon finalization, the library will internally release ownership of all
 *   CdAcm class devices back to the USB host stack.
 */
Result Finalize() NN_NOEXCEPT;

/**
 * @brief Open a handle for a USB-to-serial device
 *
 * @param[out] pDetachEventPtr
 *    An event which will be signaled when corresponding device is detached
 *
 * @param[out] pOutProfile
 *    Profile of device
 *
 * @param[in] holder
 *    Multi wait holder that was previously signaled
 *
 * @param[in] pParameters
 *    Serial port configuration information
 *
 * @retresult
 *   @handleresult{ResultSuccess, A device profile is opened}
 *   @handleresult{ResultDeviceNotAvailable, Cannot find any ready device}
 * @endretresult
 *
 * @pre
 *   Initialize() must be called once first. pDetachEvent is not
 *   nullptr and it's initialized. pParameters is not nullptr and it's initialized.
 *   pOutProfile is not nullptr.
 *
 * @post
 *   If a device handle is obtained, the ownership of pDetachEvent is transferred to
 *   CDACM and won't be relinquished until the acquired device is detached.
 *
 * @details
 *   Poll for a device which is ready to be used. This call never blocks.
 *   The profile of device is returned in pOutProfile. pDetachEvent will be
 *   signaled when this device is detached.
 */

Result OpenHandle(nn::os::SystemEventType    **pDetachEventPtr,
                  UnitProfile                 *pOutProfile,
                  nn::os::MultiWaitHolderType *holder,
                  nn::cdacm::CdAcmParameters  *pParameters) NN_NOEXCEPT;

/**
 * @brief Close a handle for a USB-to-serial device
 *
 * @param[in] pProfile
 *    Profile of device
 *
 * @retresult
 *   @handleresult{ResultSuccess, A device profile is closed}
 *   @handleresult{ResultDeviceNotAvailable, Cannot find device corresponding to handle}
 * @endretresult
 *
 * @pre
 *   Caller has obtained a valid profile/handle through OpenHandle.
 *
 * @post
 *   Profile/handle is closed.
 *
 * @details
 *   The device corresponding to the handle is cleaned up.
 */

Result CloseHandle(UnitProfile *pOutProfile) NN_NOEXCEPT;

/**
 * @brief Perform synchronous read operation.
 *
 * @param[out] pBytesRead
 *   Stores number of bytes read.
 *
 * @param[out] pInBuffer
 *   Pointer to user supplied buffer.
 *   Data read from the CdAcm device will be written to this buffer.
 *
 * @param[in] handle
 *   Handle of USB-to-serial device from which to read.
 *
 * @param[in] length
 *   Number of bytes to be read.
 *
 * @retresult
 *   @handleresult{ResultSuccess, Read operation succeeded}
 *   @handleresult{ResultMisalignedBuffer, supplied buffer is not properly aligned}
 *   @handleresult{ResultUnitHandleInvalid, Specified handle is not valid}
 * @endretresult
 *
 * @pre
 *   pInBuffer is page aligned and is big enough to hold requested data.
 *
 * @details
 *   Read data is stored in the buffer and the number of bytes read is reported.
 */
Result
Read(size_t    *pBytesRead,
     void      *pInBuffer,
     UnitHandle handle,
     uint32_t   length) NN_NOEXCEPT;

/**
 * @brief Perform synchronous write operation.
 *
 * @param[out] pBytesWritten
 *   Stores number of bytes written.
 *
 * @param[in] pOutBuffer
 *   Pointer to user supplied buffer.
 *   Data in this buffer will be written to the USB-to-serial device.
 *
 * @param[in] handle
 *   Handle of USB-to-serial device to which to write.
 *
 * @param[in] length
 *   Number of bytes to be written.
 *
 * @retresult
 *   @handleresult{ResultSuccess, Read operation succeeded}
 *   @handleresult{ResultMisalignedBuffer, supplied buffer is not properly aligned}
 *   @handleresult{ResultUnitHandleInvalid, Specified handle is not valid}
 * @endretresult
 *
 * @pre
 *   pOutBuffer is page aligned.
 *
 * @details
 *   Data is written synchronously and bytes written is reported.
 */
Result
Write(size_t * pBytesWritten,
      const void* pOutBuffer,
      UnitHandle handle,
      uint32_t length) NN_NOEXCEPT;

/**
* @brief Perform asynchronous write operation.
*
* @param[out] pOutXferId
*   Stores transfer ID for obtaining result later.
*
* @param[out] pCompletionEventPtr
*   Stores the address of the event to be set when the transfer is complete.
*
* @param[in] pOutBuffer
*   Pointer to user supplied buffer.
*   Data in this buffer will be written to the USB-to-serial device.
*
* @param[in] handle
*   Handle of USB-to-serial device to which to write.
*
* @param[in] length
*   Number of bytes to be written.
*
* @retresult
*   @handleresult{ResultSuccess, Read operation succeeded}
*   @handleresult{ResultMisalignedBuffer, supplied buffer is not properly aligned}
*   @handleresult{ResultUnitHandleInvalid, Specified handle is not valid}
* @endretresult
*
* @pre
*   pOutBuffer is page aligned.
*
* @details
*   Data is written asynchronously and event provided that will be signaled upon completion.
*/
Result
WriteAsync(
    uint32_t                 *pOutXferId,
    nn::os::SystemEventType **pCompletionEventPtr,
    const void               *pOutBuffer,
    UnitHandle                handle,
    uint32_t                  length) NN_NOEXCEPT;

/**
* @brief Poll the status of submitted transfers
*
* @param[out] pOutCount
*   Number of reports returned
*
* @param[out] pOutReport
*   Pointer to the buffer where the reports will be stored
*
* @param[in] handle
*   Handle of USB-to-serial device from which to obtain the asynch write results.
*
* @param[out] count
*   Number of reports requested
*
* @retresult
*   @handleresult{nn::ResultSuccess, Operation succeeded}
*   @handleresult{nn::usb::ResultNotInitialized, This object is not initialized}
* @endretresult
*
* @details
*   Get the status report of completed transfers.
*/
Result
GetWriteAsyncResult(uint32_t *pOutCount, nn::usb::XferReport *pOutReport, UnitHandle handle, uint32_t count) NN_NOEXCEPT;

/**
* @brief Write a caller defined formatted string to the USB-to-serial device.
*
* @param[in] handle
*   Handle of USB-to-serial device to which to write.
*
* @param[in] fmt
*   Format of string to write.
*
* @param[in] ...
*   Additional parameters defined by caller.
*
* @retresult
*   @handleresult{ResultSuccess, Read operation succeeded}
*   @handleresult{ResultInvalidArgs, formatted string is zero length}
*   @handleresult{ResultUnitHandleInvalid, Specified handle is not valid}
* @endretresult
*
* @pre
*   Formatted string is valid.
*
* @details
*   Formatted string is written synchronously.
*/
Result Print(UnitHandle handle, const char * fmt, ...) NN_NOEXCEPT;


/**
* @brief Enable Log Observer.
*/
void EnableLogObserver() NN_NOEXCEPT;

/**
* @brief Disable Log Observer.
*/
void DisableLogObserver() NN_NOEXCEPT;

}  // CDACM
}  // nn
