﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   Btm debug API
 */

#pragma once

#include <nn/nn_Result.h>
#include <nn/os/os_SystemEventTypes.h>
#include <nn/btm/btm_Types.h>

namespace nn { namespace btm { namespace debug {

//! @name SF機能
//! @{

/**
 * @brief       Btmサーバーとのセッションを取得します
 * @details     同期APIです。
 *              セッションはライブラリ内で保持されます。他のすべてのAPIより先に呼び出しておく必要があります。
 *              複数回呼び出した場合は、ライブラリ内で回数を記憶しますが、セッションは1つのみ保持します。
 */
void InitializeBtmDebugInterface() NN_NOEXCEPT;

/**
 * @brief       Btmサーバーとのセッションを破棄します
 * @details     同期APIです。
 *              セッション破棄後は、再度セッションを取得するまでSF機能 以外のAPIは使用できません。
 *              InitializeBtmInterfaceを呼び出した回数分のFinalizeBtmInterfaceが必要です。
 */
void FinalizeBtmDebugInterface() NN_NOEXCEPT;

//! @}


//! @name Discovery機能
//! @{

/**
 * @brief       Discovery完了タイミング取得用のシステムイベントを登録します
 * @details     同期APIです。
 *              サーバー内のシステムイベントのハンドルを、指定されたシステムイベントにアタッチします
 * @param[out]  pSystemEvent
 */
void RegisterSystemEventForDiscovery(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT;

/**
 * @brief       Discoveryを開始します
 * @details     非同期APIです。
 *              Discoveryが開始できた、もしくは開始できなかったタイミングで処理が返ります。
 *              Discoveryが開始された時点で、サーバー内の前回のDiscovery結果は削除されます
 *              Discovery結果は、該当システムイベントのシグナル後、GetDiscoveredDevicesで取得してください
 *              RadioOff中は ResultInvalidState が返ります。
 */
nn::Result StartDiscovery() NN_NOEXCEPT;

/**
 * @brief       実行中のDiscoveryを途中で終了させます
 * @details     非同期APIです。
 *              キャンセルの要求が受理された、もしくは受理されなかったタイミングで処理が返ります
 *              Discoveryの終了タイミングは、該当システムイベントのシグナルで判断してください
 *              RadioOff中は ResultInvalidState が返ります。
 */
nn::Result CancelDiscovery() NN_NOEXCEPT;

/**
 * @brief       Discoveryの結果を取得します
 * @details     同期APIです。
 *              セッション取得後は、Btmがどんなステートであっても、実行可能です。
 *              該当システムイベントのシグナル前に呼んだ場合は、その時点でのDiscovery結果が取得されます
 * @param[out]  pList
 */
void GetDiscoveredDevices(DevicePropertyList* pList) NN_NOEXCEPT;

//! @}




//! @name Bond機能
//! @{

/**
 * @brief       指定したデバイスとBondします
 * @details     非同期APIです。
 *              Bond処理が開始した、もしくは開始できなかったタイミングで処理が返ります。
 *              Bond完了のタイミングを取得する機能はありません。
 *              Bondの直後に自動で接続がなされるため、RegisterSystemEventForConnectedDeviceCondition の
 *              システムイベントのシグナルをもって、Bond完了を判断してください。
 *              Bond結果は、GetDeviceInfo で取得して下さい
 *              RadioOff中は ResultInvalidState が返ります。
 * @param[in]   pAddress
 */
nn::Result CreateBond(const BdAddress *pAddress) NN_NOEXCEPT;

/**
 * @brief       指定したデバイスとのBondを途中で終了させます
 * @details     非同期APIです。
 *              キャンセル処理が開始した、もしくは開始できなかったタイミングで処理が返ります。
 *              キャンセルの成否を取得する機能はありません
 *              RadioOff中は ResultInvalidState が返ります。
 * @param[in]   pAddress
 */
nn::Result CancelBond(const BdAddress *pAddress) NN_NOEXCEPT;

//! @}


//! @name Test機能
//! @{

nn::Result SetTsiMode(const BdAddress* pAddress, TsiMode tsiMode) NN_NOEXCEPT;

/**
 * @brief       内部で定義済のテストをトリガします
 * @details     同期APIです。テスト完了までブロックします。
 *              テスト内容は、 btm_Handler.cpp のAPIハンドラに記述されています。
 * @param[in]   mode
 */
void GeneralTest(int mode) NN_NOEXCEPT;

/**
 * @brief       内部情報を取得します
 * @details     同期APIです。
 *              取得内容は、 btm_Handler.cpp のAPIハンドラに記述されています。
 * @param[in]   mode
 */
void GeneralGet(int mode, GeneralInfoList* pGeneralInfoList) NN_NOEXCEPT;

//! @}


//! @name HID機能
//! @{

void HidConnect(const BdAddress* pAddress) NN_NOEXCEPT;

//! @}

//! @name BLE 機能
//! @{

//!< BLE の切断理由を表す列挙型定数です
enum BleDisconnectionReason : uint16_t
{
    BleDisconnectionReason_Unknown              = 0x0000,
    BleDisconnectionReason_L2cFailure           = 0x0001,
    BleDisconnectionReason_SupervisionTimeout   = 0x0008,
    BleDisconnectionReason_TerminatedByPeer     = 0x0013,
    BleDisconnectionReason_TerminatedByHost     = 0x0016,
    BleDisconnectionReason_LmpResponseTimeout   = 0x0022,
    BleDisconnectionReason_EstablishFailed      = 0x003E,
    BleDisconnectionReason_L2cConnectionCancled = 0x0100,
};

/**
 * @brief       Gatt Client の切断理由を取得します
 *
 * @details     同期API です。
 *              指定したアドレスと接続ハンドルの組み合わせで、記録されている最も新しい切断理由を取得します。
 *
 * @param[out]  pOutReason          取得した切断理由を格納するためのオブジェクトへのポインタ
 * @param[in]   connectionHandle    切断理由を取得する切断前の接続ハンドル
 * @param[in]   address             切断理由を取得する切断前の GATT Server の Bluetooth アドレス
 *
 * @pre
 *              - pOutReason != nullptr
 *
 * @return      切断理由の記録が存在すればTrue。存在しなければFalse。
 */
bool GetGattClientDisconnectionReason(BleDisconnectionReason* pOutReason, uint32_t connectionHandle, const BdAddress& address) NN_NOEXCEPT;

/**
 * @brief       Gatt Client の現在の接続パラメータを取得します
 *
 * @details     同期API です。
 *              接続が存在しない場合、0 をセットし、False を返します。
 *
 * @param[out]  pOutInterval        現在のコネクションインターバルを格納するための変数オブジェクトへのポインタ
 * @param[out]  pOutLatency         現在のスレーブレイテンシを格納するための変数オブジェクトへのポインタ
 * @param[out]  pOutTimeout         現在のスーパービジョンタイムアウトを格納するための変数オブジェクトへのポインタ
 * @param[in]   connectionHandle    接続パラメータを取得するGATT Server との接続ハンドル
 *
 * @pre
 *              - pOutInterval != nullptr
 *              - pOutLatency != nullptr
 *              - pOutTimeout != nullptr
 */
bool GetBleConnectionParameter(uint16_t* pOutInterval, uint16_t* pOutLatency, uint16_t* pOutTimeout,
                               uint32_t connectionHandle) NN_NOEXCEPT;

/**
 * @brief       現在接続されているGatt Server が要求した接続パラメータを取得します
 *
 * @details     同期API です。
 *              接続が存在しない場合、0 をセットし、False を返します。
 *
 * @param[out]  pOutIntervalMin     要求されたコネクションインターバルの最小値を格納するための変数オブジェクトへのポインタ
 * @param[out]  pOutIntervalMax     要求されたコネクションインターバルの最大値を格納するための変数オブジェクトへのポインタ
 * @param[out]  pOutLatency         要求されたスレーブレイテンシを格納するための変数オブジェクトへのポインタ
 * @param[out]  pOutTimeout         要求されたスーパービジョンタイムアウトを格納するための変数オブジェクトへのポインタ
 * @param[in]   connectionHandle    要求された接続パラメータを取得するGATT Server との接続ハンドル
 *
 * @pre
 *              - pOutIntervalMin != nullptr
 *              - pOutIntervalMax != nullptr
 *              - pOutLatency != nullptr
 *              - pOutTimeout != nullptr
 */
bool GetBleConnectionParameterRequest(uint16_t* pOutIntervalMin, uint16_t* pOutIntervalMax, uint16_t* pOutLatency, uint16_t* pOutTimeout,
                                      uint32_t connectionHandle) NN_NOEXCEPT;


//! @}

}}} // namespace nn::btm::debug

