﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief 電源ボタンの押下状態を取得する機能の API 宣言。
 */

#pragma once

#include <nn/nn_Result.h>
#include <nn/os.h>

#include "bpc_PowerButtonTypes.h"

namespace nn { namespace bpc {

//! @name 電源ボタンの押下状態取得
//! @{

/**
 * @brief 電源ボタンの押下状態取得関連の機能ブロックを初期化します。
 *
 * @post
 *  対象の機能ブロックが初期化済の状態になります。
 *
 * @details
 *  電源ボタンの押下状態取得関連の機能ブロックを初期化します。 @n
 *  本関数の呼び出し回数はライブラリ内部でカウントされます。
 */
void InitializePowerButton() NN_NOEXCEPT;

/**
 * @brief 電源ボタンの押下状態取得関連の機能ブロックを終了します。
 *
 * @post
 *  対象の機能ブロックが初期化前の状態になります。
 *
 * @details
 *  電源ボタンの押下状態取得関連の機能ブロックを終了します。 @n
 *  初期化を行った回数分、本関数が呼ばれた場合にのみ、対象の機能ブロックが初期化前の状態になります。
 */
void FinalizePowerButton() NN_NOEXCEPT;

/**
 * @brief       現在の電源ボタンの状態を取得します。
 *
 * @param[out]  pOutState 電源ボタンの現在の押下状態です。@n @ref nn::bpc::SleepButtonState 型で表現されます。
 *
 * @return      処理の結果を返します。
 *
 * @details     現在の電源ボタンの状態を取得します。@n
 *              @ref nn::bpc::SleepButtonState_Pushed もしくは @ref nn::bpc::SleepButtonState_Released のいずれかが取得できます。
 */
nn::Result GetSleepButtonState(SleepButtonState* pOutState) NN_NOEXCEPT;

/**
 * @brief       指定したイベントを @ref nn::os::SystemEventType 型のオブジェクトに関連付けて検知できるようにします。
 *
 * @param[out]  pOutEvent 指定したイベントに関連付けられる @ref nn::os::SystemEventType 型のオブジェクトです。
 * @param[in]   target 指定したイベントを表す @ref nn::bpc::EventTarget 型のイベント識別子です。
 *
 * @return      処理の結果を返します。
 *
 * @details     指定したイベントを @ref nn::os::SystemEventType 型のオブジェクトに関連付けて検知できるようにします。@n
 *              ｐOutEvent は @ref nn::os::EventClearMode_ManualClear で関連付けられるので、
 *              イベント取得後には @ref nn::os::ClearSystemEvent() を明示的に呼ばなければ Signal されたままの状態になります。@n
 *              複数のクライアントから呼び出しには対応していません。
 */
nn::Result GetPowerEvent(nn::os::SystemEventType* pOutEvent, EventTarget target) NN_NOEXCEPT;

//! @}

}} // namepsace nn::bpc
