﻿/*--------------------------------------------------------------------------------*
  Copyright (C) 2008 The Android Open Source Project
  Modified by Nintendo

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief
 */

#pragma once

//--------------------------------------------------------------------------
//  C/C++
//--------------------------------------------------------------------------

#ifdef __cplusplus
namespace nn { namespace bluetooth {

extern "C"
{
#endif // ifdef __cplusplus


/* Bluetooth Address */
typedef struct {
    uint8_t address[6];
} __attribute__((packed))Btbdaddr;

/* Bluetooth Device Name */
typedef struct {
    uint8_t name[249];
} __attribute__((packed))Btbdname;

/* Bluetooth Adapter Visibility Modes*/
typedef enum {
    BT_SCAN_MODE_NONE,
    BT_SCAN_MODE_CONNECTABLE,
    BT_SCAN_MODE_CONNECTABLE_DISCOVERABLE
} BluetoothScanMode;

/* Bluetooth Adapter State */
typedef enum {
    BT_STATE_OFF,
    BT_STATE_ON
} BluetoothState;

/* Bluetooth Error Status */
typedef enum {
    BT_STATUS_SUCCESS,
    BT_STATUS_FAIL,
    BT_STATUS_NOT_READY,
    BT_STATUS_NOMEM,
    BT_STATUS_BUSY,
    BT_STATUS_DONE,        /* request already completed */
    BT_STATUS_UNSUPPORTED,
    BT_STATUS_PARM_INVALID,
    BT_STATUS_UNHANDLED,
    BT_STATUS_AUTH_FAILURE,
    BT_STATUS_RMT_DEV_DOWN,
    BT_STATUS_AUTH_REJECTED
} BluetoothIfStatus;

/* Bluetooth PinKey Code */
typedef struct {
    uint8_t pin[16];
} __attribute__((packed))BluetoothPinCode;

typedef struct {
    uint8_t status;
    uint8_t ctrl_state;     /* stack reported state */
    uint64_t TxTime;       /* in ms */
    uint64_t RxTime;       /* in ms */
    uint64_t IdleTime;     /* in ms */
    uint64_t EnergyUsed;   /* a product of mA, V and ms */
} __attribute__((packed))BluetoothActivityEnergyInfo;

/* Bluetooth Adapter Discovery state */
typedef enum {
    BT_DISCOVERY_STOPPED,
    BT_DISCOVERY_STARTED
} BluetoothDiscoveryState;

/* Bluetooth ACL connection state */
typedef enum {
    BT_ACL_STATE_CONNECTED,
    BT_ACL_STATE_DISCONNECTED
} BluetoothAclState;

/* Bluetooth 128-bit UUID */
typedef struct {
    uint8_t uu[16];
} BluetoothUUID;

/* Bluetooth SDP service record */
typedef struct
{
    BluetoothUUID uuid;
    uint16_t channel;
    char name[256];
} BluetoothServiceRecord;

/* Bluetooth Remote Version info */
typedef struct
{
    int version;
    int SubVer;
    int manufacturer;
} BluetoothRemoteVersion;

/* Class of Device */
typedef struct {
    uint8_t cod[3];
} __attribute__((packed))ClassOfDevice;

typedef struct
{
    uint8_t LocalPrivacyEnabled;
    uint8_t MaxAdvInstance;
    uint8_t RpaOffloadSupported;
    uint8_t MaxIrkListSize;
    uint8_t MaxAdvFilterSupported;
    uint8_t ScanResultStorageSizeLobyte;
    uint8_t ScanResultStorageSizeHibyte;
    uint8_t ActivityEnergyInfoSupported;
}BluetoothLocalLEfeatures;

/* Bluetooth Adapter and Remote Device property types */
typedef enum {
    /* Properties common to both adapter and remote device */

    /*
     * Description - Bluetooth Device Name
     * Access mode - Adapter name can be GET/SET. Remote device can be GET
     * Data type   - Btbdname
     */
    BT_PROPERTY_BDNAME = 0x1,

    /*
     * Description - Bluetooth Device Address
     * Access mode - Only SET.
     * Data type   - Btbdaddr
     */
    BT_PROPERTY_BDADDR,

     /*
     * Description - Bluetooth Super Device Address
     * Access mode - Only SET.
     * Data type   - Btbdaddr
     */
    BT_PROPERTY_SUPER_BDADDR,

    /*
     * Description - Bluetooth Service 128-bit UUIDs
     * Access mode - Only GET.
     * Data type   - Array of BluetoothUUID (Array size inferred
     * from property length).
     */
    BT_PROPERTY_UUIDS,

    /*
     * Description - Bluetooth Class of Device as found in Assigned Numbers
     * Access mode - Only GET.
     * Data type   - uint32_t.
     */
    BT_PROPERTY_CLASS_OF_DEVICE,

    /*
     * Description - Bluetooth Feature Set
     * Access mode - GET and SET
     * Data type   - uint8_t.
     */
    BT_PROPERTY_FEATURE_SET,

    /*
     * Description - Device Type - BREDR, BLE or DUAL Mode
     * Access mode - Only GET.
     * Data type   - BluetoothDeviceType
     */
    BT_PROPERTY_TYPE_OF_DEVICE,

    /*
     * Description - Bluetooth Service Record
     * Access mode - Only GET.
     * Data type   - BluetoothServiceRecord
     */
    BT_PROPERTY_SERVICE_RECORD,

    /* Properties unique to adapter */
    /*
     * Description - Bluetooth Adapter scan mode
     * Access mode - GET and SET
     * Data type   - BluetoothScanMode.
     */
    BT_PROPERTY_ADAPTER_SCAN_MODE,

    /*
     * Description - List of bonded devices
     * Access mode - Only GET.
     * Data type   - Array of Btbdaddr of the bonded remote devices
     *               (Array size inferred from property length).
     */
    BT_PROPERTY_ADAPTER_BONDED_DEVICES,

    /*
     * Description - Bluetooth Adapter Discovery timeout (in seconds)
     * Access mode - GET and SET
     * Data type   - uint32_t
     */
    BT_PROPERTY_ADAPTER_DISCOVERY_TIMEOUT,

    /* Properties unique to remote device */
    /*
     * Description - User defined friendly name of the remote device
     * Access mode - GET and SET
     * Data type   - Btbdname.
     */
    BT_PROPERTY_REMOTE_FRIENDLY_NAME,

    /*
     * Description - RSSI value of the inquired remote device
     * Access mode - Only GET.
     * Data type   - int32_t.
     */
    BT_PROPERTY_REMOTE_RSSI,

    /*
     * Description - Remote version info
     * Access mode - SET/GET.
     * Data type   - BluetoothRemoteVersion.
     */
    BT_PROPERTY_REMOTE_VERSION_INFO,

    /*
     * Description - Local LE features
     * Access mode - GET.
     * Data type   - BluetoothLocalLEfeatures.
     */
    BT_PROPERTY_LOCAL_LE_FEATURES,

    /*
     * Description - Returns all the properties
     * Access mode - GET.
     * Data type   - FullDeviceProperties
     */
    BT_PROPERTY_ALL,

    BT_PROPERTY_REMOTE_DEVICE_TIMESTAMP = 0xFF,
} BluetoothPropertyType;

/* Bluetooth Adapter Property data structure */
typedef struct
{
    BluetoothPropertyType type;
    int len;
    void *pVal;
} BluetoothProperty;

/* Bluetooth Device Type */
typedef enum {
    BT_DEVICE_DEVTYPE_BREDR = 0x1,
    BT_DEVICE_DEVTYPE_BLE,
    BT_DEVICE_DEVTYPE_DUAL
} BluetoothDeviceType;

/* Bluetooth Bond state */
typedef enum {
    BT_BOND_STATE_NONE,
    BT_BOND_STATE_BONDING,
    BT_BOND_STATE_BONDED
} BluetoothBondState;

/* Bluetooth SSP Bonding Variant */
typedef enum {
    BT_SSP_VARIANT_PASSKEY_CONFIRMATION,
    BT_SSP_VARIANT_PASSKEY_ENTRY,
    BT_SSP_VARIANT_CONSENT,
    BT_SSP_VARIANT_PASSKEY_NOTIFICATION
} BluetoothSspVariant;

/* Bluetooth Device properties */
typedef struct {
    Btbdname BdName;    // [GET/SET] device name
    Btbdaddr BdAddress; // [GET/SET] BD_ADDR
    BluetoothUUID Uuid[1];// [GET] Array of UUID // FIXME: Set a max value for this
    uint8_t ClassOfDevice[3];// [GET]
    BluetoothDeviceType TypeOfDevice; // [GET] Device Type - BREDR, BLE or DUAL Mode
    BluetoothServiceRecord SdpRecord; // [GET] Service record
    BluetoothScanMode ScanMode;       // [GET/SET]
    Btbdaddr BondedDevices[10];  // [GET] Array of Btbdaddr of the bonded remote devices // FIXME: Set a max value for this
    uint32_t DiscoveryTimeout;   // [GET/SET] in seconds
    Btbdname RemoteFriendlyName; // [GET/SET] User defined friendly name of the remote device
    int32_t RemoteRssi; // [GET] RSSI value of the inquired remote device
    BluetoothRemoteVersion RemoteVersionInfo; //[GET/SET]
    BluetoothLocalLEfeatures LocalLEfeatures; //[GET]
} __attribute__((packed))FullDeviceProperties;


typedef enum {
    ASSOCIATE_THREAD,
    DISASSOCIATE_THREAD
} BluetoothCallbackThreadEvent;


// ///////////////////// HID INTERFACE ///////////////////// //
// ///////////////////// HID INTERFACE ///////////////////// //
// ///////////////////// HID INTERFACE ///////////////////// //

enum {BTHH_MAX_DSC_LEN=884};

/* HH connection states */
typedef enum
{
    BTHH_CONN_STATE_CONNECTED              = 0,
    BTHH_CONN_STATE_CONNECTING,
    BTHH_CONN_STATE_DISCONNECTED,
    BTHH_CONN_STATE_DISCONNECTING,
    BTHH_CONN_STATE_FAILED_MOUSE_FROM_HOST,
    BTHH_CONN_STATE_FAILED_KBD_FROM_HOST,
    BTHH_CONN_STATE_FAILED_TOO_MANY_DEVICES,
    BTHH_CONN_STATE_FAILED_NO_BTHID_DRIVER,
    BTHH_CONN_STATE_FAILED_GENERIC,
    BTHH_CONN_STATE_UNKNOWN
} BluetoothHhConnectionState;

typedef enum
{
    BTHH_OK                = 0,
    BTHH_HS_HID_NOT_READY,        /* handshake error : device not ready */
    BTHH_HS_INVALID_RPT_ID,       /* handshake error : invalid report ID */
    BTHH_HS_TRANS_NOT_SPT,        /* handshake error : transaction not supported */
    BTHH_HS_INVALID_PARAM,        /* handshake error : invalid parameter */
    BTHH_HS_ERROR,                /* handshake error : unspecified HS error */
    BTHH_HH_ERR,                  /* general BTA HH error */
    BTHH_ERR_SDP,                 /* SDP error */
    BTHH_ERR_PROTO,               /* SET_Protocol error, only used in BTA_HH_OPEN_EVT callback */
    BTHH_ERR_DB_FULL,             /* device database full error, used  */
    BTHH_ERR_TOD_UNSPT,           /* type of device not supported */
    BTHH_ERR_NO_RES,              /* out of system resources */
    BTHH_ERR_AUTH_FAILED,         /* authentication fail */
    BTHH_ERR_HDL,
    BTHH_ERR,                     /* default error */
    BTHH_ERR_BAD_PARAM,           /* bad parameter error */
    BTHH_ERR_BAD_RESPONSE,        /* bad message/response error */
    BTHH_ERR_BUSY,                /* busy/not ready error */
    BTHH_ERR_CONNECTION,          /* connection errors */
    BTHH_ERR_OPEN,                /* open/enable errors */
    BTHH_ERR_CLIENT,              /* general client errors */
    BTHH_ERR_SERVER,              /* general server errors */
    BTHH_ERR_SECURITY,            /* security errors */
    BTHH_ERR_FILE_SYS,            /* file system errors */
    BTHH_ERR_AUDIO_VIDEO,         /* audio video errors */
    BTHH_ERR_DISCOVERY,           /* discovery errors */
    BTHH_ERR_TEST_MODE,           /* test mode errors */
    BTHH_ERR_DM,                  /* DM errors errors */
    BTHH_ERR_ROBSON               /* robson errors */
}BluetoothHhStatus;

/* Protocol modes */
typedef enum {
    BTHH_REPORT_MODE       = 0x00,
    BTHH_BOOT_MODE         = 0x01,
    BTHH_UNSUPPORTED_MODE  = 0xff
}BluetoothHhProtocolMode;

/* Report types */
typedef enum {
    BTHH_INPUT_REPORT      = 1,
    BTHH_OUTPUT_REPORT,
    BTHH_FEATURE_REPORT
}BluetoothHhReportType;

typedef struct
{
    int         attr_mask;
    uint8_t     sub_class;
    uint8_t     app_id;
    int         vendor_id;
    int         product_id;
    int         version;
    uint8_t     ctry_code;
    int         dl_len;
    uint8_t     dsc_list[BTHH_MAX_DSC_LEN];
} BluetoothHhHidInfo;

#ifdef __cplusplus
}
}} // namespace nn::bluetooth
#endif // ifdef __cplusplus
