﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
/**
 * @file
 * @brief
 */

#pragma once

#include <nn/bluetooth/bluetooth_TypesHal.h>
#include <nn/bluetooth/bluetooth_LeTypes.h>
#include <nn/bluetooth/bluetooth_GattAttribute.h>

namespace nn { namespace bluetooth {

const uint16_t BUFFER_SIZE_OF_CORE_OUT=1024;

const uint16_t BUFFER_SIZE_OF_HID_DATA=1024;
const uint16_t BUFFER_SIZE_OF_HID_OUT=BUFFER_SIZE_OF_HID_DATA + 128;

const uint16_t BUFFER_SIZE_OF_BLE_OUT=1024;

const uint16_t ARRAY_SIZE_OF_CHANNEL_INFO=10;

const uint16_t BUFFER_SIZE_OF_BLE_GATT_ATTRIBUTE = static_cast<uint16_t>(GattAttributeValueSizeMax);

const uint8_t GATT_DATA_PATH_FILTER_NUM_MAX = GattAttributeCountMaxClient;

//
//Parameter definitions for APIs.
//
typedef Address BluetoothAddress;
typedef BluetoothPinCode BluetoothPinCode;
typedef Btbdname BluetoothName;
typedef BluetoothHhReportType BluetoothHhReportType;
typedef BluetoothPropertyType BluetoothPropertyType;

struct HidData
{
    uint16_t length;
    uint8_t data[639];
};

struct HidReport
{
    uint16_t reportLength;
    uint8_t handle;
    BluetoothAddress bluetoothAddress;
    uint8_t mode;
    uint8_t subClass;
    uint8_t countryCode;
    HidData reportData;
};

struct ChannelInfo
{
    uint8_t channels[ARRAY_SIZE_OF_CHANNEL_INFO];
    uint8_t count;
};

struct AdapterProperty
{
    BluetoothAddress bluetoothAddress;
    ClassOfDevice classOfDevice;
    BluetoothName bluetoothName;
    uint8_t featureSet;
};

typedef enum
{
    NoPreference,   /* No preference of physical transport */
    BrEdr,  /* Prefer BR/EDR transport */
    LowEnergy  /* Prefer LE transport */
}BluetoothTransport;

//
//Event types.
//
typedef enum
{
    EventFromAdapterStateChangedCallback,
    EventFromAdapterPropertiesCallback,
    EventFromRemoteDevicePropertiesCallback,
    EventFromDeviceFoundCallback,
    EventFromDiscoveryStateChangedCallback,
    EventFromPinRequestCallback,
    EventFromSspRequestCallback,
    EventFromBondStateChangedCallback,
    EventFromAclStateChangedCallback,
    EventFromCallbackThreadEventCallback,
    EventFromDutModeReceivedCallback,
    EventFromLEtestModeCallback,
    EventFromEnergyInfoCallback,
    EventFromFatalErrorCallback,
    EventFromLowEnergyCallbacks,
} EventType;

typedef enum
{
    EventFromLeClientStateChangedCallback,
    EventFromLeServerStateChangedCallback,
    EventFromLeConnParamUpdateCallback,
    EventFromLeConnParamUpdateReqCallback,
    EventFromLeClientConnStateChangedCallback,
    EventFromLeServerConnStateChangedCallback,
    EventFromLeScanStateChangedCallback,
    EventFromLeScanFilterStateChangedCallback,
    EventFromLeClientGattOpCallback,
    EventFromLeClientServiceDiscoveryCallback,
    EventFromLeClientServiceDiscoveryStateChangedCallback,
    EventFromLeClientConfigureMtuCallback,
    EventFromLeServerProfileChangedCallback,
    EventFromLeServerGattReqCallback,
} BleEventType;

typedef enum
{
    EventFromConnectionStateCallback,
    EventFromHidInfoCallback,
    EventFromProtocolModeCallback,
    EventFromIdleTimeCallback,
    EventFromGetReportCallback,
    EventFromVirtualUnplugCallback,
    EventFromHandshakeCallback,
    EventFromExtensionCallbacks,
    EventFromSetReportStatusCallback,
    EventFromGetReportStatusCallback,
    EventFromLeClientGattOpForHidCallback,
} HidEventType;

//[Todo] To use this enum for ExtEvent.
typedef enum
{
    EventFromTsiSetCallback,
    EventFromTsiExitCallback,
    EventFromBurstSetCallback,
    EventFromBurstExitCallback,
    EventFromSetZeroRetranCallback,
    EventFromGetPendingConnectionsCallback,
    NotDefined
} ExtEventType;

//
//Event structures.
//
struct InfoFromAdapterStateChangedCallback
{
    BluetoothState state;
};
struct InfoFromAdapterPropertiesCallback
{
    BluetoothAddress bluetoothAddress;
    BluetoothIfStatus status;
    //int NumProperties;
    //BluetoothProperty *properties;//[Todo]To define how to handle void* information.
};
struct InfoFromRemoteDevicePropertiesCallback
{
    BluetoothAddress bluetoothAddress;
    BluetoothIfStatus status;
    //int NumProperties;
    //BluetoothProperty *properties;//[Todo]To define how to handle void* information.
};

#if 0
struct InfoFromDeviceFoundCallback
{
    BluetoothAddress bluetoothAddress;
    //int NumProperties;
    //BluetoothProperty *properties;//[Todo]To define how to handle void* information.
};
#endif
typedef FullDeviceProperties InfoFromDeviceFoundCallback;

struct InfoFromDiscoveryStateChangedCallback
{
    BluetoothDiscoveryState state;
};
struct InfoFromPinRequestCallback
{
    BluetoothAddress bluetoothAddress;
    BluetoothName bluetoothName;
    ClassOfDevice classOfDevice;
};
struct InfoFromSspRequestCallback
{
    BluetoothAddress bluetoothAddress;
    BluetoothName bluetoothName;
    ClassOfDevice classOfDevice;
    BluetoothSspVariant variant;
    uint32_t passkey;
};
struct InfoFromBondStateChangedCallback
{
    BluetoothAddress bluetoothAddress;
    BluetoothIfStatus status;
    BluetoothBondState state;
};
struct InfoFromAclStateChangedCallback
{
    BluetoothAddress bluetoothAddress;
    BluetoothIfStatus status;
    BluetoothAclState state;
};
struct InfoFromCallbackThreadEventCallback
{
    BluetoothCallbackThreadEvent evt;
};
struct InfoFromDutModeReceivedCallback
{
    uint16_t opcode;
    uint8_t length;
    uint8_t *buffer;
};
struct InfoFromLEtestModeCallback
{
    BluetoothIfStatus status;
    uint16_t numPackets;
};
struct InfoFromEnergyInfoCallback
{
    BluetoothActivityEnergyInfo energyInfo;
};



//Informations from the HAL Hid callback.
struct InfoFromConnectionStateCallback
{
    BluetoothAddress bluetoothAddress;
    BluetoothHhConnectionState state;
};

struct InfoFromHidInfoCallback
{
    BluetoothAddress bluetoothAddress;
    BluetoothHhHidInfo info;
};

struct InfoFromProtocolModeCallback
{
    BluetoothAddress bluetoothAddress;
    BluetoothHhStatus status;
    BluetoothHhProtocolMode mode;
};

struct InfoFromIdleTimeCallback
{
    BluetoothAddress bluetoothAddress;
    BluetoothHhStatus status;
    int idleRate;
};

struct InfoFromGetReportCallback
{
    BluetoothAddress bluetoothAddress;
    BluetoothHhStatus status;
    int rptSize;
    HidReport rptData;
    //uint8_t rptData[BUFFER_SIZE_OF_HID_DATA];
};

struct InfoFromVirtualUnplugCallback
{
    BluetoothAddress bluetoothAddress;
    BluetoothHhStatus status;
};

struct InfoFromHandshakeCallback
{
    BluetoothAddress bluetoothAddress;
    BluetoothHhStatus status;
};

struct ZeroRetranDetails
{
    bool enable;
};

struct ConnectionDetails
{
    uint8_t numPending;
};

const size_t ExtensionDataSizeMax = 32;
struct InfoFromExtensionCallbacks
{
    BluetoothAddress bluetoothAddress;
    ExtEventType eventType;
    BluetoothHhStatus status;
    union
    {
        ZeroRetranDetails zeroRetran;
        ConnectionDetails connections;

        char _dummy[ExtensionDataSizeMax];  // for the future
    };
};

struct InfoFromSetReportStatusCallback
{
    BluetoothAddress bluetoothAddress;
    BluetoothHhStatus status;
};

struct InfoFromGetReportStatusCallback
{
    BluetoothAddress bluetoothAddress;
    BluetoothHhStatus status;
    uint16_t length;
    uint8_t  data[639];
};

const size_t RfListSizeMax = 8;
const size_t PlrListSizeMax = RfListSizeMax;
struct PlrStatistics
{
    uint32_t sampleNumber;
    struct
    {
        BluetoothAddress bluetoothAddress;
        uint32_t plr; // unused
        uint32_t received;
    } plrList[PlrListSizeMax];
};

const size_t channelMapListSizeMax = RfListSizeMax;
struct ChannelMap
{
    struct
    {
        BluetoothAddress bluetoothAddress;
        ChannelInfo channelInfo;
    } chList[channelMapListSizeMax];
};

struct BleAdvertiseFilter
{
    uint8_t         filterIndex;
    BleAdStructure  structure;
    uint8_t         mask[BleAdStructurePayloadSizeMax];
    uint8_t         maskeLength;                        // Length of mask, not AD structure length field

    bool operator == (const BleAdvertiseFilter& filter)
    {
        if (filterIndex == filter.filterIndex &&
            structure.adType == filter.structure.adType &&
            structure.length == filter.structure.length &&
            maskeLength == filter.maskeLength)
        {
            for (size_t i = 0; i < BleAdStructurePayloadSizeMax; ++i)
            {
                if (structure.data[i] != filter.structure.data[i] || mask[i] != filter.mask[i])
                {
                    return false;
                }
            }

            return true;
        }

        return false;
    }
};

struct GattId
{
    uint8_t             instanceId;
    GattAttributeUuid   uuid;
};

struct LeConnectionParams
{
    BluetoothAddress bluetoothAddress;
    uint16_t         minConnectionInterval;
    uint16_t         maxConnectionInterval;
    uint16_t         minConnectionEventLen;
    uint16_t         maxConnectionEventLen;
    uint16_t         slaveLatency;
    uint16_t         supervisionTimeout;
    bool             preference;
};

const size_t ServiceUuidMax = 6;         /* Max number of Service UUID's the device can advertise */
struct LeAdvertiseData
{
    uint8_t             len;
    uint8_t             flag;
    uint8_t             data[BleAdvertisePacketSizeMax];
    uint32_t            dataMask;
    uint16_t            appearanceData;
    uint8_t             numServices;
    GattAttributeUuid   uuid[ServiceUuidMax];
    uint8_t             serviceDataLen;
    uint16_t            serviceDataUuid;
    uint8_t             serviceData[BleAdvertisePacketSizeMax];
    bool                isScanResponse;
    uint8_t             txPower;

    // TODO: Add in tBSA_DM_BLE_INT_RANGE, tBSA_DM_BLE_128SERVICE, tBSA_DM_BLE_128SERVICE, tBSA_DM_BLE_PROPRIETARY ?
};

struct GattSdpAttribute
{
    GattId              id;
    GattAttributeType   type;
    uint16_t            handle;
    uint16_t            endGroupHandle;
    uint8_t             property;
    bool                isPrimaryService;
};

typedef enum
{
    BT_OK                = 0,
    BT_BLE_ERR,                     /* ble open fail & default GATT error */
    BT_BLE_ERR_NOT_ENABLED,         /* ble not eneabled */
    BT_BLE_ERR_NO_RESOURCE,         /* out of system resources */
    BT_BLE_ERR_INVALID_PARAM,       /* bad parameter */
    BT_GATT_ERR_INVALID_HANDLE,
    BT_GATT_ERR_READ_PERMISSION,
    BT_GATT_ERR_WRITE_PERMISSION,
    BT_GATT_ERR_INVALID_PDU,
    BT_GATT_ERR_INVALID_AUTH,
    BT_GATT_ERR_REQ_NOT_SUPP,
    BT_GATT_ERR_INVALID_OFFSET,
    BT_GATT_ERR_QUEUE_FULL,
    BT_GATT_ERR_NOT_FOUND,
    BT_GATT_ERR_NOT_LONG,
    BT_GATT_ERR_KEY_SIZE,
    BT_GATT_ERR_INVALID_ATTR_LEN,
    BT_GATT_ERR_UNLIKELY,
    BT_GATT_ERR_INVALID_ENCRYPTION,
    BT_GATT_ERR_INVALID_GROUP_TYPE,
    BT_GATT_ERR_NO_RESOURCES,
    BT_GATT_ERR_INVALID_PARAM,
    BT_GATT_ERR_INTERNAL,
    BT_GATT_ERR_WRONG_STATE,
    BT_GATT_ERR_DB_FULL,
    BT_GATT_ERR_BUSY,
    BT_GATT_ERR,
    BT_GATT_ERR_STARTED,
    BT_GATT_ERR_PENDING,
    BT_GATT_ERR_MORE,
    BT_GATT_ERR_INVALID_CONFIG,
    BT_GATT_ERR_ENCRYPTED_MITM,
    BT_GATT_ERR_ENCRYPTED_NO_MITM,
    BT_GATT_ERR_NOT_ENCRYPTED,
    BT_GATT_ERR_CONGESTED,
    BT_GATT_ERR_NO_RESULT,
    BT_HS_HID_NOT_READY,            /* handshake error : device not ready */
    BT_HS_INVALID_RPT_ID,           /* handshake error : invalid report ID */
    BT_HS_TRANS_NOT_SPT,            /* handshake error : transaction not supported */
    BT_HS_INVALID_PARAM,            /* handshake error : invalid parameter */
    BT_HS_ERROR,                    /* handshake error : unspecified HS error */
    BT_HH_ERR,                      /* general BTA HH error */
    BT_ERR_SDP,                     /* SDP error */
    BT_ERR_PROTO,                   /* SET_Protocol error, only used in BTA_HH_OPEN_EVT callback */
    BT_ERR_DB_FULL,                 /* device database full error, used  */
    BT_ERR_TOD_UNSPT,               /* type of device not supported */
    BT_ERR_NO_RES,                  /* out of system resources */
    BT_ERR_AUTH_FAILED,             /* authentication fail */
    BT_ERR_HDL,
    BT_ERR,                         /* default error */
    BT_ERR_BAD_PARAM,               /* bad parameter error */
    BT_ERR_BAD_RESPONSE,            /* bad message/response error */
    BT_ERR_BUSY,                    /* busy/not ready error */
    BT_ERR_CONNECTION,              /* connection errors */
    BT_ERR_OPEN,                    /* open/enable errors */
    BT_ERR_CLIENT,                  /* general client errors */
    BT_ERR_SERVER,                  /* general server errors */
    BT_ERR_SECURITY,                /* security errors */
    BT_ERR_FILE_SYS,                /* file system errors */
    BT_ERR_AUDIO_VIDEO,             /* audio video errors */
    BT_ERR_DISCOVERY,               /* discovery errors */
    BT_ERR_TEST_MODE,               /* test mode errors */
    BT_ERR_DM,                      /* DM errors errors */
    BT_ERR_ROBSON                   /* robson errors */
} BluetoothLeStatus;

enum : uint8_t
{
    BLE_ADV_ADDRESS_TYPE_PUBLIC = 0x00,
    BLE_ADV_ADDRESS_TYPE_RANDOM = 0x01,
};

/**
* @brief       BLE のスキャン状態を示す列挙型です
*/
typedef enum : uint8_t
{
    BLE_SCAN_STATE_SCANNING = 0x00,             //!< スキャン中です
    BLE_SCAN_STATE_COMPLETED,                   //!< スキャンが完了しました
    BLE_SCAN_STATE_FOUND_DEVICE,                //!< スキャン中にデバイスを発見しました
} BleScanState;

struct InfoFromLeScanStateChangedCallback
{
    BluetoothLeStatus   status;
    BleScanState        state;
    BleAdvertisePduType pduType;                                    // only for BLE_SCAN_STATE_FOUND_DEVICE
    uint8_t             addressType;                                // only for BLE_SCAN_STATE_FOUND_DEVICE
    BluetoothAddress    address;                                    // only for BLE_SCAN_STATE_FOUND_DEVICE
    BleAdStructure      adStructures[BleAdStructureCountMax];       // only for BLE_SCAN_STATE_FOUND_DEVICE
    uint8_t             adStructureNum;                             // only for BLE_SCAN_STATE_FOUND_DEVICE
    int                 rssi;                                       // only for BLE_SCAN_STATE_FOUND_DEVICE
};

typedef enum : uint8_t
{
    BLE_SCAN_FILTER_OP_COND_ADD = 0,
    BLE_SCAN_FILTER_OP_COND_DELETE,
    BLE_SCAN_FILTER_OP_COND_CLEAR,
    BLE_SCAN_FILTER_OP_ENABLE,
    BLE_SCAN_FILTER_OP_DISABLE,
} BleScanFilterOperation;

struct InfoFromLeScanFilterStateChangedCallback
{
    BluetoothLeStatus           status;
    BleScanFilterOperation      operation;
};

static const uint8_t BLE_INVALID_GATT_CLIENT_HANDLE = 0xFF;
static const uint8_t BLE_INVALID_GATT_SERVER_HANDLE = 0xFF;
static const uint8_t BLE_GATT_CLIENT_NUM_MAX = 4;       // BLE_CLIENT_MAX
static const uint8_t BLE_GATT_SERVER_NUM_MAX = 4;       // BLE_SERVER_MAX
struct InfoFromLeAppStateChangedCallback
{
    BluetoothLeStatus status;
    union
    {
        uint8_t clientIf;
        uint8_t serverIf;
    };
    bool isRegistered;
};

/**
* @brief       BLE の接続状態を表す列挙型です
*/
typedef enum : uint8_t
{
    BLE_CONN_STATE_CONNECTED = 0x00,     //!< GATT Server またはGATT Client に接続されています
    BLE_CONN_STATE_CANCELED,                    //!< GATT Server またはGATT Client への接続をキャンセルしています
    BLE_CONN_STATE_DISCONNECTED,                //!< GATT Server またはGATT Client と接続されていません
} BleConnectionState;

struct InfoFromLeConnStateChangedCallback
{
    BluetoothLeStatus   status;
    BleConnectionState  connState;
    union
    {
        uint8_t clientIf;
        uint8_t serverIf;
    };
    uint32_t            connId;
    BluetoothAddress    address;
    uint16_t            reason;             // only for BLE_CONN_STATE_DISCONNECTED
};

struct InfoFromLeConnParamUpdateCallback
{
    BluetoothLeStatus   status;
    uint32_t            connId;
    uint16_t            interval;
    uint16_t            slaveLatency;
    uint16_t            supervisionTimeout;
};

struct InfoFromLeConnParamUpdateReqCallback
{
    uint32_t    connId;
    uint16_t    intervalMin;
    uint16_t    intervalMax;
    uint16_t    slaveLatency;
    uint16_t    supervisionTimeout;
};

typedef enum : uint8_t
{
    BLE_GATT_REQ_TYPE_READ = 0x00,
    BLE_GATT_REQ_TYPE_WRITE,
} BleGattReqType;

struct InfoFromLeGattOperationCallback
{
    BluetoothLeStatus   status;
    uint32_t            connId;
    GattOperationType   operation;
    GattAttributeUuid   serviceUuid;
    GattAttributeUuid   charcteristicUuid;
    GattAttributeUuid   descriptorUuid;
    uint16_t            length;
    uint8_t             value[BUFFER_SIZE_OF_BLE_GATT_ATTRIBUTE];
};

struct InfoFromLeGattIndicationResponseNeededCallback
{
    uint32_t    connId;
    GattId      serviceId;
    GattId      characteristicId;
    bool        isPrimaryService;
};

/* TODO: bluetooth_GattAttribute.h defines GATT_ATTR_NUM_MAX 50, however BSA is 10. Leaving at 10 for now */
const size_t MaxSdpAttributes = 10;
struct InfoFromLeClientGattServiceDiscoveryCallback
{
    BluetoothLeStatus   status;
    uint32_t            connId;
    uint16_t            numAttributes;
    BluetoothAddress    address;
    GattSdpAttribute    attributes[MaxSdpAttributes];
};

struct InfoFromLeClientMtuConfigurationCallback
{
    BluetoothLeStatus   status;
    uint32_t            connId;
    uint16_t            mtu;
};


/**
* @brief       GATT Attribute に許可する操作を表す列挙型ビットフラグです
*/
enum GattAttributePermission : uint16_t
{
    GattAttributePermission_Read = (0x0001 << 0),
    GattAttributePermission_ReadEnc = (0x0001 << 1),
    GattAttributePermission_ReadEncMitm = (0x0001 << 2),
    GattAttributePermission_Write = (0x0001 << 4),
    GattAttributePermission_WriteEnc = (0x0001 << 5),
    GattAttributePermission_WriteEncMitm = (0x0001 << 6),
    GattAttributePermission_WriteSigned = (0x0001 << 7),
    GattAttributePermission_WriteSignedMitm = (0x0001 << 8),
};

struct InfoFromLeServerProfileChangedCallback
{
    BluetoothLeStatus   status;
    uint8_t             serverIf;
    GattAttributeUuid   uuid;
    uint16_t            serviceHandle;
    uint16_t            attributeHandle;
    GattAttributeType   type;
    uint8_t             property;
    bool                isPrimaryService;
};

struct InfoFromLeServerGattReqCallback
{
    BluetoothLeStatus   status;
    BluetoothLeStatus   responseStatus;
    uint16_t            connId;
    BleGattReqType      requestType;
    uint16_t            serviceHandle;
    uint16_t            attributeHandle;
    GattAttributeType   attrType;
    uint8_t             data[BUFFER_SIZE_OF_BLE_GATT_ATTRIBUTE];
    uint16_t            length;
    uint16_t            offset;
};

const size_t LowEnergyDataSizeMax = 100;
struct InfoFromLowEnergyCallbacks
{
    BleEventType        eventType;
    BluetoothLeStatus   status;
    BluetoothAddress    bluetoothAddress;
    BluetoothName       bluetoothName;
    ClassOfDevice       classOfDevice;
    bool                isPrimary;
    uint8_t             inquiryResultType;
    uint8_t             bleAddressType;
    uint16_t            rssi;
    GattAttributeUuid   uuid;
    GattAttributeUuid   serviceUuid;
    GattAttributeUuid   charUuid;
    GattAttributeUuid   attrUuid;
    uint16_t            connectionId;
    uint16_t            len;
    uint8_t             data[LowEnergyDataSizeMax];
    uint16_t            mtu;
    uint8_t             availFilterNum;
    bool                filterEnabled;
    union
    {
        uint8_t serverIf;
        uint8_t clientIf;
    };
};

}}
