﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file   bluetooth_Api.h
 * @brief  Bluetooth API
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/os/os_TransferMemory.h>
#include <nn/os/os_SharedMemory.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/applet/applet_FundamentalTypes.h>

#include <nn/bluetooth/bluetooth_Types.h>
#include <nn/settings/system/settings_BluetoothDevices.h>

namespace nn { namespace bluetooth {

/**
 * @name APIs for Bluetooth
 * @{
 */

/*************************************************************************
 * Core APIs
 ************************************************************************/
/**
 * @brief  Initialize the service of bluetooth ByDfc.
 *
 * @details
 * NON_BLOCKING.
 * Currently DFC mode is not supported. This API initializes
 * functions for the SF. If you use the Bluetooth module By
 * Dfc, you have to link the module as a library.This API
 * should be called before any other APIs.
 *
 * @retval void.
 *
 */
void InitializeBluetoothDriverByDfc() NN_NOEXCEPT;

/**
 * @brief  Initialize the service of bluetooth.
 *
 * @details
 * BLOCKING.
 * This API initialize functions for the SF.
 * This API should be called before any other APIs.
 *
 * @retval void.
 *
 */
void InitializeBluetoothDriver() NN_NOEXCEPT;

/**
 * @brief   Get if bluetooth process is in manufacturing test mode
 *
 * @details
 * BLOCKING
 *
 * @retval  if bluetooth process is in manufacturing mode, true; otherwise false.
 *
 */
bool GetIsManufacturingMode() NN_NOEXCEPT;

/**
 * @brief  Initialize bluetooth chip.
 *
 * @details
 * BLOCKING 100ms max.
 * In HOS we will use it to control BT_RESET and also
 * if internal error occurs, this API returns failure.
 * At that case, there were no changes to the "pSystemEvent".
 *
 * @param[in] pSystemEvent Used to signal when a bluetooth event occurs.
 *
 * @retval ResultSuccess	 If okay.
 * @retval ResultUnsupported Wrong board type.
 * @retval ResultFail		 Event creation/initialization failure.
 *
 */
nn::Result InitializeBluetooth(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT;

/**
 * @brief  Enable the bluetooth Stack.
 *
 * @details
 * BLOCKING 500ms max.
 * Used to enable the stack. Enable the core stuff such
 * as the BSA server, HID queues, global heap, etc.
 *
 * @retval ResultSuccess	 If okay.
 * @retval ResultFail		 BtHalBsaCoreClientInit failure.
 *
 */
nn::Result EnableBluetooth() NN_NOEXCEPT;

/**
 * @brief  Disables bluetooth.
 *
 * @details
 * BLOCKING ~5s max. This could be shortened.
 * This API is pairs with "EnableBluetooth".
 * This API cuts the power to the bluetooth chip,
 * destroys threads, events, and queues, closes
 * the client & servers, etc.
 *
 * @retval ResultSuccess.
 *
 */
nn::Result DisableBluetooth() NN_NOEXCEPT;

/**
 * @brief  Cleanup the bluetooth stack.
 *
 * @details
 * BLOCKING
 * This API destroys threads, events, and queues, closes
 * the client & servers, etc.
 *
 * @retval ResultSuccess.
 *
 */
nn::Result CleanupBluetooth() NN_NOEXCEPT;

/**
 * @brief  Finalize the service of Bluetooth.
 *
 * @details
 * BLOCKING.
 * This API finalize functions for the SF.
 * This API is pairs with "InitializeBluetoothDriver".
 * This API should be called last when shutting down.
 *
 * @retval void.
 *
 */
void FinalizeBluetoothDriver() NN_NOEXCEPT;

/**
 * @brief  Get all adapter properties.
 *
 * @details
 * BLOCKING.
 * Get all supported adapter properties. bluetoothAddress, classOfDevice,
 * bluetoothName, and featureSet are currently supported
 *
 * @param[out] pAdapterProperty Property value structure.
 *
 * @retval ResultSuccess 	  If okay.
 * @retval ResultBadParameter Null or invalid parameter passed.
 *
 */
nn::Result GetAdapterProperties(AdapterProperty* pAdapterProperty) NN_NOEXCEPT;

/**
 * @brief  Get adapter properties.
 *
 * @details
 * BLOCKING.
 * Get a single Adapter(host) property.
 *
 * @param[in] propertyType BT_PROPERTY_BDADDR/BT_PROPERTY_BDNAME/BT_PROPERTY_CLASS_OF_DEVICE are supported currently.
 * @param[in] value        Property values array.
 * @param[in] sizeOfValue  Array size.
 *
 * @retval ResultSuccess 	  If okay.
 * @retval ResultBadParameter Null or invalid parameter passed.
 *
 */
nn::Result GetAdapterProperty(BluetoothPropertyType propertyType, uint8_t value[] ,int16_t sizeOfValue) NN_NOEXCEPT;

/**
 * @brief  Set adapter properties.
 *
 * @details
 * BLOCKING.
 * Set a single Adapter(host) property. Usually this
 * is only used for GET BD_ADDR however for our system
 * we need to set the host bd_addr.
 *
 * @param[in] propertyType Only BT_PROPERTY_BDADDR & BT_PROPERTY_SUPER_BDADDR are supported currently.
 * @param[in] value        Property values array.
 * @param[in] sizeOfValue  Array size.
 *
 * @retval ResultSuccess 	  If okay.
 * @retval ResultBadParameter Null or invalid parameter passed.
 *
 */
nn::Result SetAdapterProperty(BluetoothPropertyType propertyType, const uint8_t value[] ,int16_t sizeOfValue) NN_NOEXCEPT;

/**
 * @brief  Start discovery.
 *
 * @details
 * BLOCKING
 * Begins the discovery process and state machine. The radio
 * configuration should not be modified during this time.
 *
 * @retval ResultSuccess      If okay.
 * @retval ResultBadParameter Invalid passed.
 * @retval ResultClientError  bsa_send_receive_message failure.
 *
 */
nn::Result StartDiscovery() NN_NOEXCEPT;

/**
 * @brief  Cancel discovery.
 *
 * @details
 * BLOCKING
 * Aborts discovery.
 *
 * @retval ResultSuccess.
 *
 */
nn::Result CancelDiscovery() NN_NOEXCEPT;

/**
 * @brief  Create a bond for specific device.
 *
 * @details
 * BLOCKING
 * This API should be used after receiving a device found callback.
 * Discovery should be canceled before calling this API.
 *
 * @param[in] pBluetoothAddress Address of the target device.
 * @param[in] transport.
 *
 * @retval ResultSuccess.
 *
 */
nn::Result CreateBond(const BluetoothAddress *pBluetoothAddress, BluetoothTransport transport) NN_NOEXCEPT;

/**
 * @brief  Remove a bond for specific device.
 *
 * @details
 * BLOCKING
 * Removes a device from the database in the Bluetooth process and in the
 * Bluetooth chip. Typically called when devices are un-paired from the system.
 *
 * @param[in] pBluetoothAddress Address of the target device.
 *
 * @retval ResultSuccess      If okay.
 * @retval ResultBadParameter Invalid pBluetoothAddress passed.
 * @retval ResultClientError  bsa_send_receive_message failure.
 *
 */
nn::Result RemoveBond(const BluetoothAddress *pBluetoothAddress) NN_NOEXCEPT;

/**
 * @brief  Cancel a bond for specific device.
 *
 * @details
 * BLOCKING
 *
 * @param[in] pBluetoothAddress Address of the target device.
 *
 * @retval ResultSuccess      If okay.
 * @retval ResultBadParameter Invalid pBluetoothAddress passed.
 * @retval ResultClientError  bsa_send_receive_message failure.
 *
 */
nn::Result CancelBond(const BluetoothAddress *pBluetoothAddress) NN_NOEXCEPT;

/**
 * @brief  Reply to the PinRequest event.
 *
 * @details
 * BLOCKING
 * When you call the CreateBond API or a Bluetooth device issues the page,
 * you will get the PinRequest event if the Pin security is chosen by each
 * device's capability.
 *
 * @param[in] pBluetoothAddress Address of the target device.
 * @param[in] isAccept     		1 to accept, 0 to reject.
 * @param[in] pPinCode 			Pin code.
 * @param[in] pinLength			Pin code length.
 *
 * @retval ResultSuccess      If okay.
 * @retval ResultBadParameter Invalid pBluetoothAddress passed.
 * @retval ResultClientError  bsa_send_receive_message failure.
 *
 */
nn::Result PinReply(const BluetoothAddress *pBluetoothAddress, bool isAccept, const BluetoothPinCode *pPinCode, uint8_t pinLength) NN_NOEXCEPT;

/**
 * @brief  Reply to the SspRequest event.
 *
 * @details
 * BLOCKING
 * When you call the CreateBond API or a Bluetooth device issues the page,
 * you will get the SspRequest event if the Ssp security is chosen by each
 * device's capability.
 *
 * @param[in] pBluetoothAddress Address of the target device.
 * @param[in] variant     		Numeric comparison or passkey entry.
 * @param[in] isAccept 			1 to accept, 0 to reject.
 * @param[in] passkey			Pass key provided.
 *
 * @retval ResultSuccess      If okay.
 * @retval ResultBadParameter Invalid pBluetoothAddress passed.
 * @retval ResultClientError  bsa_send_receive_message failure.
 *
 */
nn::Result SspReply(const BluetoothAddress *pBluetoothAddress, BluetoothSspVariant variant, bool isAccept, uint32_t passkey) NN_NOEXCEPT;

//[Todo]To separate the GetAPI. To care padding, 64bit.
/**
 * @brief  Get the information from the Server.
 *
 * @details
 * BLOCKING
 * When you get a signal of the event object that initialized in
 * the InitializeBluetooth, you have to call this API. If you don't
 * call this API for it, the information from the Server is queued
 * and discarded. It causes a gap of a state.
 *
 * @param[out] pOutEventType Event type.
 * @param[out] outBuffer     Event Data.
 * @param[in]  sizeOfBuffer  Size of event data.
 *
 * @retval ResultSuccess.
 *
 */
nn::Result GetEventInfo(nn::bluetooth::EventType *pOutEventType, uint8_t outBuffer[], uint16_t sizeOfBuffer) NN_NOEXCEPT;

/*************************************************************************
 * Unsupported Core APIs
 ************************************************************************/
//static const void* GetProfileInterface(const char *pProfileId) NN_NOEXCEPT;
//nn::Result DutModeConfigure(uint8_t enable) NN_NOEXCEPT;
//nn::Result DutModeSend(uint16_t opcode, uint8_t *buffer, uint8_t len) NN_NOEXCEPT;
//nn::Result LEtestMode(uint16_t opcode, uint8_t *buffer, uint8_t len) NN_NOEXCEPT;
//nn::Result ConfigHciSnoopLog(uint8_t enable) NN_NOEXCEPT;
//nn::Result ReadEnergyInfo(void) NN_NOEXCEPT;
//nn::Result GetAdapterProperty(BluetoothPropertyType type) NN_NOEXCEPT; // Waiting for implementing in the HAL
//nn::Result GetRemoteDeviceProperties(Btbdaddr *pRemoteAddr) NN_NOEXCEPT;
//nn::Result GetRemoteDeviceProperty(Btbdaddr *pRemoteAddr,BluetoothPropertyType type) NN_NOEXCEPT;
//nn::Result SetRemoteDeviceProperty(Btbdaddr *pRemoteAddr,const BluetoothProperty *property) NN_NOEXCEPT;
//nn::Result GetRemoteServiceRecord(Btbdaddr *pRemoteAddr,BluetoothUUID *uuid) NN_NOEXCEPT;
//nn::Result GetRemoteServices(Btbdaddr *pRemoteAddr) NN_NOEXCEPT;

/*************************************************************************
 * Hid APIs
 ************************************************************************/
/**
 * @brief  Initialize a Hid profile of the Bluetooth stack.
 *
 * @details
 * BLOCKING.
 * This API should be called before any other HID APIs. Currently this
 * API needs to be called after EnableBluetooth.
 *
 * @param[in] pSystemEvent     Used to signal when a Hid event occurs.
 * @param[in] interfaceVersion 0:(default value) Bluetooth process uses pairing settings directly (legacy support, will be deprecated).
 *							   1:upper layer manages pairing settings, using HidSetPairedDevice/HidGetPairedDevice APIs.
 *
 * @retval ResultSuccess	  If okay.
 * @retval ResultBadParameter Invalid callbacks passed.
 * @retval ResultFail		  Event creation/initialization failure.
 *
 */
nn::Result InitializeHid(nn::os::SystemEventType* pSystemEvent, uint16_t interfaceVersion = 0) NN_NOEXCEPT;

/**
 * @brief  Cleanup a Hid profile of the Bluetooth stack.
 *
 * @details
 * BLOCKING.
 * This API is pairs with "InitializeHid". You should destroy the
 * system event that is initialized by using the "InitializeHid".
 *
 * @retval ResultSuccess.
 *
 */
nn::Result CleanupHid() NN_NOEXCEPT;

/**
 * @brief  Connect to the specific HID device.
 *
 * @details
 * BLOCKING.
 * Feature is currently unsupported!
 * Bluetooth chip send the page to the device. If the paring information
 * of the device is already stored to the database of the Bluetooth stack,
 * connection is automatically finished. If not, you will get the
 * Pin/SspRequest event.
 *
 * @param[in] pBluetoothAddress Address of the target device.
 *
 * @retval ResultSuccess.
 *
 */
nn::Result HidConnect(const BluetoothAddress *pBluetoothAddress) NN_NOEXCEPT;

/**
 * @brief  Disconnect to the specific HID device.
 *
 * @details
 * BLOCKING.
 * Disconnect to the specific HID divece. This API does not remove the
 * paring information from the database of the Bluetooth stack. If you
 * want to it, you should use the "RemoveBond" API in addition to this API.
 *
 * @param[in] pBluetoothAddress Address of the target device.
 *
 * @retval ResultSuccess      If okay.
 * @retval ResultBadParameter Invalid pBluetoothAddress passed.
 * @retval ResultClientError  bsa_send_receive_message failure.
 *
 */
nn::Result HidDisconnect(const BluetoothAddress *pBluetoothAddress) NN_NOEXCEPT;

/**
 * @brief  Send a HID packet to the device.
 *
 * @details
 * BLOCKING.
 * You should create the Hid Data including the ReportID. The HIDP-Hdr
 * is automatically added by the stack. You can use the length field of
 * the "HidData" structure but you should create the whole size of buffer
 * for the structure. Because the SF transport whole size of "HidData"
 * structure.
 * Note: We have a plan to create the variable size transfer if we will
 * use the Bluetooth module as a process.
 *
 * @param[in] pBluetoothAddress Address of the target device.
 * @param[in] buffer            Hid data buffer.
 *
 * @retval ResultSuccess      	   If okay.
 * @retval ResultHandshakeNotReady Bonding in progress, can't send data.
 * @retval ResultBadParameter 	   Invalid pBluetoothAddress passed.
 * @retval ResultClientError  	   bsa_send_receive_message failure.
 *
 */
nn::Result HidSendData(const BluetoothAddress *pBluetoothAddress, const HidData* pData) NN_NOEXCEPT;

/**
 * @brief  Send a HID packet to the device.
 *
 * @details
 * BLOCKING.
 * This API requires "HidData" format for the "buffer".
 * You can use variable size transfer between the SF.
 * Between the SF, copied the part of "buffer" indicated by "sizeOfBuffer".
 *
 * @param[in] pBluetoothAddress Address of the target device.
 * @param[in] buffer            Data buffer.
 * @param[in] sizeOfBuffer      Data size.
 *
 * @retval ResultSuccess      If okay.
 * @retval ResultBadParameter Invalid pBluetoothAddress passed.
 * @retval ResultClientError  bsa_send_receive_message failure.
 *
 */
nn::Result HidSendData(const BluetoothAddress *pBluetoothAddress, const uint8_t buffer[], uint16_t sizeOfBuffer) NN_NOEXCEPT;

/**
 * @brief  Send a SetReport packet to the device.
 *
 * @details
 * BLOCKING.
 * You can use the length field of the "HidData" structure but
 * you should create the whole size of buffer for the structure.
 * Because the SF transport whole size of "HidData" structure.
 * note: We have a plan to create the variable size transfer if
 * we will use the Bluetooth module as a process.
 *
 * @param[in] pBluetoothAddress Address of the target device.
 * @param[in] reportType        Input, output, or feature type.
 * @param[in] pData             Report data.
 *
 * @retval ResultSuccess      If okay.
 * @retval ResultBadParameter Invalid pBluetoothAddress passed.
 * @retval ResultClientError  bsa_send_receive_message failure.
 *
 */
nn::Result HidSetReport(const BluetoothAddress *pBluetoothAddress, BluetoothHhReportType reportType, const HidData* pData) NN_NOEXCEPT;

/**
 * @brief  Send a GetReport packet to the device.
 *
 * @details
 * BLOCKING.
 *
 * @param[in] pBluetoothAddress Address of the target device.
 * @param[in] reportType        Input, output, or feature type.
 * @param[in] reportId          0x30-0x33, 0x3F.
 *
 * @retval ResultSuccess      If okay.
 * @retval ResultBadParameter Invalid pBluetoothAddress passed.
 * @retval ResultClientError  bsa_send_receive_message failure.
 *
 */
nn::Result HidGetReport(const BluetoothAddress *pBluetoothAddress, BluetoothHhReportType reportType, uint8_t reportId) NN_NOEXCEPT;

/**
 * @brief  Send wake command to a remote device.
 *
 * @details
 * BLOCKING.
 * After reset controllers are in a default sleep state. Use
 * this API to wake them. Device must be paired beforehand.
 * If a controller was successfully woken, a ConnectionStateCallback
 * will be sent to the upper layer within about 100ms with state
 * BTHH_CONN_STATE_CONNECTED. If the controller can't be woken (bad battery,
 * or out of range), a ConnectionStateCallback will be sent to the upper
 * layer within a few seconds with state BTHH_CONN_STATE_FAILED_GENERIC.
 *
 * @param[in] pBluetoothAddress Address of device to be awoken.
 *
 * @retval ResultSuccess      If okay.
 * @retval ResultBadParameter Invalid pBluetoothAddress passed.
 * @retval ResultClientError  bsa_send_receive_message failure.
 *
 */
nn::Result HidWakeController(const BluetoothAddress *pBluetoothAddress) NN_NOEXCEPT;

/**
 * @brief  Add a paired device's settings.
 *
 * @details
 * BLOCKING.
 * This API is for adding paired device settings to the
 * remote devices database. To be used upon startup
 * to get info from NAND with clients version 1 and above.
 *
 * @param[in] pDevice Settings to be added.
 *
 * @retval ResultSuccess 	  If okay.
 * @retval ResultBadParameter Null pointer passed.
 * @retval ResultNoResources  Database is full.
 * @retval ResultClientError  bsa_send_receive_message failure.
 *
 */
nn::Result HidAddPairedDevice(const nn::settings::system::BluetoothDevicesSettings *pDevice) NN_NOEXCEPT;

/**
 * @brief  Get a paired device's settings.
 *
 * @details
 * BLOCKING.
 * This API is for getting paired device settings out of the
 * remote devices database. To be used after a new device
 * pairs with clients version 1 and above.
 *
 * @param[in]  pBluetoothAddress The device you want settings for.
 * @param[out] pDeviceOut		 Settings info.
 *
 * @retval ResultSuccess 	  If okay.
 * @retval ResultBadParameter Null pointer or invalid pBluetoothAddress.
 * @retval ResultBadResponse  Info is empty for pBluetoothAddress.
 *
 */
nn::Result HidGetPairedDevice(const BluetoothAddress *pBluetoothAddress, nn::settings::system::BluetoothDevicesSettings *pDeviceOut) NN_NOEXCEPT;

//[Todo]To separate the GetAPI. To care padding, 64bit.
/**
 * @brief  Get the information from the Server.
 *
 * @details
 * BLOCKING.
 * When you get a signal of the event object that initialized in
 * the InitializeHid, you have to call this API. If you don't call
 * this API for it, the information from the Server is queued
 * and discarded. It causes a gap of a state.
 *
 * @param[out] outEventType Hid event type.
 * @param[out] outBuffer    Hid data.
 * @param[in]  sizeOfBuffer Size of data.
 *
 * @retval ResultSuccess.
 *
 */
nn::Result HidGetEventInfo(nn::bluetooth::HidEventType *outEventType, uint8_t outBuffer[], uint16_t sizeOfBuffer) NN_NOEXCEPT;

/**
 * @brief  Register Hid events.
 *
 * @details
 * BLOCKING.
 * Attach a system event for notification of HID input report,
 * SetReport and GetReport.
 *
 * @param[in] pSystemEvent Hid event to be registered.
 *
 * @retval ResultSuccess if okay, otherwise ResultFail.
 *
 */
nn::Result RegisterHidReportEvent(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT;

/**
 * @brief  Get Hid information from the Server.
 *
 * @details
 * BLOCKING.
 * When you get a signal of the event object that is initialized in
 * the RegisterHidReportEvent, you have to call this API.
 *
 * @param[out] outEventType Hid report event type.
 * @param[out] outBuffer    Hid report data.
 * @param[in]  sizeOfBuffer Size of report data.
 *
 * @retval ResultSuccess.
 *
 */
nn::Result HidGetReportEventInfo(nn::bluetooth::HidEventType *outEventType, uint8_t outBuffer[], uint16_t sizeOfBuffer) NN_NOEXCEPT;

/*************************************************************************
 * Unsupported Hid APIs
 ************************************************************************/
/* Virtual UnPlug (VUP) the specified HID device */
//nn::Result VirtualUnplug(Btbdaddr *pBdAddr) NN_NOEXCEPT;

/* Set the HID device descriptor for the specified HID device. */
//nn::Result SetInfo(Btbdaddr *pBdAddr, BluetoothHhHidInfo HidInfo) NN_NOEXCEPT;

/* Get the HID proto mode. */
//nn::Result GetProtocol(Btbdaddr *pBdAddr, BluetoothHhProtocolMode protocolMode) NN_NOEXCEPT;

/* Set the HID proto mode. */
//nn::Result SetProtocol(Btbdaddr *pBdAddr, BluetoothHhProtocolMode protocolMode) NN_NOEXCEPT;

/*************************************************************************
 * Extension APIs
 ************************************************************************/
/**
 * @brief  Sets Tsi mode.
 *
 * @details
 * BLOCKING.
 * Tsi values 0-10 & 0xFF are valid. 0xFF means active mode. To switch
 * between TSI modes, you must first be in active mode.
 *
 * @param[in] pBluetoothAddress Address of device to be set.
 * @param[in] tsiNumber         Tsi mode to be set.
 *
 * @retval ResultSuccess If okay, client or robson error code otherwise.
 *
 */
nn::Result ExtSetTsi(const BluetoothAddress *pBluetoothAddress, uint8_t tsiNumber) NN_NOEXCEPT;

/**
 * @brief  Sets burst mode on/off.
 *
 * @details
 * BLOCKING.
 * Cannot be called while bonding, will error and exit immediately otherwise.
 *
 * @param[in] pBluetoothAddress Address of device to be set.
 * @param[in] isBurst 			Set false to exit burst, set true to set burst.
 *
 * @retval ResultSuccess If okay, client or robson error code otherwise.
 *
 */
nn::Result ExtSetBurstMode(const BluetoothAddress *pBluetoothAddress, bool isBurst) NN_NOEXCEPT;

/**
 * @brief  Sets 0-re-transmission.
 *
 * @details
 * BLOCKING.
 *
 * @param[in] pBluetoothAddress Address of device to be set.
 * @param[in] numReportIds	    Must not be greater than MAX_NUM_ZERORETRAN_REPORTID.
 * @param[in] pReportIds
 *
 * @retval ResultSuccess If okay, client or robson error code otherwise.
 *
 */
nn::Result ExtSetZeroRetran(const BluetoothAddress *pBluetoothAddress, uint8_t numReportIds, uint8_t *pReportIds) NN_NOEXCEPT;

/**
 * @brief  Sets the MC mode.
 *
 * @details
 * BLOCKING 500ms max.
 * Enables or disables mobile coexistent mode. Will error if you set
 * the same mode it's already in.
 *
 * @param[in] isMcModeEnabled Set false to turn off mc mode, set true to turn on mc mode.
 *
 * @retval ResultSuccess If okay, client or robson error code otherwise.
 *
 */
nn::Result ExtSetMcMode(bool isMcModeEnabled) NN_NOEXCEPT;

/**
 * @brief  Starts LLR scan mode.
 *
 * @details
 * BLOCKING 750ms max.
 * This API is to be used when LLR scan mode (sleep) is off. Otherwise
 * will print that it's already sleeping and exit immediately. Currently
 * this API waits 500ms after sending disconnects to connected controllers,
 * but this should be done at the upper layer. It also waits up to an
 * additional 50ms for the set-config callback confirming LLR scan has started.
 *
 * @retval ResultSuccess If okay, client or robson error code otherwise.
 *
 */
nn::Result ExtStartLlrMode() NN_NOEXCEPT;

/**
 * @brief  Exits LLR scan.
 *
 * @details
 * BLOCKING 250ms max.
 * This API is to be used when LLR scan mode (sleep) is on. Otherwise
 * will print that it's already awake and exit immediately.
 *
 * @retval ResultSuccess If okay, client or robson error code otherwise.
 *
 */
nn::Result ExtExitLlrMode() NN_NOEXCEPT;

/**
 * @brief  Sets the radio on/off.
 *
 * @details
 * BLOCKING 2000ms max.
 * Turning off the radio stops bluetooth communication and
 * disconnects any controllers connected.  Turning on the radio
 * enables bluetooth communication and allows controllers to connect.
 *
 * @param[in] enable Set false to turn off the radio, set true to turn on the radio.
 *
 * @retval ResultSuccess If okay, client or robson error code otherwise.
 *
 */
nn::Result ExtSetRadio(bool enable) NN_NOEXCEPT;

/**
 * @brief  Set the discoverability and connectability on/off.
 *
 * @details
 * BLOCKING.
 * If discoverability is off this device will not show up in other
 * devices discoveries.  If connectability is off new connections
 * will be refused, however existing connections will continue.
 *
 * @param[in] discoverable Set false to be undiscoverable, set true to be discoverable.
 * @param[in] connectable  Set false to be unconnectable, set true to be connectable.
 *
 * @retval ResultSuccess If okay, client error code otherwise.
 *
 */
nn::Result ExtSetVisibility(bool discoverable, bool connectable) NN_NOEXCEPT;

/**
 * @brief  Set TBFC scan on/off.
 *
 * @details
 * BLOCKING.
 *
 * @param[in] enable Set false to disable TBFC scan, set true to enable TBFC scan.
 *
 * @retval ResultSuccess If okay, client error code otherwise.
 *
 */
nn::Result ExtSetTbfcScan(bool enable) NN_NOEXCEPT;

/**
 * @brief  Gets the number of pending connections.
 *
 * @details
 * BLOCKING.
 *
 *
 *
 * @retval ResultSuccess If okay, client error code otherwise.
 *
 */
nn::Result ExtGetPendingConnections() NN_NOEXCEPT;

/**
 * @brief   Gets current AFH channel map.

 * @details
 * BLOCKING
 *
 * @retval  ResultSuccess If okay, client error code otherwise.
 *
 */
nn::Result GetChannelMap(ChannelMap* pOutChannelMap) NN_NOEXCEPT;

/**
 * @brief   EnableBluetoothBoostSetting
 * @details
 * BLOCKING
 *
 * @retval  None
 *
 */
void EnableBluetoothBoostSetting(bool enable) NN_NOEXCEPT;

/**
 * @brief   IsBluetoothBoostSettingEnabled
 * @details
 * BLOCKING
 *
 * @retval  True if enabled, false if disabled
 *
 */
bool IsBluetoothBoostSettingEnabled() NN_NOEXCEPT;

/**
 * @brief   EnableBluetoothAfhSetting
 * @details
 * BLOCKING
 *
 * @retval  None
 *
 */
void EnableBluetoothAfhSetting(bool enable) NN_NOEXCEPT;

/**
 * @brief   IsBluetoothAfhSettingEnabled
 * @details
 * BLOCKING
 *
 * @retval  True if enabled, false if disabled
 *
 */
bool IsBluetoothAfhSettingEnabled() NN_NOEXCEPT;

// For debugging
nn::Result GetLatestPlr(nn::bluetooth::PlrStatistics *outPlrStatistics) NN_NOEXCEPT;

/**
 * @}
 */

 /*************************************************************************
 * Bluetooth Lowenergy Core APIs
 ************************************************************************/

 /**
 * @brief  Initialize bluetooth low energy system events.
 *
 * @details
 * BLOCKING
 *
 * @param[in] pSystemEvent Used to signal when a bluetooth event occurs.
 *
 * @retval ResultSuccess	 If okay.
 * @retval ResultFail		 Event creation/initialization failure.
 *
 */
nn::Result InitializeBluetoothLe(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT;


/**
 * @brief  Finalize bluetooth low energy system events.
 *
 * @details
 * NON_BLOCKING.
 * This API is pairs with "InitializeBluetoothLe". You should destroy the
 * system event that is initialized by using the "InitializeBluetoothLe".
 *
 * @callback None
 *
 * @retval void.
 *
 */
nn::Result CleanupBluetoothLe() NN_NOEXCEPT;

/**
 * @brief  Enable bluetooth low energy.
 *
 * @details
 * BLOCKING.
 * Used to enable the bluetooth low energy service.
 *
 * @callback None
 *
 * @retval ResultSuccess	 If okay.
 * @retval ResultFail		 BtHalBsaCoreClientInit failure.
 *
 */
nn::Result EnableBluetoothLe() NN_NOEXCEPT;

/**
 * @brief  Disables bluetooth low energy.
 *
 * @details
 * BLOCKING.
 * This API disables the bluetooth low energy service.
 *
 * @callback None
 *
 * @retval ResultSuccess.
 *
 */
nn::Result DisableBluetoothLe() NN_NOEXCEPT;

/**
 * @brief  Set the BLE discoverability and connectability on/off.
 *
 * @details
 * BLOCKING.
 * If discoverability is off this device will not show up in other
 * devices discoveries.  If connectability is off new connections
 * will be refused, however existing connections will continue.
 *
 * @param[in] discoverable Set false to be undiscoverable, set true to be discoverable.
 * @param[in] connectable  Set false to be unconnectable, set true to be connectable.
 *
 * @callback None
 *
 * @retval ResultSuccess If okay, client error code otherwise.
 *
 */
nn::Result SetLeVisibility(bool discoverable, bool connectable) NN_NOEXCEPT;

/**
 * @brief  Set the connection parameters
 *
 * @details
 * BLOCKING.
 *
 * @param[in]
 *
 * @callback None
 *
 * @retval ResultSuccess If okay, client error code otherwise.
 *
 */
nn::Result SetLeConnectionParameter(LeConnectionParams connectionParams) NN_NOEXCEPT;

/**
 * @brief  Set the default connection parameters
 *
 * @details
 * BLOCKING.
 *
 * @param[in]
 *
 * @callback None
 *
 * @retval ResultSuccess If okay, client error code otherwise.
 *
 */
nn::Result SetLeDefaultConnectionParameter(LeConnectionParams connectionParams) NN_NOEXCEPT;

/**
 * @brief  Set the advertise data
 *
 * @details
 * BLOCKING.
 *
 * @param[in]
 *
 * @callback None
 *
 * @retval ResultSuccess If okay, client error code otherwise.
 *
 */
nn::Result SetLeAdvertiseData(LeAdvertiseData advertiseData) NN_NOEXCEPT;

/**
 * @brief  Set the advertise data parameters
 *
 * @details
 * BLOCKING.
 *
 * @param[in]
 *
 * @callback None
 *
 * @retval ResultSuccess If okay, client error code otherwise.
 *
 */
nn::Result SetLeAdvertiseParameter(const BluetoothAddress *pBluetoothAddress, uint16_t minAdvInt, uint16_t maxAdvInt) NN_NOEXCEPT;

/*************************************************************************
* Low Energy Discovery APIs
************************************************************************/
/**
 * @brief  Start BLE Scan.
 *
 * @details
 * BLOCKING
 * Begins the scan process and state machine. The radio
 * configuration should not be modified during this time.
 *
 * @callback EventFromLeScanStateChangedCallback
 * @callback EventFromLeScanCompleteCallback
 * @callback EventFromDeviceFoundCallback
 *
 * @retval ResultSuccess      If okay.
 * @retval ResultBadParameter Invalid passed.
 * @retval ResultClientError  bsa_send_receive_message failure.
 *
 */
nn::Result StartLeScan() NN_NOEXCEPT;
//bt_status_t (*scan)( bool start ); Android API example

/**
 * @brief  Stop BLE Scan.
 *
 * @details
 * BLOCKING
 * Aborts scan.
 *
 * @callback EventFromLeScanCompleteCallback
 *
 * @retval ResultSuccess.
 *
 */
nn::Result StopLeScan() NN_NOEXCEPT;

/**
 * @brief  Set BLE Scan parameters.
 *
 * @details
 * BLOCKING
 * Set BLE scan interval and window.
 * Interval must be less than or equal to window.
 *
 * @retval ResultSuccess        If okay.
 * @retval ResultBadParameter   Invalid passed.
 */
nn::Result SetLeScanParameter(uint16_t scanInterval, uint16_t scanWindow) NN_NOEXCEPT;

/**
 * @brief  Add BLE Scan filter condition.
 *
 * @details
 * BLOCKING
 * Add BLE Scan filter condition. Currently all the conditions are ORed.
 *
 * @callback EventFromLeScanStateChangedCallback
 *
 * @retval ResultSuccess.
 *
 */
nn::Result AddLeScanFilterCondition(const BleAdvertiseFilter* pFilter) NN_NOEXCEPT;

/**
 * @brief  Delete BLE Scan filter condition.
 *
 * @details
 * BLOCKING
 * Delete BLE Scan filter condition.
 *
 * @callback EventFromLeScanStateChangedCallback
 *
 * @retval ResultSuccess.
 *
 */
nn::Result DeleteLeScanFilterCondition(const BleAdvertiseFilter* pFilter) NN_NOEXCEPT;

/**
 * @brief  Delete a specific BLE Scan filter.
 *
 * @details
 * BLOCKING
 * Delete a specific BLE Scan filter
 *
 * @callback EventFromLeScanStateChangedCallback
 *
 * @retval ResultSuccess.
 *
 */
nn::Result DeleteLeScanFilter(uint8_t filterIndex) NN_NOEXCEPT;

/**
 * @brief  Clear all BLE Scan filters.
 *
 * @details
 * BLOCKING
 * Delete all BLE Scan filters.
 *
 * @callback EventFromLeScanStateChangedCallback
 *
 * @retval ResultSuccess.
 *
 */
nn::Result ClearLeScanFilters() NN_NOEXCEPT;

/**
 * @brief  Enable BLE Scan filter
 *
 * @details
 * BLOCKING
 * Enable all BLE Scan filters
 *
 * @callback EventFromLeScanStateChangedCallback
 *
 * @retval ResultSuccess.
 *
 */
nn::Result EnableLeScanFilter(bool enable) NN_NOEXCEPT;

/*************************************************************************
* Low Energy Client APIs
************************************************************************/
/**
 * @brief  Register BLE GATT client.
 *
 * @details
 * BLOCKING
 *
 * @callback EventFromRegisterLeClientCallback
 *
 * @retval ResultSuccess.
 *
 */
nn::Result RegisterLeClient(GattAttributeUuid uuid) NN_NOEXCEPT;

/**
 * @brief  Deregister BLE GATT client.
 *
 * @details
 * BLOCKING
 *
 * @callback None
 *
 * @retval ResultSuccess.
 *
 */
nn::Result UnregisterLeClient(uint8_t clientIf) NN_NOEXCEPT;

/**
 * @brief  Deregister all BLE GATT clients.
 *
 * @details
 * BLOCKING
 *
 * @callback None
 *
 * @retval ResultSuccess.
 *
 */
nn::Result UnregisterLeClientAll() NN_NOEXCEPT;

/**
 * @brief  Connect to a BLE GATT server.
 *
 * @details
 * BLOCKING
 *
 * @callback EventFromClConnectCallback
 *
 * @retval ResultSuccess.
 *
 */
nn::Result LeClientConnect(const nn::applet::AppletResourceUserId& aruid, uint8_t clientIf, const BluetoothAddress *pBluetoothAddress, bool isDirect) NN_NOEXCEPT;
//bt_status_t (*connect)( int client_if, const bt_bdaddr_t *bd_addr, bool is_direct, int transport ); Android API example

/**
* @brief  Cancel connection request to a BLE GATT server.
*
* @details
* BLOCKING
*
* @callback EventFromClConnectCallback
*
* @retval ResultSuccess.
*
*/
nn::Result LeClientCancelConnection(uint8_t clientIf, const BluetoothAddress *pBluetoothAddress, bool isDirect) NN_NOEXCEPT;

/**
 * @brief  Disconnect from a BLE GATT server.
 *
 * @details
 * BLOCKING
 *
 * @callback EventFromClDisconnectCallback
 *
 * @retval ResultSuccess.
 *
 */
nn::Result LeClientDisconnect(uint32_t connId) NN_NOEXCEPT;
//bt_status_t (*disconnect)( int client_if, const bt_bdaddr_t *bd_addr, int conn_id); Android API example

/**
 * @brief  Get the attributes of a device.
 *
 * @details
 * BLOCKING
 *
 * @callback
 *
 * @retval ResultSuccess.
 *
 */
nn::Result LeClientGetAttributes(uint32_t connId, const BluetoothAddress *pBluetoothAddress) NN_NOEXCEPT;

/**
 * @brief  Discover BLE GATT service.
 *
 * @details
 * BLOCKING
 * Discover BLE GATT server service.
 *
 * @callback EventFromLeDiscoverServiceCallback
 *
 * @retval ResultSuccess.
 *
 */
nn::Result LeClientDiscoverService(uint32_t connId, GattAttributeUuid filterUuid) NN_NOEXCEPT;
//bt_status_t (*search_service)(int conn_id, bt_uuid_t *filter_uuid ); Android API example


/**
 * @brief  Configure MTU to operate larger data
 *
 * @details
 * BLOCKING
 *
 * @callback EventFromLeClConfigureMtuCallback
 *
 * @retval ResultSuccess.
 *
 */
nn::Result LeClientConfigureMtu(uint32_t connId, uint16_t mtu);

/*************************************************************************
* Low Energy Server APIs
************************************************************************/
/**
 * @brief  Register BLE GATT server.
 *
 * @details
 * BLOCKING
 *
 * @callback EventFromRegisterLeServerCallback
 *
 * @retval ResultSuccess.
 *
 */
nn::Result RegisterLeServer(GattAttributeUuid uuid) NN_NOEXCEPT;
// bt_status_t (*register_server)( bt_uuid_t *uuid ); Android API example

/**
 * @brief  Deregister BLE GATT server.
 *
 * @details
 * BLOCKING
 *
 * @callback None
 *
 * @retval ResultSuccess.
 *
 */
nn::Result UnregisterLeServer(uint8_t serverIf) NN_NOEXCEPT;
// bt_status_t (*unregister_server)(int server_if ); Android API example

/**
 * @brief  Deregister all BLE GATT servers.
 *
 * @details
 * BLOCKING
 *
 * @callback None
 *
 * @retval ResultSuccess.
 *
 */
nn::Result UnregisterAllLeServers() NN_NOEXCEPT;

/**
 * @brief  Initialize a BLE GATT server service.
 *
 * @details
 * BLOCKING
 * This function is called to reserve a block of handles for a service.
 * It should be called only once per service instance.
 *
 * @callback EventFromCreateLeServiceCallback
 *
 * @retval ResultSuccess.
 *
 */
nn::Result CreateLeService(uint8_t serverIf, GattAttributeUuid uuid, uint8_t handleNum, bool isPrimaryService) NN_NOEXCEPT;
//bt_status_t (*add_service)( int server_if, btgatt_srvc_id_t *srvc_id, int num_handles); Android API example

/**
 * @brief  Initialize a BLE GATT server service.
 *
 * @details
 * BLOCKING
 * This function is called to add an included service.
 *
 * @callback EventFromAddLeIncludedServiceCallback
 *
 * @retval ResultSuccess.
 *
 */
nn::Result AddLeIncludedService(uint8_t serverIf, GattAttributeUuid uuid, uint8_t handles) NN_NOEXCEPT;
//bt_status_t (*add_included_service)( int server_if, int service_handle, int included_handle); Android API example

/**
 * @brief  Start a local BLE GATT server service.
 *
 * @details
 * BLOCKING
 * This function is called to start a service with GATT.
 *
 * @callback EventFromStartLeServiceCallback
 *
 * @retval ResultSuccess.
 *
 */
nn::Result StartLeService(uint8_t serverIf, GattAttributeUuid uuid) NN_NOEXCEPT;
//bt_status_t (*start_service)(int server_if, int service_handle, int transport); Android API example

/**
 * @brief  Stop a local BLE GATT server service.
 *
 * @details
 * BLOCKING
 * This function is called to stop a service.
 *
 * @callback EventFromStopLeServiceCallback
 *
 * @retval ResultSuccess.
 *
 */
nn::Result StopLeService(uint8_t serverIf, GattAttributeUuid uuid) NN_NOEXCEPT;
//bt_status_t (*stop_service)(int server_if, int service_handle);  Android API example

/**
 * @brief  Add a BLE GATT characteristic to a service.
 *
 * @details
 * BLOCKING
 * This function is called to add a characteristic into a service.
 * It will add a characteristic declaration and characteristic
 * value declaration into the service database identified by the
 * service handle.
 *
 * @callback EventFromAddLeCharacteristicCallback
 *
 * @retval ResultSuccess.
 *
 */
nn::Result AddLeCharacteristic(uint8_t serverIf, GattAttributeUuid serviceUuid, GattAttributeUuid charUuid, uint16_t permissions, uint8_t properties) NN_NOEXCEPT;
//bt_status_t (*add_characteristic)( int server_if, int service_handle, bt_uuid_t *uuid, int properties, int permissions); Android API example

/**
 * @brief  Add a BLE GATT characteristic to a service.
 *
 * @details
 * BLOCKING
 * This function is called to add a characteristic descriptor
 * into a service database. Add descriptor should follow add char
 * to which it belongs, and next add char should be done only
 * after all add descriptors for the previous char.
 *
 * @callback EventFromAddLeDescriptorCallback
 *
 * @retval ResultSuccess.
 *
 */
nn::Result AddLeDescriptor(uint8_t serverIf, GattAttributeUuid serviceUuid, GattAttributeUuid charUuid, uint16_t permissions) NN_NOEXCEPT;
//bt_status_t (*add_descriptor)(int server_if, int service_handle, bt_uuid_t *uuid, int permissions); Android API example

/**
 * @brief  Connect to a remote peripheral.
 *
 * @details
 * BLOCKING
 *
 * @callback EventFromSeConnectCallback
 *
 * @retval ResultSuccess.
 *
 */
nn::Result LeServerConnect(uint8_t serverIf, const BluetoothAddress *pBluetoothAddress, bool isDirect) NN_NOEXCEPT;
//bt_status_t (*connect)(int server_if, const bt_bdaddr_t *bd_addr, bool is_direct, int transport);  Android API example

/**
 * @brief  Disconnect from a remote device or cancel pending connection.
 *
 * @details
 * BLOCKING
 *
 * @callback EventFromSeDisconnectCallback
 *
 * @retval ResultSuccess.
 *
 */
nn::Result LeServerDisconnect(uint8_t serverIf, const BluetoothAddress *pBluetoothAddress) NN_NOEXCEPT;
//bt_status_t (*disconnect)(int server_if, const bt_bdaddr_t *bd_addr, int conn_id ); Android API example

/**
 * @brief  Get the information regarding BLE core event info from the Server.
 *
 * @details
 * BLOCKING
 *
 * @param[out] pOutEventType Event type.
 * @param[out] outBuffer     Event Data.
 * @param[in]  sizeOfBuffer  Size of event data.
 *
 * @retval ResultSuccess.
 *
 */
nn::Result GetLeCoreEventInfo(nn::bluetooth::BleEventType *pOutEventType, uint8_t outBuffer[], uint16_t sizeOfBuffer) NN_NOEXCEPT;

/**
 * @brief  Get the first characteristic from a service.
 *
 * @details
 * BLOCKING
 *
 *
 * @retval ResultSuccess.
 *
 */
nn::Result LeGetFirstCharacteristic(GattId *charOut, uint8_t *property, uint32_t connId, GattId serviceId, bool isPrimary, GattAttributeUuid charFilter) NN_NOEXCEPT;

/**
 * @brief  Get the subsequent characteristic from a service.
 *
 * @details
 * BLOCKING
 *
 *
 * @retval ResultSuccess.
 *
 */
nn::Result LeGetNextCharacteristic(GattId *charOut, uint8_t *property, uint32_t connId, GattId serviceId, bool isPrimary, GattId charIn, GattAttributeUuid charFilter) NN_NOEXCEPT;

/**
 * @brief  Get the first descriptor from a service.
 *
 * @details
 * BLOCKING
 *
 *
 * @retval ResultSuccess.
 *
 */
nn::Result LeGetFirstDescriptor(GattId *descOut, uint32_t connId, GattId serviceId, bool isPrimary, GattId charIn, GattAttributeUuid descrFilter) NN_NOEXCEPT;

/**
 * @brief  Get the subsequent descriptor from a service.
 *
 * @details
 * BLOCKING
 *
 *
 * @retval ResultSuccess.
 *
 */
nn::Result LeGetNextDescriptor(GattId *descOut, uint32_t connId, GattId serviceId, bool isPrimary, GattId charIn, GattId descrIn, GattAttributeUuid descrFilter) NN_NOEXCEPT;

/**
 * @brief  Register GATT service UUID to be informed via BLE Core System Event
 *
 * @details
 * BLOCKING
 *
 * @param[in]    uuid        GATT Service UUID
 *
 * @retval ResultSuccess.
 * @retval ResultFail    if the database is full
 *
 */
nn::Result RegisterLeCoreDataPath(const nn::bluetooth::GattAttributeUuid& uuid) NN_NOEXCEPT;

/**
 * @brief  Unregister GATT service UUID to be informed via BLE Core System Event
 *
 * @details
 * BLOCKING
 *
 * @param[in]    uuid        GATT Service UUID
 *
 * @retval ResultSuccess.
 * @retval ResultFail    if the path is not registered
 *
 */
nn::Result UnregisterLeCoreDataPath(const nn::bluetooth::GattAttributeUuid& uuid) NN_NOEXCEPT;

/**
 * @brief  Register GATT service UUID to be informed via BLE HID System Event
 *
 * @details
 * BLOCKING
 *
 * @param[in]    uuid        GATT Service UUID
 *
 * @retval ResultSuccess.
 * @retval ResultFail    if the database is full
 *
 */
nn::Result RegisterLeHidDataPath(const nn::bluetooth::GattAttributeUuid& uuid) NN_NOEXCEPT;

/**
 * @brief  Unregister GATT service UUID to be informed via BLE HID System Event
 *
 * @details
 * BLOCKING
 *
 * @param[in]    uuid        GATT Service UUID
 *
 * @retval ResultSuccess.
 * @retval ResultFail    if the path is not registered
 *
 */
nn::Result UnregisterLeHidDataPath(const nn::bluetooth::GattAttributeUuid& uuid) NN_NOEXCEPT;

/**
 * @brief  Register GATT service UUID to be informed via BLE System Event
 *
 * @details
 * BLOCKING
 *
 * @param[in]    uuid        GATT Service UUID
 *
 * @retval ResultSuccess.
 * @retval ResultFail    if the database is full
 *
 */
nn::Result RegisterLeDataPath(const nn::bluetooth::GattAttributeUuid& uuid) NN_NOEXCEPT;

/**
 * @brief  Unregister GATT service UUID to be informed via BLE System Event
 *
 * @details
 * BLOCKING
 *
 * @param[in]    uuid        GATT Service UUID
 *
 * @retval ResultSuccess.
 * @retval ResultFail    if the path is not registered
 *
 */
nn::Result UnregisterLeDataPath(const nn::bluetooth::GattAttributeUuid& uuid) NN_NOEXCEPT;

/**
 * @brief  Register BLE Hid events.
 *
 * @details
 * BLOCKING.
 * Attach a system event for notification of BLE HID GATT operation.
 *
 * @param[in] pSystemEvent Hid event to be registered.
 *
 * @retval ResultSuccess if okay, otherwise ResultFail.
 *
 */
nn::Result RegisterBleHidEvent(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT;

/**
 * @brief  Get the information regarding BLE HID event info from the Server.
 *
 * @details
 * BLOCKING
 *
 * @param[out] pOutEventType Event type.
 * @param[out] outBuffer     Event Data.
 * @param[in]  sizeOfBuffer  Size of event data.
 *
 * @retval ResultSuccess.
 *
 */
nn::Result GetLeHidEventInfo(nn::bluetooth::BleEventType *pOutEventType, uint8_t outBuffer[], uint16_t sizeOfBuffer) NN_NOEXCEPT;

/**
 * @brief  Read a characteristic
 *
 * @details
 * BLOCKING
 * Read a characteristic on a remote device
 *
 * @callback EventFromLeReadCharacteristicCallback
 *
 * @retval ResultSuccess.
 *
 */
nn::Result LeClientReadCharacteristic(uint32_t connId, GattId serviceId, bool isPrimary, GattId charId, uint8_t authType) NN_NOEXCEPT;
//bt_status_t (*read_characteristic)( int conn_id, btgatt_srvc_id_t *srvc_id, btgatt_gatt_id_t *char_id, int auth_req ); Android API example

/**
 * @brief  Read a descriptor
 *
 * @details
 * BLOCKING
 * Read the descriptor for a given characteristic
 *
 * @callback EventFromLeReadDescriptorCallback
 *
 * @retval ResultSuccess.
 *
 */
nn::Result LeClientReadDescriptor(uint32_t connId, GattId serviceId, bool isPrimary, GattId charId, GattId descrId, uint8_t authType) NN_NOEXCEPT;
//bt_status_t (*read_descriptor)( int conn_id, btgatt_srvc_id_t *srvc_id, btgatt_gatt_id_t *char_id, btgatt_gatt_id_t *descr_id, int auth_req); Android API example

/**
 * @brief  Write a characteristic
 *
 * @details
 * BLOCKING
 * Write a characteristic on a remote device
 *
 * @callback EventFromLeWriteCharacteristicCallback
 *
 * @retval ResultSuccess.
 *
 */
nn::Result LeClientWriteCharacteristic(uint32_t connId, GattId serviceId, bool isPrimary, GattId charId, const uint8_t* pData, uint16_t len, uint8_t authType, bool withResponse) NN_NOEXCEPT;
//bt_status_t (*write_characteristic)(int conn_id, btgatt_srvc_id_t *srvc_id, btgatt_gatt_id_t *char_id, int write_type, int len, int auth_req, char* p_value); Android API example

/**
 * @brief  Write a descriptor
 *
 * @details
 * BLOCKING
 * Write the descriptor for a given characteristic
 *
 * @callback EventFromLeWriteDescriptorCallback
 *
 * @retval ResultSuccess.
 *
 */
nn::Result LeClientWriteDescriptor(uint32_t connId, GattId serviceId, bool isPrimary, GattId charId, GattId descrId, const uint8_t* pData, uint16_t len, uint8_t authType) NN_NOEXCEPT;
//bt_status_t (*read_descriptor)( int conn_id, btgatt_srvc_id_t *srvc_id, btgatt_gatt_id_t *char_id, btgatt_gatt_id_t *descr_id, int auth_req); Android API example

/**
 * @brief  Register a notification
 *
 * @details
 * BLOCKING
 * Register to receive notifications or indications for a given
 * characteristic from a service
 *
 * @callback EventFromLeRegisterNotificationCallback
 * @callback EventFromLeNotificationCallback
 *
 * @retval ResultSuccess.
 *
 */
nn::Result LeClientRegisterNotification(uint32_t connId, GattId serviceId, bool isPrimary, GattId charId) NN_NOEXCEPT;
//bt_status_t (*register_for_notification)( int client_if, const bt_bdaddr_t *bd_addr, btgatt_srvc_id_t *srvc_id, btgatt_gatt_id_t *char_id); Android API example

/**
 * @brief  Deregister a notification
 *
 * @details
 * BLOCKING
 * Deregister a previous request for notifications/indications
 *
 * @callback EventFromLeDeregisterNotificationCallback
 *
 * @retval ResultSuccess.
 *
 */
nn::Result LeClientDeregisterNotification(uint32_t connId, GattId serviceId, bool isPrimary, GattId charId) NN_NOEXCEPT;

} // namespace bluetooth
} // namespace nn
