﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>
#include <nn/bgtc/bgtc_Types.h>
#include <nn/os.h>

namespace nn
{
namespace bgtc
{
    ///////////////////////////////////////////////////////////////////////////
    //! @name 初期化・終了 API
    //! @{

    /**
     * @brief   ライブラリの初期化をおこない、機能を使用可能な状態にします。
     *
     * @return  処理の結果が返ります。
     *
     * @details
     *  ライブラリが提供する API は、個別に明示されていないかぎり、利用前にこの関数でライブラリを初期化しておく必要があります。@n
     *
     *  ライブラリがすでに初期化された状態でも重ねて呼ぶことができますが、その場合、ライブラリの利用を完全に終了するにはこの関数を呼んだのと同じ回数だけ
     *  @ref nn::bgtc::FinalizeStateControl 関数を呼ぶ必要があります。
     */
    Result InitializeStateControl() NN_NOEXCEPT;

    /**
     * @brief   ライブラリの利用を終了します。
     *
     * @return  処理の結果が返ります。
     *
     * @details
     *  @ref nn::bgtc::InitializeStateControl 関数を複数回呼んでいる場合、ライブラリの利用を完全に終了するには同じ回数だけこの関数を呼ぶ必要があります。@n
     *
     *  ライブラリの利用が完全に終了されると、システムで確保していたリソースが解放されます。@n
     *
     *  ライブラリが未初期化の状態でコールしても副作用はありません。
     */
    void FinalizeStateControl() NN_NOEXCEPT;

    //! @}

    ///////////////////////////////////////////////////////////////////////////
    //! @name 状態制御 API
    //! @{

    /**
     * @brief   BGTC の状態を取得します。
     *
     * @return  BGTC の状態。
     *
     * @details BGTC の状態が State_Idle である場合、バックグラウンド処理を行うべきモジュールがいないことを表します。
     *          これをもって上位モジュールは電源状態をバックグラウンド処理不可能な状態に遷移してもよいことを判断できます。
     */

    State GetState() NN_NOEXCEPT;

    /**
     * @brief   半起床状態を開始できることを BGTC に通知します。
     *
     * @return  なし。
     *
     * @details スリープ中にバックグラウンド処理を行うための状態（BGTC では半起床状態と呼びます）を開始できることを通知します。
     *          通知されると BGTC は実際にバックグラウンド処理を行うモジュールと連携し、必要な処理が行われるように調整を開始します。
     *          呼び出し直後から一定時間は必ず GetState() は State_Idle 以外を返します。
     */
    void NotifyEnteringHalfAwake() NN_NOEXCEPT;

    /**
     * @brief   半起床状態を終了しなければいけないことを BGTC に通知します。
     *
     * @return  なし。
     *
     * @details TBD: BGTC が半起床状態の終了を契機に行う処理は実際にはありません。
     *          BGTC を含む各モジュールは電源状態に応じたハンドリングをもともと行っているため、スリープ前に必要な処理があればそのタイミングで行われます。
     *
     *          NOTE: バックグランド処理を行うモジュールは PSC を使わずに済むように BGTC へ機能を追加するかは今後の検討課題です。
     *                スリープ/シャットダウン前に行う処理である都合上 ACK を待つ必要がありそうですが、その分時間がかかってしまいますし、そもそも各モジュールがそのような処理を行う必要があるのか、の検討が必要です。
     */
    void NotifyLeavingHalfAwake() NN_NOEXCEPT;

    /**
     * @brief   BGTC の状態が変わったことを知るためイベントを取得します。
     *
     * @return  BGTC の状態が変わったことを知るためイベント。
     *
     * @details GetState() と組み合わせて使用します。
     */
    nn::os::SystemEvent& GetStateChangedEvent() NN_NOEXCEPT;

    /**
     * @brief   スリープによって状態が保持できなくなるデバイスを利用中であるかを BGTC に通知します。
     *
     * @return  なし。
     *
     * @details スリープによって状態が保持できなくなるデバイス(USB-Ether など)を利用中の場合、bgtc は十分な電源が供給されている状況下ではスリープに入らずに半起床の状態を維持しようとします。
     */
    void SetIsUsingSleepUnsupportedDevices(bool bIsUsing) NN_NOEXCEPT;

    //! @}
}
}
