﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>
#include <nn/bgtc/bgtc_Types.h>
#include <nn/os.h>

namespace nn
{
namespace bgtc
{
    ///////////////////////////////////////////////////////////////////////////
    //! @name 初期化・終了 API
    //! @{

    /**
     * @brief   ライブラリの初期化をおこない、機能を使用可能な状態にします。
     *
     * @return  処理の結果が返ります。
     *
     * @details
     *  ライブラリが提供する API は、個別に明示されていないかぎり、利用前にこの関数でライブラリを初期化しておく必要があります。@n
     *
     *  ライブラリがすでに初期化された状態でも重ねて呼ぶことができますが、その場合、ライブラリの利用を完全に終了するにはこの関数を呼んだのと同じ回数だけ
     *  @ref nn::bgtc::Finalize 関数を呼ぶ必要があります。
     */
    Result Initialize() NN_NOEXCEPT;

    /**
     * @brief   ライブラリが初期化済みかを判定します。
     *
     * @return  ライブラリがすでに初期化されていれば true を、まだ初期化されていなければ false を返します。
     *
     * @details
     */
    bool IsInitialized() NN_NOEXCEPT;

    /**
     * @brief   ライブラリの利用を終了します。
     *
     * @return  処理の結果が返ります。
     *
     * @details
     *  @ref nn::bgtc::Initialize 関数を複数回呼んでいる場合、ライブラリの利用を完全に終了するには同じ回数だけこの関数を呼ぶ必要があります。@n
     *
     *  ライブラリの利用が完全に終了されると、システムで確保していたリソースが解放されます。@n
     *
     *  ライブラリが未初期化の状態でコールしても副作用はありません。
     */
    void Finalize() NN_NOEXCEPT;

    //! @}

    ///////////////////////////////////////////////////////////////////////////
    //! @name タスク管理 API
    //! @{

    /**
     * @brief   タスクの処理を開始したことをコーディネータに通知します。
     *
     * @return  処理の結果が返ります。
     * @retval  ResultSuccess                   成功
     * @retval  ResultNotPermitted              タスク処理が許可されていません。システム電源状態が FullAwake もしくは MinimumAwakeForBackgroundTask ではありません。
     *
     * @details MinimumAwakeForBackgroundTask で呼び出しておくと、コーディネータは MinimumAwakeForBackgroundTask に滞留するようにシステムに働きかけます。
     *          ただし、本関数を呼び出しに成功しても、バッテリー残量低下やユーザ操作等の外部要因により、タスク処理を続行できない電源状態に移行する可能性が常にあることに注意してください。
     *          関数呼び出し直前まで成功する条件が整っていたとしても、入れ違いで失敗する可能性があるため、失敗で ABORT するようなハンドリングは避けてください。
     *          @ref NotifyTaskFinished によりタスクの完了通知しないままセルフリフレッシュモードに遷移した場合、セルフリフレッシュモードから復帰した後に改めて本関数を呼び出す必要があります。
     *
     *          FullAwake 時に呼び出しておくと、セルフリフレッシュモードに遷移した一定時間後に MinimumAwakeForBackgroundTask に一度だけ遷移するように動作します。
     *          この動作は @ref ScheduleTask による起動予約とは独立しているため @ref GetScheduleEvent はシグナルされません。
     */
    Result NotifyTaskStarting() NN_NOEXCEPT;

    /**
     * @brief   タスクの処理が完了したことをコーディネータに通知します。
     *
     * @return  なし。
     *
     * @details 処理中のタスクが一つも存在しなくなると、コーディネータは一定期間待った後にシステム電源状態の維持を解除します。
     *          @ref NotifyTaskStarting を呼び出した後は必ずこの関数を呼び出すようにしてください。呼び出しを忘れるといつまでたっても電源状態がセルフリフレッシュモードに戻らないため、本体がスリープ中の様に振る舞っていても電池を早く消耗してしまうようになってしまいます。
     *
     */
    void NotifyTaskFinished() NN_NOEXCEPT;

    /**
     * @brief   タスクの処理を開始可能になったタイミングを知るための自動クリアイベントを取得します。
     *
     * @return  なし。
     *
     * @details 取得されたイベントはシステム電源状態が FullAwake, MinimumAwakeForBackgroundTask になったタイミングでシグナルされます。
     *          FullAwake に遷移したタイミングは PSC 経由でも知ることができますが、MinimumAwakeForBackgroundTask に遷移したタイミングはこのイベントでのみ知ることができます。
     *
     *          一般的な処理フローでは、このイベントのシグナルを契機として自律動作を開始し、@ref NotifyTaskStarting を呼び出すことになります。
     */
    nn::os::SystemEvent& GetTriggerEvent() NN_NOEXCEPT;

    /**
     * @brief   システム電源状態が MinimumAwakeForBackgroundTask であるかどうかを判定します
     *
     * @return  判定結果。
     *
     * @details
     */
    bool IsInHalfAwake();

    /**
     * @brief   bgtc が認識しているシステム電源状態が FullAwake であるかどうかを判定します。
     *
     * @return  判定結果。
     *
     * @details タイミングによっては実際のシステム電源状態と異なる場合があります。
     */
    bool IsInFullAwake();

    /**
     * @brief   タスクの処理を行う予定時刻をコーディネータに通知し、設定された時刻にタスク処理を行えるようシステム電源状態の遷移を予約します。
     *
     * @return  なし。
     *
     * @details システム電源状態がセルフリフレッシュモードになり、タスクを処理できない状態になっていても、設定された時刻になるとコーディネータは電源状態を MinimumAwakeForBackgroundTask へと遷移できるようハードウェアを設定します。
     *          MinimumAwakeForBackgroundTask に遷移する要因は複数あるため、実際に予定した時刻になっているかどうかは @ref GetScheduleEvent で確認するようにしてください。
     *
     *          既に設定があった場合は上書きされます。
     *
     * @attention 本 API は BGTC がスリープ前処理を行う前に呼び出す必要があります。つまり、PSC の依存モジュールに BGTC を加えた上で、PSC からの SleepReady 通知に対して Acknowledge を返す前に呼び出さなければいけません。
     *          そのため、PSC のイベントを処理するスレッドでのみ呼び出すのが安全です。それ以外のスレッドから呼び出すことは控えてください。
     */
    void ScheduleTask(Interval intervalSeconds) NN_NOEXCEPT;
    void ScheduleTaskUnsafe(Interval intervalSeconds) NN_NOEXCEPT;

    /**
     * @brief   タスクの処理を行う予定時刻をコーディネータに通知し、設定された時刻にタスク処理を行えるようシステム電源状態の遷移を予約します。@ref ScheduleTask の周期版です。
     *
     * @return  なし。
     *
     * @details 利用方法や注意点は @ref ScheduleTask に準じます。
     *
     *          本APIによってスケジューリングされる実際のタスク間隔は、設定した周期よりも広がる可能性があります。
     *          具体的には、スケジュール実行予定時刻になってから、スケジュール実行条件となる電源状態を満たした時点から設定された周期に基づいて次のスケジュールが設定されます。
     *          つまり、@ref ScheduleTask を毎回手動で呼び出したときの挙動とほぼ同じであり、自動再設定を行っているとみなすこともできます。
     */
    void SchedulePeriodicTask(Interval intervalSecondsFirst, Interval intervalSecondsPeriodic) NN_NOEXCEPT;

    /**
     * @brief   @ref ScheduleTaskで設定した予定時刻まであと何秒かを取得します。
     *
     * @return  処理の結果が返ります。
     * @retval  ResultSuccess                   成功
     * @retval  ResultActiveAlarmNotExist       有効なアラーム設定がありません
     *
     * @details 既に予定時刻を経過していた場合は pOutSeconds に 0 が返ります。
     *
     */
    Result GetScheduledTaskInterval(Interval* pOutSeconds) NN_NOEXCEPT;

    /**
     * @brief   @ref ScheduleTaskで設定した予定時刻の設定を解除します。
     *
     * @return  処理の結果が返ります。
     * @retval  ResultSuccess                   成功
     * @retval  ResultActiveAlarmNotExist       有効なアラーム設定がありません
     *
     * @details @ref GetScheduleEvent で取得できるイベントのシグナル状態は解除されます。
     *
     */
    Result UnscheduleTask() NN_NOEXCEPT;

    /**
     * @brief   @ref ScheduleTaskで設定した予定時刻になるとシグナル状態になる自動リセットイベントを取得します。
     *
     * @return  イベント
     *
     * @details シグナル状態を解除するには @ref UnscheduleTask を呼び出してスケジュール設定を削除するか @ref ScheduleTask で新たなスケジュールを設定します。
     */
    nn::os::SystemEvent& GetScheduleEvent() NN_NOEXCEPT;

    //! @}

    ///////////////////////////////////////////////////////////////////////////
    //! @name 情報取得 API
    //! @{

    /**
        * @brief   電源状態に基づいた動作モードを取得します
        *
        * @return  動作モード
        *
        * @details bgtc は電池残量や充電状態に応じて起床間隔の調整を行います。この API はその動作のもととなる動作モードを返します。
        */
    OperationMode GetOperationMode() NN_NOEXCEPT;

    /**
        * @brief   スリープへの遷移によってネットワーク接続が失われるかを取得します
        *
        * @return  利用しているか
        *
        * @details スリープへの遷移によってバックグラウンド処理を継続できなくなるデバイスを利用中かどうかを取得します。
        *          true の場合は適切な終了処理が必要です。
        */
    bool WillDisconnectNetworkWhenEnteringSleep() NN_NOEXCEPT;

    /**
        * @brief   スリープ後に半起床に滞留する予定かを取得します
        *
        * @return  利用しているか
        *
        * @details スリープ中の動作をサポートしていないデバイスを利用してリアルタイムなバックグラウンド通信を行うために、
        *          半起床に滞留したままの疑似スリープ状態になる予定かを取得します。
        *          スリープ中の動作をサポートしていないデバイスを利用していても、給電状態でない場合は通常のスリープ動作を行います。
        */
    bool WillStayHalfAwakeInsteadSleep() NN_NOEXCEPT;

    //! @}

    ///////////////////////////////////////////////////////////////////////////
    //! @name 情報設定 API
    //! @{

    /**
        * @brief   タスクのクライアント名を設定します。
        *
        * @return  なし。
        *
        * @details
        */
    void SetTaskClientName(const char* pName) NN_NOEXCEPT;

    //! @}
}
}
