﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Result.h>
#include <nn/sdmmc/sdmmc_Common.h>
#include <nn/bconfig/bconfig_Types.h>

namespace nn { namespace bconfig {

/**
 * @brief BootConfig の非署名領域を読み込みます
 *
 * @param[out] pOutBootConfig   BootConfig を保持する構造体へのポインタ
 *
 * @details
 *  BootConfig の非署名領域を読み込み pOutBootConfig に格納します。
 *
 */
void LoadBootConfig(BootConfig* pOutBootConfig) NN_NOEXCEPT;

/**
 * @brief BootConfig の非署名領域に書き込みます
 *
 * @param[in] bootConfig   BootConfig を保持する構造体へのポインタ
 *
 * @details
 *  bootConfig が保持する情報を BootConfig の非署名領域に書き込みます。
 *
 */
void SaveBootConfig(const BootConfig* bootConfig) NN_NOEXCEPT;


/**
 * @brief メモリモード値を設定します
 *
 * @param[in] bootConfig   BootConfig を保持する構造体へのポインタ
 * @param[in] memoryMode   設定するメモリモード値
 */
void SetMemoryMode(BootConfig* bootConfig, Bit8 memoryMode) NN_NOEXCEPT;

/**
 * @brief メモリモード値を取得します
 *
 * @param[in] bootConfig   BootConfig を保持する構造体へのポインタ
 *
 * @return メモリモード値
 */
Bit8 GetMemoryMode(const BootConfig* bootConfig) NN_NOEXCEPT;

/**
 * @brief 非ゼロ値でメモリを埋めるかどうかを設定します
 *
 * @param[in] bootConfig   BootConfig を保持する構造体へのポインタ
 * @param[in] enabled      false なら無効、true なら有効
 */
void SetEnableNonZeroFillMemory(BootConfig* bootConfig, bool enabled) NN_NOEXCEPT;

/**
 * @brief 非ゼロ値でメモリを埋めるかどうかを取得します
 *
 * @param[in] bootConfig   BootConfig を保持する構造体へのポインタ
 *
 * @return false なら無効、true なら有効
 */
bool GetEnableNonZeroFillMemory(const BootConfig* bootConfig) NN_NOEXCEPT;

/**
 * @brief ユーザ例外ハンドラの有効・無効を設定します
 *
 * @param[in] bootConfig   BootConfig を保持する構造体へのポインタ
 * @param[in] enabled      false なら無効、true なら有効
 */
void SetEnableUserExceptionHandler(BootConfig* bootConfig, bool enabled) NN_NOEXCEPT;

/**
 * @brief ユーザ例外ハンドラの有効・無効を取得します
 *
 * @param[in] bootConfig   BootConfig を保持する構造体へのポインタ
 *
 * @return false なら無効、true なら有効
 */
bool GetEnableUserExceptionHandler(const BootConfig* bootConfig) NN_NOEXCEPT;


/**
 * @brief PMU の有効・無効を設定します
 *
 * @param[in] bootConfig   BootConfig を保持する構造体へのポインタ
 * @param[in] enabled      false なら無効、true なら有効
 */
void SetPerformanceMonitoringUnit(BootConfig* bootConfig, bool enabled) NN_NOEXCEPT;

/**
 * @brief PMU の有効・無効を取得します
 *
 * @param[in] bootConfig   BootConfig を保持する構造体へのポインタ
 *
 * @return false なら無効、true なら有効
 */
bool GetPerformanceMonitoringUnit(const BootConfig* bootConfig) NN_NOEXCEPT;

/**
 * @brief システムエラー時にエラー画面を表示するかどうかを設定します
 *
 * @param[in] bootConfig   BootConfig を保持する構造体へのポインタ
 * @param[in] enabled      false なら無効、true なら有効
 */
void SetCallShowErrorOnPanic(BootConfig* bootConfig, bool enabled) NN_NOEXCEPT;

/**
 * @brief システムエラー時にエラー画面を表示するかどうかを取得します
 *
 * @param[in] bootConfig   BootConfig を保持する構造体へのポインタ
 *
 * @return false なら無効、true なら有効
 */
bool GetCallShowErrorOnPanic(const BootConfig* bootConfig) NN_NOEXCEPT;

/**
 * @brief TSC (Tick) の初期値を設定します
 *
 * @param[in] bootConfig   BootConfig を保持する構造体へのポインタ
 * @param[in] value        TSC の初期値
 *
 * @detail
 *  BootConfig に TSC の初期値を書き込み、「TSC の初期値有効」フラグを立てます。
 *  ただし、value が 0 の場合、「TSC の初期値有効」フラグを落とします。
 */
void SetInitialTscValue(BootConfig* bootConfig, Bit64 value) NN_NOEXCEPT;

/**
 * @brief TSC (Tick) の初期値を取得します
 *
 * @param[in] bootConfig   BootConfig を保持する構造体へのポインタ
 *
 * @return TSC の初期値
 *
 * @detail
 *  BootConfig の「TSC の初期値有効」フラグが立っている場合、
 *  BootConfig に書かれた TSC の初期値を返します。
 *  フラグが立っていない場合 0 が返ります。
 */
Bit64 GetInitialTscValue(const BootConfig* bootConfig) NN_NOEXCEPT;


}} // namespace nn::bconfig
