﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/bcat/bcat_Types.h>

namespace nn { namespace bcat { namespace service {

/*!
    @brief      BCAT アーカイブの復号鍵サイズの最大値です。
*/
static const size_t ArchiveKeySizeMax = 256 / 8;

/*!
    @brief      BCAT アーカイブのブロックサイズです。
*/
static const size_t ArchiveBlockSize = 16;

/*!
    @brief      BCAT アーカイブの署名サイズです。
*/
static const size_t ArchiveSignatureSize = 256;

/*!
    @brief      プッシュ通知受信コールバックの最大登録数です。
*/
static const int NotificationReceiveCallbackRegistrationCountMax = 4;

/*!
    @brief      暗号種別です。
*/
enum EncryptionType : Bit8
{
    EncryptionType_NoEncrypted = 0, //!< 暗号化されていません。
    EncryptionType_AesCtr128   = 1, //!< AES-CTR（鍵長 128 ビット）で暗号化されています。
    EncryptionType_AesCtr192   = 2, //!< AES-CTR（鍵長 192 ビット）で暗号化されています。
    EncryptionType_AesCtr256   = 3  //!< AES-CTR（鍵長 256 ビット）で暗号化されています。
};

/*!
    @brief      署名種別です。
*/
enum SignatureType : Bit8
{
    SignatureType_RsaPkcs1Sha1   = 0, //!< RSA-PKCS1-SHA1 による署名です。
    SignatureType_RsaPkcs1Sha256 = 1, //!< RSA-PKCS1-SHA256 による署名です。
    SignatureType_RsaPssSha1     = 2, //!< RSA-PSS-SHA1 による署名です。
    SignatureType_RsaPssSha256   = 3  //!< RSA-PSS-SHA256 による署名です。
};

/*!
    @brief      BCAT アーカイブのヘッダーです。
*/
struct ArchiveHeader
{
    Bit8 magic[4];                        //!< マジック。"bcat" 固定です。
    Bit8 version;                         //!< ヘッダーバージョン。
    EncryptionType encryptionType;        //!< 暗号種別。
    SignatureType signatureType;          //!< 署名種別。
    Bit8 saltIndex;                       //!< ソルトインデックス。
    Bit8 reserved[8];                     //!< 予約領域。
    Bit8 iv[ArchiveBlockSize];            //!< 初期ベクタ。
    Bit8 signature[ArchiveSignatureSize]; //!< 復号後のデータに対する署名。
};

/*!
    @brief      BCAT アーカイブの復号鍵です。
*/
struct ArchiveDecryptionKey
{
    Bit8 value[ArchiveKeySizeMax]; //!< 復号鍵。
};

/*!
    @brief      ETag です。
*/
struct ETag
{
    char value[40];
};

/*!
    @brief      BCAT アーカイブのメタデータです。
*/
struct ArchiveMetadata
{
    Bit8 magic[4];         //!< マジック。"meta" 固定です。
    Bit8 version;          //!< メタデータバージョン。
    Bit8 reserved[3];      //!< 予約領域。
    int64_t contentLength; //!< ダウンロードするデータの総サイズ。
    ETag eTag;             //!< ETag。
    ArchiveHeader header;  //!< ヘッダー。
};

}}}
