﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/bcat/service/bcat_TypesService.h>
#include <nn/nn_Result.h>
#include <nn/util/util_TypedStorage.h>

namespace nn { namespace bcat { namespace service {

/*!
    @brief      BCAT アーカイブの検証器です。
*/
class ArchiveVerifier
{
public:
    /*!
        @brief      コンストラクタです。
    */
    ArchiveVerifier() NN_NOEXCEPT;

    /*!
        @brief      デストラクタです。
    */
    ~ArchiveVerifier() NN_NOEXCEPT;

    /*!
        @brief      初期化します。

        @param[in]  signatureType   署名種別。

        @details
                    署名検証用の公開鍵は、BCAT システムが管理するデータ配信用の公開鍵を使用します。
    */
    void Initialize(SignatureType signatureType) NN_NOEXCEPT;

    /*!
        @brief      初期化します。

        @param[in]  signatureType       署名種別。
        @param[in]  pubkeyModulus       公開鍵の法データ。
        @param[in]  pubkeyModulusSize   公開鍵の法データのサイズ。
        @param[in]  pubkeyExponent      公開鍵の公開指数データ。
        @param[in]  pubkeyExponentSize  公開鍵の公開指数データのサイズ。

        @pre
            - pubkeyModulus != nullptr
            - pubkeyModulusSize > 0
            - pubkeyExponent != nullptr
            - pubkeyExponentSize > 0
    */
    void Initialize(SignatureType signatureType,
        const void* pubkeyModulus, size_t pubkeyModulusSize, const void* pubkeyExponent, size_t pubkeyExponentSize) NN_NOEXCEPT;

    /*!
        @brief      検証データを流し込み、検証器を更新します。

        @param[in]  buffer  検証データ。
        @param[in]  size    検証データのサイズ。

        @pre
            - buffer != nullptr
    */
    void Update(const void* buffer, size_t size) NN_NOEXCEPT;

    /*!
        @brief      署名を検証します。

        @param[in]  signature   署名。
        @param[in]  size        署名のサイズ。

        @return     検証結果。

        @pre
            - signature != nullptr
    */
    bool Verify(const void* signature, size_t size) NN_NOEXCEPT;

public:

    /*!
        @brief      ファイルの署名を検証します。

        @param[out] outResult   検証結果。
        @param[in]  path        ファイルパス。
        @param[in]  header      BCAT アーカイブのヘッダ。
        @param[in]  work        ファイル読み込み用のワークバッファ。
        @param[in]  workSize    ファイル読み込み用のワークバッファのサイズ。

        @return     処理結果。

        @pre
            - outResult != nullptr
            - path != nullptr
            - work != nullptr
            - workSize > 0

        @details
                    署名検証用の公開鍵は、BCAT システムが管理するデータ配信用の公開鍵を使用します。
    */
    static nn::Result VerifyFile(bool* outResult, const char* path,
        const ArchiveHeader& header, void* work, size_t workSize) NN_NOEXCEPT;


    /*!
        @brief      ファイルの署名を検証します。

        @param[out] outResult           検証結果。
        @param[in]  path                ファイルパス。
        @param[in]  header              BCAT アーカイブのヘッダ。
        @param[in]  work                ファイル読み込み用のワークバッファ。
        @param[in]  workSize            ファイル読み込み用のワークバッファのサイズ。
        @param[in]  pubkeyModulus       公開鍵の法データ。
        @param[in]  pubkeyModulusSize   公開鍵の法データのサイズ。
        @param[in]  pubkeyExponent      公開鍵の公開指数データ。
        @param[in]  pubkeyExponentSize  公開鍵の公開指数データのサイズ。

        @return     処理結果。

        @pre
            - outResult != nullptr
            - path != nullptr
            - work != nullptr
            - workSize > 0
            - pubkeyModulus != nullptr
            - pubkeyModulusSize > 0
            - pubkeyExponent != nullptr
            - pubkeyExponentSize > 0
    */
    static nn::Result VerifyFile(bool* outResult, const char* path,
        const ArchiveHeader& header, void* work, size_t workSize,
        const void* pubkeyModulus, size_t pubkeyModulusSize, const void* pubkeyExponent, size_t pubkeyExponentSize) NN_NOEXCEPT;

private:
    //
    using Destructor = void (*)(ArchiveVerifier* pThis);
    //
    using UpdateFunction = void (*)(ArchiveVerifier* pThis, const void* buffer, size_t size);
    using VerifyFunction = bool (*)(ArchiveVerifier* pThis, const void* signature, size_t size);

private:
    //
    nn::util::TypedStorage<Bit64, 480, sizeof (std::max_align_t)> m_VerifierStorage;
    //
    Destructor m_Destructor;
    UpdateFunction m_UpdateFunction;
    VerifyFunction m_VerifyFunction;
    //
    bool m_IsInitialized;
};

}}}
