﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/bcat/detail/service/bcat_Common.h>

namespace nn { namespace bcat { namespace detail { namespace service { namespace core {

/*!
    @brief      タスクの実行キューです。
*/
class ExecutionQueue
{
public:
    /*!
        @brief      キューに積むことができるアイテムの最大数です。
    */
    static const int ItemCountMax = 128;

public:
    /*!
        @brief      キューアイテムです。
    */
    struct Item
    {
        TaskId taskId;
        SessionId sessionId;
        nn::ApplicationId appId;
        uint32_t appVersion;
        DirectoryName dirName;
    };

private:
    /*!
        @brief      コンストラクタです。
    */
    ExecutionQueue() NN_NOEXCEPT;

public:
    /*!
        @brief      インスタンスを取得します。

        @return     インスタンス。
    */
    static ExecutionQueue& GetInstance() NN_NOEXCEPT
    {
        NN_FUNCTION_LOCAL_STATIC(ExecutionQueue, s_Instance);
        return s_Instance;
    }

public:
    /*!
        @brief      クリアします。
    */
    void Clear() NN_NOEXCEPT;

    /*!
        @brief      先頭にタスクを追加します。

        @param[out] outTaskId   タスク ID。
        @param[in]  sessionId   セッション ID。
        @param[in]  appId       アプリケーション ID。
        @param[in]  appVersion  アプリケーションバージョン。
        @param[in]  dirName     ディレクトリ名。

        @return     追加できたかどうか。

        @pre
            - sessionId != InvalidSessionId
            - appId != nn::ApplicationId::GetInvalidId()

        @details
                    ディレクトリ名に nullptr を指定した場合、ディレクトリ指定ではなく全体の同期が行われます。
    */
    bool AddToHead(TaskId* outTaskId, SessionId sessionId, nn::ApplicationId appId, uint32_t appVersion,
        const char* dirName = nullptr) NN_NOEXCEPT;

    /*!
        @brief      末尾にタスクを追加します。

        @param[out] outTaskId   タスク ID。
        @param[in]  sessionId   セッション ID。
        @param[in]  appId       アプリケーション ID。
        @param[in]  appVersion  アプリケーションバージョン。
        @param[in]  dirName     ディレクトリ名。

        @return     追加できたかどうか。

        @pre
            - sessionId != InvalidSessionId
            - appId != nn::ApplicationId::GetInvalidId()

        @details
                    ディレクトリ名に nullptr を指定した場合、ディレクトリ指定ではなく全体の同期が行われます。
    */
    bool AddToTail(TaskId* outTaskId, SessionId sessionId, nn::ApplicationId appId, uint32_t appVersion,
        const char* dirName = nullptr) NN_NOEXCEPT;

    /*!
        @brief      タスクを削除します。

        @param[in]  taskId  タスク ID。

        @pre
            - taskId != InvalidTaskId
    */
    void Remove(TaskId taskId) NN_NOEXCEPT;

    /*!
        @brief      タスクにキャンセルを通知し、その後削除します。

        @param[in]  taskId  タスク ID。

        @pre
            - taskId != InvalidTaskId
    */
    void CancelAndRemove(TaskId taskId) NN_NOEXCEPT;

    /*!
        @brief      先頭のタスクを取得し、キューから削除します。

        @param[out] outItem アイテム。

        @return     取得できたかどうか。

        @pre
            - outItem != nullptr
    */
    bool Pop(Item* outItem) NN_NOEXCEPT;

    /*!
        @brief      タスクが存在するかどうかを確認します。

        @return     タスクが存在するかどうか。
    */
    bool IsEmpty() const NN_NOEXCEPT;

    /*!
        @brief      指定したタスクが存在するかどうかを確認します。

        @param[in]  taskId  タスク ID。

        @pre
            - taskId != InvalidTaskId
    */
    bool Exists(TaskId taskId) NN_NOEXCEPT;

    /*!
        @brief      タスク数を取得します。

        @return     タスク数。
    */
    int GetCount() const NN_NOEXCEPT;

    /*!
        @brief      タスクが追加された時に Signal されるイベントを取得します。

        @return     イベント。
    */
    nn::os::Event& GetEvent() NN_NOEXCEPT;

private:
    //
    mutable nn::os::Mutex m_Mutex;
    //
    nn::os::Event m_Event;
    //
    Item m_Items[ItemCountMax];
    int m_Count;

private:
    //
    int Search(TaskId taskId) const NN_NOEXCEPT;
    //
    void RemoveItem(int index) NN_NOEXCEPT;
};

}}}}}
