﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/bcat/detail/service/bcat_Common.h>
#include <nn/bcat/detail/service/core/bcat_MetaView.h>
#include <nn/bcat/detail/service/core/bcat_DiffView.h>

namespace nn { namespace bcat { namespace detail { namespace service { namespace core {

/*!
    @brief      ディレクトリの同期を行うモジュールです。
*/
class DirectorySynchronizer
{
public:
    /*!
        @brief      コンストラクタです。
    */
    DirectorySynchronizer() NN_NOEXCEPT;

    /*!
        @brief      キャンセルイベントを設定します。

        @param[in]  pCancelEvent    キャンセルイベント。

        @pre
            - pCancelEvent != nullptr
    */
    void SetCancelEvent(nn::os::Event* pCancelEvent) NN_NOEXCEPT;

    /*!
        @brief      同期処理を行います。

        @param[in]  appId       アプリケーション ID。
        @param[in]  dirMeta     ディレクトリのメタ情報。
        @param[in]  buffer      ワークバッファ。
        @param[in]  bufferSize  ワークバッファのサイズ。

        @return     処理結果。

        @pre
            - appId != nn::ApplicationId::GetInvalidId()
            - buffer != nullptr
            - bufferSize > 0

        @details
                    本関数は、最初にトランザクションファイルを作成してコミットします。

                    ディレクトリの同期処理中に電源断が発生した場合、データ配信キャッシュストレージ上のファイルは不定な状態になります。@n
                    不定な状態でデータ配信キャッシュストレージ上のファイルにアクセスしてはいけません。
    */
    nn::Result Sync(nn::ApplicationId appId, const DirectoryMeta& dirMeta, void* buffer, size_t bufferSize) NN_NOEXCEPT;

    /*!
        @brief      復帰処理を行います。

        @param[in]  appId       アプリケーション ID。
        @param[in]  buffer      ワークバッファ。
        @param[in]  bufferSize  ワークバッファのサイズ。

        @return     処理結果。

        @pre
            - appId != nn::ApplicationId::GetInvalidId()
            - buffer != nullptr
            - bufferSize > 0

        @details
                    本関数は、トランザクションファイルの情報を元に、再度ディレクトリの同期処理を行います。@n
                    トランザクションファイルが存在しない場合、本関数は何も行いません。
    */
    nn::Result Recover(nn::ApplicationId appId, void* buffer, size_t bufferSize) NN_NOEXCEPT;

public:
    /*!
        @brief      トランザクションが作成されているかどうかを確認します。

        @param[in]  appId   アプリケーション ID。

        @return     トランザクションが作成されているかどうか。

        @pre
            - appId != nn::ApplicationId::GetInvalidId()
    */
    static bool IsTransactionCreated(nn::ApplicationId appId) NN_NOEXCEPT;

private:
    //
    nn::os::Mutex m_Mutex;
    //
    nn::ApplicationId m_AppId;
    DirectoryMeta m_DirMeta;
    //
    nn::os::Event* m_pCancelEvent;
    //
    DirectoryMetaView m_DirectoryMetaView;
    //
    ListFile m_Files[DeliveryCacheFileCountMaxPerDirectory];
    int m_FileCount;
    //
    FileMetaView m_FileMetaView;
    FileDiffView m_FileDiffView;

private:
    //
    nn::Result StepSetup(void* buffer, size_t bufferSize) NN_NOEXCEPT;
    nn::Result StepRemoveRemovedFiles() NN_NOEXCEPT;
    nn::Result StepRemoveOverwrittenFiles() NN_NOEXCEPT;
    nn::Result StepCopyFiles(void* buffer, size_t bufferSize) NN_NOEXCEPT;
    nn::Result StepCommit() NN_NOEXCEPT;
    //
    nn::Result CopyFiles(DiffType diffType, void* buffer, size_t bufferSize) NN_NOEXCEPT;
    nn::Result RemoveUnstableDirectory() NN_NOEXCEPT;
    //
    nn::Result LoadTransaction() NN_NOEXCEPT;
    nn::Result CreateTransaction() NN_NOEXCEPT;
    nn::Result DeleteTransaction() NN_NOEXCEPT;
    //
    bool IsCanceled() NN_NOEXCEPT;
};

}}}}}
