﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/bcat/detail/service/bcat_Common.h>
#include <nn/bcat/detail/service/msgpack/bcat_Msgpack.h>
#include <nn/bcat/detail/service/msgpack/bcat_MsgpackEventHandler.h>

namespace nn { namespace bcat { namespace detail { namespace service { namespace core {

/*!
    @brief      配信リストのリーダーです。
*/
class DeliveryListReader : public detail::service::msgpack::MsgpackEventHandler
{
private:
    NN_DISALLOW_COPY(DeliveryListReader);
    NN_DISALLOW_MOVE(DeliveryListReader);

public:
    /*!
        @brief      リストヘッダーのパース結果です。
    */
    struct ListHeaderParseBits
    {
        Bit64 topicId : 1;
        Bit64 isInService : 1;
        Bit64 isNintendoAccountRequired : 1;
        Bit64 requiredAppVersion : 1;
    };

    /*!
        @brief     ディレクトリのパース結果です。
    */
    struct ListDirectoryParseBits
    {
        Bit64 name : 1;
        Bit64 mode : 1;
        Bit64 digest : 1;
    };

    /*!
        @brief      ファイルのパース結果です。
    */
    struct ListFileParseBits
    {
        Bit64 name : 1;
        Bit64 dataId : 1;
        Bit64 size : 1;
        Bit64 url : 1;
        Bit64 digest : 1;
    };

    /*!
        @brief      走査コールバック用のワークバッファです。
    */
    typedef std::aligned_storage<sizeof (ListFile) * DeliveryCacheFileCountMaxPerDirectory>::type WalkWorkBuffer;

    /*!
        @brief      走査コールバックです。

        @param[in]  dir     ディレクトリ。
        @param[in]  files   ファイルリスト。
        @param[in]  count   ファイルリストの要素数。
        @param[in]  param   コールバックパラメータ。

        @details
                    本コールバックは、ディレクトリのパース完了毎に呼び出されます。
    */
    typedef void (*WalkCallback)(const ListDirectory& dir, const ListFile* files, int count, void* param);

public:
    /*!
        @brief      コンストラクタです。
    */
    DeliveryListReader() NN_NOEXCEPT;

    /*!
        @brief      ディレクトリリストを読み込むためのセットアップを行います。

        @param[out] outCount    取得したディレクトリ数。
        @param[out] outDirs     ディレクトリリスト。
        @param[in]  count       ディレクトリリストの要素数。

        @pre
            - outCount != nullptr
            - outDirs != nullptr
            - count > 0
    */
    void SetupForDirectoryListing(int* outCount, ListDirectory* outDirs, int count) NN_NOEXCEPT;

    /*!
        @brief      ファイルリストを読み込むためのセットアップを行います。

        @param[out] outCount    取得したファイル数。
        @param[out] outFiles    ファイルリスト。
        @param[in]  count       ファイルリストの要素数。
        @param[in]  dirName     ディレクトリ名。

        @pre
            - outCount != nullptr
            - outFiles != nullptr
            - count > 0
            - dirName != nullptr
    */
    void SetupForFileListing(int* outCount, ListFile* outFiles, int count, const char* dirName) NN_NOEXCEPT;

    /*!
        @brief      配信リストを走査するためのセットアップを行います。

        @param[in]  callback    走査コールバック関数。
        @param[in]  param       走査コールバック関数のパラメータ。
        @param[in]  buffer      ワークバッファ。

        @pre
            - callback != nullptr
    */
    void SetupForWalking(WalkCallback callback, void* param, WalkWorkBuffer& buffer) NN_NOEXCEPT;

    /*!
        @brief      配信リストを読み込みます。

        @param[out] outHeader   配信リストヘッダー。省略可能です。
        @param[in]  stream      入力ストリーム。

        @pre
            - セットアップを行っている。
    */
    nn::Result Read(ListHeader* outHeader, nne::nlib::InputStream& stream) NN_NOEXCEPT;

private:
    //
    detail::service::msgpack::JsonPath m_JsonPath;
    //
    ListHeader m_ListHeader;
    ListHeaderParseBits m_ListHeaderBits;
    //
    ListDirectory m_CurrentDir;
    ListDirectoryParseBits m_CurrentDirBits;
    //
    ListFile m_CurrentFile;
    ListFileParseBits m_CurrentFileBits;
    //
    bool m_IsWalkingEnabled;
    bool m_IsFileListingRequired;
    //
    const char* m_SpecifiedDirName;
    int m_SpecifiedDirFileCount;
    //
    bool m_IsSpecifiedDir;
    bool m_IsSpecifiedDirFound;
    //
    ListDirectory* m_Dirs;
    int m_DirIndex;
    int m_DirCountMax;
    //
    ListFile* m_Files;
    int m_FileIndex;
    int m_FileCountMax;
    //
    int* m_OutCount;
    //
    WalkCallback m_Callback;
    void* m_Param;

private:
    // MsgpackEventHandler
    virtual bool OnStartObject(const detail::service::msgpack::JsonPath& jsonPath) NN_NOEXCEPT NN_OVERRIDE;
    virtual bool OnEndObject(const detail::service::msgpack::JsonPath& jsonPath) NN_NOEXCEPT NN_OVERRIDE;
    virtual bool OnStartArray(const detail::service::msgpack::JsonPath& jsonPath) NN_NOEXCEPT NN_OVERRIDE;
    virtual bool OnBoolean(const detail::service::msgpack::JsonPath& jsonPath, bool value) NN_NOEXCEPT NN_OVERRIDE;
    virtual bool OnInteger(const detail::service::msgpack::JsonPath& jsonPath, const detail::service::msgpack::AnyInteger& value) NN_NOEXCEPT NN_OVERRIDE;
    virtual bool OnString(const detail::service::msgpack::JsonPath& jsonPath, const char* value, size_t length, bool isOverflowed) NN_NOEXCEPT NN_OVERRIDE;
};

}}}}}
