﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/bcat/detail/service/bcat_Common.h>
#include <nn/bcat/detail/service/core/bcat_FileSystemConfig.h>

namespace nn { namespace bcat { namespace detail { namespace service { namespace core {

/*!
    @brief      データ配信キャッシュストレージの管理モジュールです。
*/
class DeliveryCacheStorageManager
{
private:
    NN_DISALLOW_COPY(DeliveryCacheStorageManager);
    NN_DISALLOW_MOVE(DeliveryCacheStorageManager);

public:
    /*!
        @brief      データ配信キャッシュストレージの同時マウント数の最大値です。
    */
    static const int MountCountMax = 4;

    /*!
        @brief      データ配信キャッシュストレージのジャーナルサイズです。

        @details
                    設定値：128 blocks: 2048KB
    */
    static const int64_t StorageJournalSize = 128 * BlockSize;

private:
    /*!
        @brief      コンストラクタです。
    */
    DeliveryCacheStorageManager() NN_NOEXCEPT;

public:
    /*!
        @brief      インスタンスを取得します。

        @return     インスタンス。
    */
    static DeliveryCacheStorageManager& GetInstance() NN_NOEXCEPT
    {
        NN_FUNCTION_LOCAL_STATIC(DeliveryCacheStorageManager, s_Instance);
        return s_Instance;
    }

public:
    /*!
        @brief      テストモードを有効にします。

        @details
                    テストモードが有効な場合、以下の API はストレージへのアクセスを行いません。

                    - DeliveryCacheStorageManager::Mount
                    - DeliveryCacheStorageManager::Unmount
                    - DeliveryCacheStorageManager::Commit

                    テスト実行前に必要とするストレージをマウントしてください。
    */
    void EnableTestMode() NN_NOEXCEPT;

    /*!
        @brief      データ配信キャッシュストレージをマウントします。

        @param[in]  appId   アプリケーション ID。

        @return     処理結果。

        @pre
            - appId != nn::ApplicationId::GetInvalidId()

        @details
                    アプリケーション、または、バックグラウンド処理の要求によりすでにマウントされていた場合、 ResultLocked が返ります。
    */
    nn::Result Mount(nn::ApplicationId appId) NN_NOEXCEPT;

    /*!
        @brief      データ配信キャッシュストレージをアンマウントします。

        @param[in]  appId   アプリケーション ID。

        @pre
            - appId != nn::ApplicationId::GetInvalidId()
    */
    void Unmount(nn::ApplicationId appId) NN_NOEXCEPT;

    /*!
        @brief      データ配信キャッシュストレージをコミットします。

        @param[in]  appId   アプリケーション ID。

        @pre
            - appId != nn::ApplicationId::GetInvalidId()
    */
    nn::Result Commit(nn::ApplicationId appId) NN_NOEXCEPT;

    /*!
        @brief      パスフレーズの保存を要求します。

        @param[in]  appId       アプリケーション ID。
        @param[in]  passphrase  パスフレーズ。

        @pre
            - appId != nn::ApplicationId::GetInvalidId()
            - passphrase != nullptr

        @details
                    データ配信キャッシュストレージのマウント状態によって、パスフレーズの保存先を変更します。@n
                    アプリケーションがストレージにアクセスしている間、または、ダウンロード処理が行われている間はテンポラリ領域に保存します。

                    パスフレーズがテンポラリ領域に保存されている場合、ストレージをマウントしているモジュールは適切なタイミングでストレージに保存する必要があります。

        @see
            - PassphraseManager::Save
            - PassphraseManager::SaveTemporary
    */
    nn::Result RequestSavePassphrase(nn::ApplicationId appId, const char* passphrase) NN_NOEXCEPT;

    /*!
        @brief      ストレージの空き容量を取得します。

        @param[out] outSize 空き容量。
        @param[in]  appId   アプリケーション ID。

        @return     処理結果。

        @pre
            - outSize != nullptr
            - appId != nn::ApplicationId::GetInvalidId()
    */
    nn::Result GetFreeSpaceSize(int64_t* outSize, nn::ApplicationId appId) NN_NOEXCEPT;

    /*!
        @brief      パスフレーズにアクセスするためのパスを作成します。

        @param[in]  path    バッファ。
        @param[in]  size    バッファサイズ。
        @param[in]  appId   アプリケーション ID。

        @pre
            - path != nullptr
            - size > 0
            - appId != nn::ApplicationId::GetInvalidId()
    */
    void MakePassphrasePath(char* path, size_t size, nn::ApplicationId appId) NN_NOEXCEPT;

    /*!
        @brief      配信リストにアクセスするためのパスを作成します。

        @param[in]  path    バッファ。
        @param[in]  size    バッファサイズ。
        @param[in]  appId   アプリケーション ID。

        @pre
            - path != nullptr
            - size > 0
            - appId != nn::ApplicationId::GetInvalidId()
    */
    void MakeListPath(char* path, size_t size, nn::ApplicationId appId) NN_NOEXCEPT;

    /*!
        @brief      ETag にアクセスするためのパスを作成します。

        @param[in]  path    バッファ。
        @param[in]  size    バッファサイズ。
        @param[in]  appId   アプリケーション ID。

        @pre
            - path != nullptr
            - size > 0
            - appId != nn::ApplicationId::GetInvalidId()
    */
    void MakeETagPath(char* path, size_t size, nn::ApplicationId appId) NN_NOEXCEPT;

    /*!
        @brief      ニンテンドーアカウント必須フラグにアクセスするためのパスを作成します。

        @param[in]  path    バッファ。
        @param[in]  size    バッファサイズ。
        @param[in]  appId   アプリケーション ID。

        @pre
            - path != nullptr
            - size > 0
            - appId != nn::ApplicationId::GetInvalidId()
    */
    void MakeNintendoAccountRequiredPath(char* path, size_t size, nn::ApplicationId appId) NN_NOEXCEPT;

    /*!
        @brief      トランザクション情報にアクセスするためのパスを作成します。

        @param[in]  path    バッファ。
        @param[in]  size    バッファサイズ。
        @param[in]  appId   アプリケーション ID。

        @pre
            - path != nullptr
            - size > 0
            - appId != nn::ApplicationId::GetInvalidId()
    */
    void MakeTransactionPath(char* path, size_t size, nn::ApplicationId appId) NN_NOEXCEPT;

    /*!
        @brief      ファイルにアクセスするためのパスを作成します。

        @param[in]  path        バッファ。
        @param[in]  size        バッファサイズ。
        @param[in]  appId       アプリケーション ID。
        @param[in]  dirName     ディレクトリ名。
        @param[in]  fileName    ファイル名。

        @pre
            - path != nullptr
            - size > 0
            - appId != nn::ApplicationId::GetInvalidId()
            - dirName != nullptr
            - fileName != nullptr
    */
    void MakeFilePath(char* path, size_t size, nn::ApplicationId appId, const char* dirName, const char* fileName) NN_NOEXCEPT;

    /*!
        @brief      ファイルメタにアクセスするためのパスを作成します。

        @param[in]  path    バッファ。
        @param[in]  size    バッファサイズ。
        @param[in]  appId   アプリケーション ID。
        @param[in]  dirName ディレクトリ名。

        @pre
            - path != nullptr
            - size > 0
            - appId != nn::ApplicationId::GetInvalidId()
            - dirName != nullptr
    */
    void MakeFileMetaPath(char* path, size_t size, nn::ApplicationId appId, const char* dirName) NN_NOEXCEPT;

    /*!
        @brief      ディレクトリ群のルートディレクトリのパスを作成します。

        @param[in]  path    バッファ。
        @param[in]  size    バッファサイズ。
        @param[in]  appId   アプリケーション ID。

        @pre
            - path != nullptr
            - size > 0
            - appId != nn::ApplicationId::GetInvalidId()
    */
    void MakeRootDirectoryPath(char* path, size_t size, nn::ApplicationId appId) NN_NOEXCEPT;

    /*!
        @brief      ディレクトリにアクセスするためのパスを作成します。

        @param[in]  path    バッファ。
        @param[in]  size    バッファサイズ。
        @param[in]  appId   アプリケーション ID。
        @param[in]  dirName ディレクトリ名。

        @pre
            - path != nullptr
            - size > 0
            - appId != nn::ApplicationId::GetInvalidId()
            - dirName != nullptr
    */
    void MakeDirectoryPath(char* path, size_t size, nn::ApplicationId appId, const char* dirName) NN_NOEXCEPT;

    /*!
        @brief      ディレクトリメタにアクセスするためのパスを作成します。

        @param[in]  path    バッファ。
        @param[in]  size    バッファサイズ。
        @param[in]  appId   アプリケーション ID。

        @pre
            - path != nullptr
            - size > 0
            - appId != nn::ApplicationId::GetInvalidId()
    */
    void MakeDirectoryMetaPath(char* path, size_t size, nn::ApplicationId appId) NN_NOEXCEPT;

private:
    //
    struct Record
    {
        nn::ApplicationId appId;
        int referenceCount;
    };

private:
    //
    nn::os::Mutex m_Mutex;
    //
    Record m_Records[MountCountMax];
    //
    bool m_IsTestModeEnabled;

private:
    //
    int SearchRecord(nn::ApplicationId appId) NN_NOEXCEPT;
    int SearchEmptyRecord() NN_NOEXCEPT;
};

}}}}}

/*!
    @brief      データ配信キャッシュストレージを局所的にマウントします。
*/
#define NN_DETAIL_BCAT_DELIVERY_CACHE_STORAGE_SCOPED_MOUNT(appId) \
    NN_RESULT_DO(detail::service::core::DeliveryCacheStorageManager::GetInstance().Mount(appId)); \
    NN_UTIL_SCOPE_EXIT                                                                            \
    {                                                                                             \
        detail::service::core::DeliveryCacheStorageManager::GetInstance().Unmount(appId);         \
    }
