﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/bcat/detail/service/bcat_Common.h>

namespace nn { namespace bcat { namespace detail { namespace service { namespace core {

/*!
    @brief      データ配信の進捗を管理するモジュールです。
*/
class DeliveryCacheProgressManager
{
private:
    NN_DISALLOW_COPY(DeliveryCacheProgressManager);
    NN_DISALLOW_MOVE(DeliveryCacheProgressManager);

public:
    /*!
        @brief      管理する進捗イベントの最大数です。

        @details
                    クライアント向け IPC セッション＋BG 処理用です。
    */
    static const int CountMax = detail::ipc::DeliveryCacheProgressServiceCountMax + 1;

private:
    /*!
        @brief      コンストラクタです。
    */
    DeliveryCacheProgressManager() NN_NOEXCEPT;

    /*!
        @brief      デストラクタです。
    */
    ~DeliveryCacheProgressManager() NN_NOEXCEPT;

public:
    /*!
        @brief      インスタンスを取得します。

        @return     インスタンス。
    */
    static DeliveryCacheProgressManager& GetInstance() NN_NOEXCEPT
    {
        NN_FUNCTION_LOCAL_STATIC(DeliveryCacheProgressManager, s_Instance);
        return s_Instance;
    }

public:
    /*!
        @brief      タスクを登録します。

        @param[in]  taskId  タスク ID。

        @pre
            - taskId != InvalidTaskId
    */
    void Register(TaskId taskId) NN_NOEXCEPT;

    /*!
        @brief      タスクの登録を解除します。

        @param[in]  taskId  タスク ID。

        @pre
            - taskId != InvalidTaskId

        @details
                    更新検知イベントがアタッチされていた場合、自動的に削除します。
    */
    void Unregister(TaskId taskId) NN_NOEXCEPT;

    /*!
        @brief      更新検知イベントをアタッチします。

        @param[in]  taskId      タスク ID。
        @param[in]  eventHandle イベントのハンドル。

        @pre
            - taskId != InvalidTaskId

        @details
                    eventHandle には、システムイベントの WritableHandle を指定してください。
    */
    void AttachUpdateDetectionEvent(TaskId taskId, nn::os::NativeHandle eventHandle) NN_NOEXCEPT;

    /*!
        @brief      進捗を取得します。

        @param[out] outProgress 進捗。
        @param[in]  taskId      タスク ID。

        @pre
            - outProgress != nullptr
            - taskId != InvalidTaskId
    */
    void GetProgress(DeliveryCacheProgressImpl* outProgress, TaskId taskId) NN_NOEXCEPT;

    /*!
        @brief      進捗を設定します。

        @param[in]  taskId      タスク ID。
        @param[in]  progress    進捗。

        @pre
            - taskId != InvalidTaskId
    */
    void SetProgress(TaskId taskId, const DeliveryCacheProgressImpl& progress) NN_NOEXCEPT;

private:
    //
    struct Record
    {
        TaskId taskId;
        nn::os::SystemEventType systemEvent;
        bool isSystemEventAttached;
        DeliveryCacheProgressStatusImpl status;
        int32_t result;
        DirectoryName dirName;
        FileName fileName;
        int64_t currentDownloaded;
        int64_t currentTotal;
        int64_t wholeDownloaded;
        int64_t wholeTotal;
    };

private:
    //
    nn::os::Mutex m_Mutex;
    //
    Record m_Records[CountMax];

private:
    //
    int Search(TaskId taskId) const NN_NOEXCEPT;
    //
    void Signal(int index) NN_NOEXCEPT;
};

}}}}}
