﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/bcat/detail/service/bcat_Common.h>

namespace nn { namespace bcat { namespace detail { namespace service { namespace core {

/*!
    @brief      バックグラウンドワーカースレッドです。
*/
class BackgroundWorkerThread
{
public:
    /*!
        @brief      スレッドの実行を開始します。
    */
    static void Start() NN_NOEXCEPT;

    /*!
        @brief      スレッドの実行を終了します。
    */
    static void Stop() NN_NOEXCEPT;

    /*!
        @brief      進捗を取得します。

        @param[out] outProgress 進捗。
        @param[in]  taskId      タスク ID。

        @pre
            - outProgress != nullptr
            - taskId != InvalidTaskId
    */
    static void GetProgress(DeliveryCacheProgressImpl* outProgress, TaskId taskId) NN_NOEXCEPT;

    /*!
        @brief      現在実行中のバックグラウンドタスクに紐付くアプリケーション ID を取得します。

        @param[out] outAppId    アプリケーション ID。

        @return     現在実行中のバックグラウンドタスクが存在するかどうか。

        @pre
            - outAppId != nullptr
    */
    static bool GetCurrentBackgroundTaskApplicationId(nn::ApplicationId* outAppId) NN_NOEXCEPT;

    /*!
        @brief      指定したアプリケーションのバックグラウンドタスクを登録します。

        @param[in]  appId       アプリケーション ID。
        @param[in]  appVersion  アプリケーションバージョン。

        @pre
            - appId != nn::ApplicationId::GetInvalidId()
    */
    static void RegisterBackgroundTask(nn::ApplicationId appId, uint32_t appVersion) NN_NOEXCEPT;

    /*!
        @brief      指定したアプリケーションのバックグラウンドタスクの登録を解除します。

        @param[in]  appId   アプリケーション ID。

        @pre
            - appId != nn::ApplicationId::GetInvalidId()
    */
    static void UnregisterBackgroundTask(nn::ApplicationId appId) NN_NOEXCEPT;

    /*!
        @brief      指定したアプリケーションのバックグラウンドタスクを一時停止します。

        @param[in]  appId   アプリケーション ID。

        @pre
            - appId != nn::ApplicationId::GetInvalidId()
    */
    static void SuspendBackgroundTask(nn::ApplicationId appId) NN_NOEXCEPT;

    /*!
        @brief      指定したアプリケーションのバックグラウンドタスクを再開します。

        @param[in]  appId   アプリケーション ID。

        @pre
            - appId != nn::ApplicationId::GetInvalidId()
    */
    static void ResumeBackgroundTask(nn::ApplicationId appId) NN_NOEXCEPT;

    /*!
        @brief      指定したアプリケーションのタスク実行をブロックします。

        @param[in]  appId   アプリケーション ID。

        @pre
            - appId != nn::ApplicationId::GetInvalidId()

        @details
                    指定したアプリケーションのタスクを即座に停止し、完了するまで待機します。@n
                    ブロックしている間、フォアグラウンド／バックグラウンドに関わらずタスクは実行されません。
    */
    static void BlockTask(nn::ApplicationId appId) NN_NOEXCEPT;

    /*!
        @brief      タスク実行のブロックを解除します。

        @param[in]  appId   アプリケーション ID。

        @pre
            - appId != nn::ApplicationId::GetInvalidId()
    */
    static void UnblockTask(nn::ApplicationId appId) NN_NOEXCEPT;

    /*!
        @brief      現在実行中のバックグラウンドタスクをキャンセルします。
    */
    static void CancelCurrentBackgroundTask() NN_NOEXCEPT;

    /*!
        @brief      指定したセッションのタスクをキャンセルします。

        @param[in]  sessionId   セッション ID。

        @pre
            - sessionId != InvalidSessionId
    */
    static void CancelTask(SessionId sessionId) NN_NOEXCEPT;

    /*!
        @brief      次回のスケジュールまでの時間間隔を取得します。

        @param[out] outInterval 次回のスケジュールまでの時間間隔。

        @return     スケジュールする必要があるかどうか。

        @pre
            - outInterval != nullptr
    */
    static bool GetNextScheduleInterval(nn::TimeSpan* outInterval) NN_NOEXCEPT;

    /*!
        @brief      NPNS の接続を通知します。
    */
    static void NotifyNpnsConnected() NN_NOEXCEPT;

    /*!
        @brief      システムの実行可能状態の変更を通知します。

        @param[in]  isRunnable  実行可能かどうか。
    */
    static void NotifySystemRunnableStateChanged(bool isRunnable) NN_NOEXCEPT;
};

}}}}}
