﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/bcat/bcat_Types.h>
#include <nn/bcat/bcat_TypesDebug.h>
#include <nn/nn_StaticAssert.h>

namespace nn { namespace bcat { namespace detail {

/*!
    @brief      1 ストレージあたりの最大ファイル数です。
*/
// const int DeliveryCacheFileCountMaxPerStorage = 2000;

/*!
    @brief      ディレクトリのモードです。

    @details
                ディレクトリ内エントリーの扱いを指定します。
*/
enum DirectoryMode : int8_t
{
    DirectoryMode_Sync = 0, //!< 同期モード。ディレクトリ内のファイルは完全同期されます。
    DirectoryMode_Copy = 1  //!< 複製モード。ディレクトリ内のファイルは同期されず、アプリケーションの要求によってダウンロード処理が行われます。
};

/*!
    @brief      差分種別です。
*/
enum DiffType : int8_t
{
    DiffType_Unknown     = 0, //!< 不明。
    DiffType_NotModified = 1, //!< 更新なし。
    DiffType_Add         = 2, //!< 追加。
    DiffType_Remove      = 3, //!< 削除。
    DiffType_Update      = 4  //!< 更新。
};

/*!
    @brief      データ配信の進捗の状態です。
*/
enum DeliveryCacheProgressStatusImpl : int32_t
{
    DeliveryCacheProgressStatusImpl_None        = 0, //!< 進捗なし。
    DeliveryCacheProgressStatusImpl_Queued      = 1, //!< 実行待ち。
    DeliveryCacheProgressStatusImpl_Connect     = 2, //!< ネットワーク接続中。
    DeliveryCacheProgressStatusImpl_ProcessList = 3, //!< データ配信リストの処理中。
    DeliveryCacheProgressStatusImpl_Download    = 4, //!< ダウンロード処理中。
    DeliveryCacheProgressStatusImpl_Commit      = 5, //!< コミット処理中。
    DeliveryCacheProgressStatusImpl_Done        = 9  //!< 完了。
};

/*!
    @brief      リビジョンのハッシュ値です。
*/
struct RevisionHash
{
    Bit8 value[8];
};

/*!
    @brief      トピック ID です。
*/
struct TopicId
{
    char value[32];
};

/*!
    @brief      配信データのダウンロード URL です。
*/
struct Url
{
    char value[160];
};

/*!
    @brief      配信リストのヘッダーです。
*/
struct ListHeader
{
    TopicId topicId;
    bool isInService;
    bool isNintendoAccountRequired;
    uint32_t requiredAppVersion;
};

/*!
    @brief      配信リストのディレクトリです。
*/
struct ListDirectory
{
    DirectoryName name;
    DirectoryMode mode;
    Bit8 reserved[7];
    Digest digest;
};

/*!
    @brief      配信リストのファイルエントリーです。
*/
struct ListFile
{
    FileName name;
    uint64_t dataId;
    uint64_t size;
    Url url;
    Digest digest;
};

/*!
    @brief      ファイルのメタ情報です。
*/
struct FileMeta
{
    FileName name;
    uint64_t dataId;
    uint64_t size;
    Digest digest;
    Bit8 reserved1[16]; // service.id
    Bit8 reserved2[16]; // ap.uuid
    Bit8 reserved3[32];

public:
    /*!
        @brief      構造体のバージョンです。
    */
    static const int32_t StructureVersion = 1;
};

NN_STATIC_ASSERT(sizeof (FileMeta) == 128);

/*!
    @brief      ディレクトリのメタ情報です。
*/
struct DirectoryMeta
{
    DirectoryName name;
    Digest digest;
    Bit8 reserved[16];

public:
    /*!
        @brief      構造体のバージョンです。
    */
    static const int32_t StructureVersion = 1;
};

NN_STATIC_ASSERT(sizeof (DirectoryMeta) == 64);

/*!
    @brief      データ配信の進捗情報を格納する構造体です。
*/
struct DeliveryCacheProgressImpl
{
    DeliveryCacheProgressStatusImpl status; //!< 進捗の状態。
    int32_t result;                         //!< 処理結果。
    DirectoryName dirName;                  //!< 処理中のディレクトリ名。
    FileName fileName;                      //!< 処理中のファイル名。
    int64_t currentDownloaded;              //!< 現在処理しているファイルのダウンロード済みサイズ。
    int64_t currentTotal;                   //!< 現在処理しているファイルのサイズ。
    int64_t wholeDownloaded;                //!< 全体のダウンロード済みサイズ。
    int64_t wholeTotal;                     //!< 全体のサイズ。

    Bit8 reserved[404];

public:
    /*!
        @brief      クリアします。
    */
    void Clear() NN_NOEXCEPT;

    /*!
        @brief      ネットワーク接続の開始を通知します。
    */
    void NotifyStartConnect() NN_NOEXCEPT;

    /*!
        @brief      配信リストの処理の開始を通知します。
    */
    void NotifyStartProcessList() NN_NOEXCEPT;

    /*!
        @brief      ダウンロードの全体サイズを設定します。

        @param[in]  wholeTotal  全体のサイズ。
    */
    void SetWholeDownloadSize(int64_t wholeTotal) NN_NOEXCEPT;

    /*!
        @brief      ダウンロード進捗を設定します。

        @param[in]  downloaded  ダウンロード済みファイルの合計サイズ。
        @param[in]  dirName     最後にダウンロードしたディレクトリのディレクトリ名。
    */
    void SetDownloadProgress(int64_t downloaded, const DirectoryName& dirName) NN_NOEXCEPT;

    /*!
        @brief      ファイルのダウンロード開始を通知します。

        @param[in]  dirName     ディレクトリ名。
        @param[in]  fileName    ファイル名。
        @param[in]  fileSize    ファイルサイズ。
    */
    void NotifyStartDownloadFile(const DirectoryName& dirName, const FileName& fileName, int64_t fileSize) NN_NOEXCEPT;

    /*!
        @brief      ファイルのダウンロードの進捗を更新します。

        @param[in]  downloaded  ダウンロードしたサイズ。

        @return     進捗が前回より進んだかどうか。
    */
    bool UpdateDownloadFileProgress(int64_t downloaded) NN_NOEXCEPT;

    /*!
        @brief      ディレクトリのコミット開始を通知します。

        @param[in]  dirName ディレクトリ名。
    */
    void NotifyStartCommitDirectory(const DirectoryName& dirName) NN_NOEXCEPT;

    /*!
        @brief      処理の完了を通知します。

        @param[in]  result  処理結果。
    */
    void NotifyDone(nn::Result result) NN_NOEXCEPT;
};

NN_STATIC_ASSERT(sizeof (DeliveryCacheProgressImpl) == 512);

/*!
    @brief      セッション ID です。
*/
struct SessionId
{
    uint64_t value; //!< 値。
};

inline bool operator==(const SessionId& lhs, const SessionId& rhs) NN_NOEXCEPT
{
    return lhs.value == rhs.value;
}

inline bool operator!=(const SessionId& lhs, const SessionId& rhs) NN_NOEXCEPT
{
    return !(lhs == rhs);
}

/*!
    @brief      無効なセッション ID です。
*/
const SessionId InvalidSessionId = {0};

/*!
    @brief      タスク ID です。
*/
struct TaskId
{
    uint64_t value; //!< 値。
};

inline bool operator==(const TaskId& lhs, const TaskId& rhs) NN_NOEXCEPT
{
    return lhs.value == rhs.value;
}

inline bool operator!=(const TaskId& lhs, const TaskId& rhs) NN_NOEXCEPT
{
    return !(lhs == rhs);
}

/*!
    @brief      無効なタスク ID です。
*/
const TaskId InvalidTaskId = {0};

}}}
