﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_TimeSpan.h>
#include <nn/applet/applet_Types.h>
#include <nn/audio/audio_Result.h>

namespace nn {
namespace audio {

/**
 * @brief       applet を指定した audio 機能の一時停止、再開を行う際の推奨フェード時間です。
 */
const nn::TimeSpan AppletTransitionFadeTime = nn::TimeSpan::FromMilliSeconds(10);

/**
 * @brief       applet を指定したマスターボリューム変更時に指定できる最小ボリューム値です。
 */
const float ProcessMasterVolumeMin = 0.0f;

/**
 * @brief       applet を指定した録音ボリューム変更時に指定できる最大ボリューム値です。
 */
const float ProcessRecordVolumeMax = 1.0f;

/**
 * @brief       applet を指定した録音ボリューム変更時に指定できる最小ボリューム値です。
 */
const float ProcessRecordVolumeMin = 0.0f;

/**
 * @brief       applet を指定したマスターボリューム変更時に指定できる最大ボリューム値です。
 */
const float ProcessMasterVolumeMax = 1.0f;

/**
 * @brief       インスタンスを操作するための applet 識別子を登録します。
 * @param[in]   id                      applet 識別子
 * @retresult
 * @handleresult{nn::audio::ResultMaxAppletResourceUserIdRegistered}
 * @endretresult
 * @details      applet 識別子毎の本関数の呼び出し回数はライブラリ内部でカウントされます。
 */
nn::Result RegisterAppletResourceUserId(nn::applet::AppletResourceUserId id) NN_NOEXCEPT;

/**
 * @brief       インスタンスを操作するための applet 識別子の登録を解除します。
 * @param[in]   id                      applet 識別子
 * @retresult
 * @handleresult{nn::audio::ResultAppletResourceUserIdNotFound}
 * @endretresult
 * @details     applet 識別子毎の RegisterAppletResourceUserId の呼び出し回数と同じ回数分、
 *              本関数が呼ばれた場合にのみ、実際に登録が解除されます。
 */
nn::Result UnregisterAppletResourceUserId(nn::applet::AppletResourceUserId id) NN_NOEXCEPT;

/**
 * @brief       指定した applet に紐づくすべての AudioIn インスタンスを一時停止します。
 * @param[in]   id                      applet 識別子
 * @param[in]   fadeTime                一時停止までのフェード時間
 * @retresult
 * @handleresult{nn::audio::ResultAppletResourceUserIdNotFound}
 * @endretresult
 * @pre
 * - nn::TimeSpan::FromMilliSeconds(0) <= fadeTime && fadeTime <= nn::TimeSpan::FromMilliSeconds(10)
 * @detail
 * 本関数は applet 遷移目的で呼ばれることを想定したものです。それ以外の用途に利用することはできません。
 *
 * 本関数は再帰的に呼び出すことはできません。
 */
nn::Result RequestSuspendAudioIns(nn::applet::AppletResourceUserId id, nn::TimeSpan fadeTime) NN_NOEXCEPT;

/**
 * @brief       指定した applet に紐づくすべての AudioIn インスタンスを再開します。
 * @param[in]   id                      applet 識別子
 * @param[in]   fadeTime                再開までのフェード時間
 * @retresult
 * @handleresult{nn::audio::ResultAppletResourceUserIdNotFound}
 * @endretresult
 * @pre
 * - nn::TimeSpan::FromMilliSeconds(0) <= fadeTime && fadeTime <= nn::TimeSpan::FromMilliSeconds(10)
 * @detail
 * 本関数は applet 遷移目的で呼ばれることを想定したものです。それ以外の用途に利用することはできません。
 */
nn::Result RequestResumeAudioIns(nn::applet::AppletResourceUserId id, nn::TimeSpan fadeTime) NN_NOEXCEPT;

/**
 * @brief       指定した applet に紐づくすべての AudioIn インスタンスのマスターボリュームを取得します。
 * @param[out]  pOutVolume              ボリューム
 * @param[in]   id                      applet 識別子
 * @retresult
 * @handleresult{nn::audio::ResultAppletResourceUserIdNotFound}
 * @endretresult
 * @pre
 * - pOutVolume != nullptr
 */
nn::Result GetAudioInsProcessMasterVolume(float* pOutVolume, nn::applet::AppletResourceUserId id) NN_NOEXCEPT;

/**
 * @brief       指定した applet に紐づくすべての AudioIn インスタンスのマスターボリュームを設定します。
 * @param[in]   id                      applet 識別子
 * @param[in]   volume                  ボリューム
 * @param[in]   fadeTime                指定したボリュームに変化するまでのフェード時間
 * @retresult
 * @handleresult{nn::audio::ResultAppletResourceUserIdNotFound}
 * @endretresult
 * @pre
 * - nn::audio::ProcessMasterVolumeMin <= volume && volume <= nn::audio::ProcessMasterVolumeMax
 * - nn::TimeSpan::FromMilliSeconds(0) <= fadeTime && fadeTime <= nn::TimeSpan::FromMilliSeconds(1000)
 */
nn::Result SetAudioInsProcessMasterVolume(nn::applet::AppletResourceUserId id, float volume, nn::TimeSpan fadeTime) NN_NOEXCEPT;


/**
 * @brief       指定した applet に紐づくすべての AudioOut インスタンスを一時停止します。
 * @param[in]   id                      applet 識別子
 * @param[in]   fadeTime                一時停止までのフェード時間
 * @retresult
 * @handleresult{nn::audio::ResultAppletResourceUserIdNotFound}
 * @endretresult
 * @pre
 * - nn::TimeSpan::FromMilliSeconds(0) <= fadeTime && fadeTime <= nn::TimeSpan::FromMilliSeconds(10)
 * @detail
 * 本関数は applet 遷移目的で呼ばれることを想定したものです。それ以外の用途に利用することはできません。
 *
 * 本関数は再帰的に呼び出すことはできません。
 */
nn::Result RequestSuspendAudioOuts(nn::applet::AppletResourceUserId id, nn::TimeSpan fadeTime) NN_NOEXCEPT;

/**
 * @brief       指定した applet に紐づくすべての AudioOut インスタンスを再開します。
 * @param[in]   id                      applet 識別子
 * @param[in]   fadeTime                再開までのフェード時間
 * @retresult
 * @handleresult{nn::audio::ResultAppletResourceUserIdNotFound}
 * @endretresult
 * @pre
 * - nn::TimeSpan::FromMilliSeconds(0) <= fadeTime && fadeTime <= nn::TimeSpan::FromMilliSeconds(10)
 * @detail
 * 本関数は applet 遷移目的で呼ばれることを想定したものです。それ以外の用途に利用することはできません。
 */
nn::Result RequestResumeAudioOuts(nn::applet::AppletResourceUserId id, nn::TimeSpan fadeTime) NN_NOEXCEPT;

/**
 * @brief       指定した applet に紐づくすべての AudioOut インスタンスのマスターボリュームを取得します。
 * @param[out]  pOutVolume              ボリューム
 * @param[in]   id                      applet 識別子
 * @retresult
 * @handleresult{nn::audio::ResultAppletResourceUserIdNotFound}
 * @endretresult
 * @pre
 * - pOutVolume != nullptr
 */
nn::Result GetAudioOutsProcessMasterVolume(float* pOutVolume, nn::applet::AppletResourceUserId id) NN_NOEXCEPT;

/**
 * @brief       指定した applet に紐づくすべての AudioOut インスタンスのマスターボリュームを設定します。
 * @param[in]   id                      applet 識別子
 * @param[in]   volume                  ボリューム
 * @param[in]   fadeTime                指定したボリュームに変化するまでのフェード時間
 * @retresult
 * @handleresult{nn::audio::ResultAppletResourceUserIdNotFound}
 * @endretresult
 * @pre
 * - nn::audio::ProcessMasterVolumeMin <= volume && volume <= nn::audio::ProcessMasterVolumeMax
 * - nn::TimeSpan::FromMilliSeconds(0) <= fadeTime && fadeTime <= nn::TimeSpan::FromMilliSeconds(1000)
 */
nn::Result SetAudioOutsProcessMasterVolume(nn::applet::AppletResourceUserId id, float volume, nn::TimeSpan fadeTime) NN_NOEXCEPT;

/**
 * @brief       指定した applet に紐づくすべての AudioOut インスタンスの録音ボリュームを取得します。
 * @param[out]  pOutVolume              ボリューム
 * @param[in]   id                      applet 識別子
 * @retresult
 * @handleresult{nn::audio::ResultAppletResourceUserIdNotFound}
 * @endretresult
 * @pre
 * - pOutVolume != nullptr
 */
nn::Result GetAudioOutsProcessRecordVolume(float* pOutVolume, nn::applet::AppletResourceUserId id) NN_NOEXCEPT;

/**
 * @brief       指定した applet に紐づくすべての AudioOut インスタンスの録音ボリュームを設定します。
 * @param[in]   id                      applet 識別子
 * @param[in]   volume                  ボリューム
 * @param[in]   fadeTime                指定したボリュームに変化するまでのフェード時間
 * @retresult
 * @handleresult{nn::audio::ResultAppletResourceUserIdNotFound}
 * @endretresult
 * @pre
 * - nn::audio::ProcessRecordVolumeMin <= volume && volume <= nn::audio::ProcessRecordVolumeMax
 * - nn::TimeSpan::FromMilliSeconds(0) <= fadeTime && fadeTime <= nn::TimeSpan::FromMilliSeconds(1000)
 */
nn::Result SetAudioOutsProcessRecordVolume(nn::applet::AppletResourceUserId id, float volume, nn::TimeSpan fadeTime) NN_NOEXCEPT;

/**
 * @brief       指定した applet に紐づくすべての AudioRenderer インスタンスを一時停止します。
 * @param[in]   id                      applet 識別子
 * @param[in]   fadeTime                一時停止までのフェード時間
 * @retresult
 * @handleresult{nn::audio::ResultAppletResourceUserIdNotFound}
 * @endretresult
 * @pre
 * - nn::TimeSpan::FromMilliSeconds(0) <= fadeTime && fadeTime <= nn::TimeSpan::FromMilliSeconds(10)
 * @detail
 * 本関数は applet 遷移目的で呼ばれることを想定したものです。それ以外の用途に利用することはできません。
 *
 * 本関数は再帰的に呼び出すことはできません。
 */
nn::Result RequestSuspendAudioRenderers(nn::applet::AppletResourceUserId id, nn::TimeSpan fadeTime) NN_NOEXCEPT;

/**
 * @brief       指定した applet に紐づくすべての AudioRenderer インスタンスを再開します。
 * @param[in]   id                      applet 識別子
 * @param[in]   fadeTime                再開までのフェード時間
 * @retresult
 * @handleresult{nn::audio::ResultAppletResourceUserIdNotFound}
 * @endretresult
 * @pre
 * - nn::TimeSpan::FromMilliSeconds(0) <= fadeTime && fadeTime <= nn::TimeSpan::FromMilliSeconds(10)
 * @detail
 * 本関数は applet 遷移目的で呼ばれることを想定したものです。それ以外の用途に利用することはできません。
 */
nn::Result RequestResumeAudioRenderers(nn::applet::AppletResourceUserId id, nn::TimeSpan fadeTime) NN_NOEXCEPT;

/**
 * @brief       指定した applet に紐づくすべての AudioRenderer インスタンスのマスターボリュームを取得します。
 * @param[out]  pOutVolume              ボリューム
 * @param[in]   id                      applet 識別子
 * @retresult
 * @handleresult{nn::audio::ResultAppletResourceUserIdNotFound}
 * @endretresult
 * @pre
 * - pOutVolume != nullptr
 */
nn::Result GetAudioRenderersProcessMasterVolume(float* pOutVolume, nn::applet::AppletResourceUserId id) NN_NOEXCEPT;

/**
 * @brief       指定した applet に紐づくすべての AudioRenderer インスタンスのマスターボリュームを設定します。
 * @param[in]   id                      applet 識別子
 * @param[in]   volume                  ボリューム
 * @param[in]   fadeTime                指定したボリュームに変化するまでのフェード時間
 * @retresult
 * @handleresult{nn::audio::ResultAppletResourceUserIdNotFound}
 * @endretresult
 * @pre
 * - nn::audio::ProcessMasterVolumeMin <= volume && volume <= nn::audio::ProcessMasterVolumeMax
 * - nn::TimeSpan::FromMilliSeconds(0) <= fadeTime && fadeTime <= nn::TimeSpan::FromMilliSeconds(1000)
 */
nn::Result SetAudioRenderersProcessMasterVolume(nn::applet::AppletResourceUserId id, float volume, nn::TimeSpan fadeTime) NN_NOEXCEPT;

/**
 * @brief       指定した applet に紐づくすべての AudioRenderer インスタンスの録音ボリュームを取得します。
 * @param[out]  pOutVolume              ボリューム
 * @param[in]   id                      applet 識別子
 * @retresult
 * @handleresult{nn::audio::ResultAppletResourceUserIdNotFound}
 * @endretresult
 * @pre
 * - pOutVolume != nullptr
 */
nn::Result GetAudioRenderersProcessRecordVolume(float* pOutVolume, nn::applet::AppletResourceUserId id) NN_NOEXCEPT;

/**
 * @brief       指定した applet に紐づくすべての AudioRenderer インスタンスの録音ボリュームを設定します。
 * @param[in]   id                      applet 識別子
 * @param[in]   volume                  ボリューム
 * @param[in]   fadeTime                指定したボリュームに変化するまでのフェード時間
 * @retresult
 * @handleresult{nn::audio::ResultAppletResourceUserIdNotFound}
 * @endretresult
 * @pre
 * - nn::audio::ProcessRecordVolumeMin <= volume && volume <= nn::audio::ProcessRecordVolumeMax
 * - nn::TimeSpan::FromMilliSeconds(0) <= fadeTime && fadeTime <= nn::TimeSpan::FromMilliSeconds(1000)
 */
nn::Result SetAudioRenderersProcessRecordVolume(nn::applet::AppletResourceUserId id, float volume, nn::TimeSpan fadeTime) NN_NOEXCEPT;

/**
 * @brief       インスタンスを操作するための applet 識別子を登録します。
 * @param[in]   id                      applet 識別子
 * @retresult
 * @handleresult{nn::audio::ResultMaxAppletResourceUserIdRegistered}
 * @endretresult
 * @details      applet 識別子毎の本関数の呼び出し回数はライブラリ内部でカウントされます。
 */
nn::Result RegisterAppletResourceUserId(nn::applet::AppletResourceUserId id) NN_NOEXCEPT;

/**
 * @brief       インスタンスを操作するための applet 識別子の登録を解除します。
 * @param[in]   id                      applet 識別子
 * @retresult
 * @handleresult{nn::audio::ResultAppletResourceUserIdNotFound}
 * @endretresult
 * @details     applet 識別子毎の RegisterAppletResourceUserId の呼び出し回数と同じ回数分、
 *              本関数が呼ばれた場合にのみ、実際に登録が解除されます。
 */
nn::Result UnregisterAppletResourceUserId(nn::applet::AppletResourceUserId id) NN_NOEXCEPT;

}  // namespace audio
}  // namespace nn
