﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/arp/arp_Result.h>
#include <nn/arp/detail/arp_PropertyTable.h>

#include <nn/result/result_HandlingUtility.h>

namespace nn {
namespace arp {
namespace detail {

template <typename PropertyTableType>
class RegistrarImpl
{
    NN_DISALLOW_COPY(RegistrarImpl);

private:
    PropertyTableType& m_Table;
    Property* m_pProperty;

    void Release() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_pProperty != nullptr);
        m_Table.Release(m_pProperty);
        m_pProperty = nullptr;
    }

public:
    explicit RegistrarImpl(PropertyTableType& table) NN_NOEXCEPT
        : m_Table(table)
        , m_pProperty(nullptr)
    {
    }
    ~RegistrarImpl() NN_NOEXCEPT
    {
        if (m_pProperty != nullptr)
        {
            Release();
            NN_SDK_ASSERT(m_pProperty == nullptr);
        }
    }
    Result Initialize() NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(m_pProperty == nullptr, ResultInvalidProtocolAccess());
        m_pProperty = m_Table.Reserve();
        NN_RESULT_THROW_UNLESS(m_pProperty != nullptr, ResultRegistryFull());
        NN_RESULT_SUCCESS;
    }
    nn::Result Issue(nn::os::ProcessId pid) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(pid != os::ProcessId::GetInvalidId(), ResultInvalidProcessId());
        NN_RESULT_THROW_UNLESS(m_pProperty != nullptr, ResultInvalidProtocolAccess());
        m_Table.Issue(pid, m_pProperty);
        m_pProperty = nullptr;
        NN_RESULT_SUCCESS;
    }
    nn::Result SetApplicationLaunchProperty(const nn::arp::ApplicationLaunchProperty& launchProp) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(m_pProperty != nullptr, ResultInvalidProtocolAccess());
        m_pProperty->launchProp = launchProp;
        NN_RESULT_SUCCESS;
    }
    nn::Result SetApplicationControlProperty(const nn::ns::ApplicationControlProperty& ctrlProp) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(m_pProperty != nullptr, ResultInvalidProtocolAccess());
        m_pProperty->ctrlProp = ctrlProp;
        NN_RESULT_SUCCESS;
    }
};

} // ~namespace nn::arp::detail
}
}
