﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/arp/arp_Types.h>

#include <utility>

#include <nn/nn_ApplicationId.h>
#include <nn/ncm/ncm_ContentMetaId.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_Types.h>

namespace nn {
namespace arp {
namespace detail {

struct Property
{
    ApplicationLaunchProperty launchProp;
    ns::ApplicationControlProperty ctrlProp;
};

template <size_t TableSize>
class PropertyTable
{
private:
    mutable os::Mutex m_Lock;

    struct Row
    {
        enum State
        {
            State_Empty,
            State_Reserved,
            State_InUse,
        } state;
        os::ProcessId pid;
        Property prop;

        Row* Reserve() NN_NOEXCEPT;
        void Set(const os::ProcessId& pid) NN_NOEXCEPT;
        void Reset() NN_NOEXCEPT;
    } m_Table[TableSize];

    static const Row InitialStateOfRow;

public:
    PropertyTable() NN_NOEXCEPT;

    Property* Reserve() NN_NOEXCEPT;
    void Release(const Property* pProp) NN_NOEXCEPT;
    void Issue(const os::ProcessId& pid, const Property* pProp) NN_NOEXCEPT;
    void Delete(const os::ProcessId& pid) NN_NOEXCEPT;

    bool GetApplicationLaunchProperty(ApplicationLaunchProperty* pOut, const os::ProcessId& pid) const NN_NOEXCEPT;
    bool GetApplicationLaunchProperty(ApplicationLaunchProperty* pOut, const ApplicationId& applicationId) const NN_NOEXCEPT;
    bool GetApplicationControlProperty(ns::ApplicationControlProperty* pOut, const os::ProcessId& pid) const NN_NOEXCEPT;
    bool GetApplicationControlProperty(ns::ApplicationControlProperty* pOut, const ApplicationId& applicationId) const NN_NOEXCEPT;

    void Clear() NN_NOEXCEPT;
};

} // ~namespace nn::arp::detail
}
}

/* --------------------------------------------------------------------------------------------
    実装
 */

#include <nn/arp/detail/arp_LogUtil.h>

#include <mutex>

namespace nn {
namespace arp {
namespace detail {

// Row --------------------------------------------------------

template <size_t TableSize>
const typename PropertyTable<TableSize>::Row PropertyTable<TableSize>::InitialStateOfRow = { PropertyTable::Row::State_Empty, {}, {} };

template <size_t TableSize>
typename PropertyTable<TableSize>::Row* PropertyTable<TableSize>::Row::Reserve() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(state == State_Empty);
    state = State_Reserved;
    return this;
}
template <size_t TableSize>
void PropertyTable<TableSize>::Row::Set(const os::ProcessId& given) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(state == State_Reserved);
    NN_SDK_REQUIRES(given != os::ProcessId::GetInvalidId());
    pid = given;
    state = State_InUse;
}
template <size_t TableSize>
void PropertyTable<TableSize>::Row::Reset() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(state == State_Reserved || state == State_InUse);
    *this = InitialStateOfRow;
}

// Table --------------------------------------------------------

template <size_t TableSize>
PropertyTable<TableSize>::PropertyTable() NN_NOEXCEPT
    : m_Lock(false)
{
    Clear();
}
template <size_t TableSize>
void PropertyTable<TableSize>::Clear() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(m_Lock);
    for (auto& row : m_Table)
    {
        row = InitialStateOfRow;
    }
}
template <size_t TableSize>
Property* PropertyTable<TableSize>::Reserve() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(m_Lock);
    for (auto& row : m_Table)
    {
        if (row.state == Row::State_Empty)
        {
            return &row.Reserve()->prop;
        }
    }
    return nullptr;
}
template <size_t TableSize>
void PropertyTable<TableSize>::Release(const Property* pProp) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(m_Lock);
    for (auto& row : m_Table)
    {
        if (row.state == Row::State_Reserved && &row.prop == pProp)
        {
            row.Reset();
            return;
        }
    }
    NN_SDK_ASSERT(NN_STATIC_CONDITION(false), "[nn::arp] Unreachable");
}
template <size_t TableSize>
void PropertyTable<TableSize>::Issue(const os::ProcessId& pid, const Property* pProp) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(pid != os::ProcessId::GetInvalidId());
    std::lock_guard<os::Mutex> lock(m_Lock);
    for (auto& row : m_Table)
    {
        if (row.state == Row::State_Reserved && &row.prop == pProp)
        {
            row.Set(pid);
            LogRegistration(pid, row.prop.launchProp, row.prop.ctrlProp);
            return;
        }
    }
    NN_ABORT("[nn::arp] Unreachable");
}
template <size_t TableSize>
void PropertyTable<TableSize>::Delete(const os::ProcessId& pid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(pid != os::ProcessId::GetInvalidId());
    std::lock_guard<os::Mutex> lock(m_Lock);
    for (auto& row : m_Table)
    {
        if (row.state == Row::State_InUse && row.pid == pid)
        {
            row.Reset();
            return;
        }
    }
}
template <size_t TableSize>
bool PropertyTable<TableSize>::GetApplicationLaunchProperty(ApplicationLaunchProperty* pOut, const os::ProcessId& pid) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(pid != os::ProcessId::GetInvalidId());
    std::lock_guard<os::Mutex> lock(m_Lock);
    for (auto& row : m_Table)
    {
        if (row.state == Row::State_InUse && row.pid == pid)
        {
            *pOut = row.prop.launchProp;
            return true;
        }
    }
    return false;
}
template <size_t TableSize>
bool PropertyTable<TableSize>::GetApplicationLaunchProperty(ApplicationLaunchProperty* pOut, const ApplicationId& applicationId) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(applicationId != ApplicationId::GetInvalidId());
    nn::ncm::ApplicationId ncmApplicationId = { applicationId.value };
    std::lock_guard<os::Mutex> lock(m_Lock);
    for (auto& row : m_Table)
    {
        if (row.state == Row::State_InUse && row.prop.launchProp.id == ncmApplicationId)
        {
            *pOut = row.prop.launchProp;
            return true;
        }
    }
    return false;
}
template <size_t TableSize>
bool PropertyTable<TableSize>::GetApplicationControlProperty(ns::ApplicationControlProperty* pOut, const os::ProcessId& pid) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(pid != os::ProcessId::GetInvalidId());
    std::lock_guard<os::Mutex> lock(m_Lock);
    for (auto& row : m_Table)
    {
        if (row.state == Row::State_InUse && row.pid == pid)
        {
            *pOut = row.prop.ctrlProp;
            return true;
        }
    }
    return false;
}
template <size_t TableSize>
bool PropertyTable<TableSize>::GetApplicationControlProperty(ns::ApplicationControlProperty* pOut, const ApplicationId& applicationId) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(applicationId != ApplicationId::GetInvalidId());
    nn::ncm::ApplicationId ncmApplicationId = { applicationId.value };
    std::lock_guard<os::Mutex> lock(m_Lock);
    for (auto& row : m_Table)
    {
        if (row.state == Row::State_InUse && row.prop.launchProp.id == ncmApplicationId)
        {
            *pOut = row.prop.ctrlProp;
            return true;
        }
    }
    return false;
}

} // ~namespace nn::arp::detail
}
}
