﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/arp/detail/arp_InternalTypes.h>
#include <nn/arp/detail/arp_PropertyTable.h>
#include <nn/arp/detail/arp_ServiceImpl.h>

#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/sf/sf_ExpHeapAllocator.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/sf/sf_ISharedObject.h>

namespace nn {
namespace arp {

template <size_t HeapSize, size_t PropertyTableSize>
class ServiceResource
{
private:
    // オブジェクトの動的生成用のアロケータ
    template <size_t S>
    class Allocator
        : public sf::ExpHeapAllocator
    {
    private:
        typename std::aligned_storage<S>::type m_Buffer;
        lmem::HeapHandle m_HeapHandle;

    public:
        typedef sf::ExpHeapAllocator::Policy Policy;

        Allocator() NN_NOEXCEPT
        {
            m_HeapHandle = lmem::CreateExpHeap(&m_Buffer, sizeof(m_Buffer), 0);
            this->Attach(m_HeapHandle);
        }
        ~Allocator() NN_NOEXCEPT
        {
            lmem::DestroyExpHeap(m_HeapHandle);
        }
    };

    // オブジェクトの動的生成用アロケータ
    typedef Allocator<HeapSize> AllocatorType;
    AllocatorType m_ObjectAllocator;

    // オブジェクトの動的生成ファクトリ
    typedef sf::ObjectFactory<typename AllocatorType::Policy> Factory;

    // アプリケーションプロセスとその性質のマッピングを保持するためのテーブル
    detail::PropertyTable<PropertyTableSize> m_PropertyTable;

    // サービス実装
    detail::ServiceImpl<AllocatorType, PropertyTableSize> m_ServiceImpl;

public:
    ServiceResource() NN_NOEXCEPT
        : m_ServiceImpl(m_ObjectAllocator, m_PropertyTable)
    {
    }
    sf::SharedPointer<detail::IWriter> GetWriterPtr() NN_NOEXCEPT
    {
        auto p = Factory::template CreateSharedWithoutManagement<detail::IWriter>(&m_ObjectAllocator, &m_ServiceImpl);
        NN_ABORT_UNLESS(p, "[nn::arp] Failed to create sf::SharedPointer<detail::IWriter> for GetWriterPtr()\n");
        return p;
    }
    sf::SharedPointer<detail::IReader> GetReaderPtr() NN_NOEXCEPT
    {
        auto p = Factory::template CreateSharedWithoutManagement<detail::IReader>(&m_ObjectAllocator, &m_ServiceImpl);
        NN_ABORT_UNLESS(p, "[nn::arp] Failed to create sf::SharedPointer<detail::IReader> for GetReaderPtr()\n");
        return p;
    }
};

typedef ServiceResource<detail::FactoryHeapSize, detail::ApplicationProcessCountMax> DefaultServiceResource;

}
}
