﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/applet/applet_Types.h>

namespace nn { namespace applet {

/**
    @brief ストレージハンドル型

    @details
     内部フィールドに直接アクセスしないでください。
*/
struct StorageHandle
{
    void* _p;

    friend bool operator==(const StorageHandle& lhs, const StorageHandle& rhs) NN_NOEXCEPT
    {
        return lhs._p == rhs._p;
    }

    friend bool operator!=(const StorageHandle& lhs, const StorageHandle& rhs) NN_NOEXCEPT
    {
        return !(lhs == rhs);
    }
};

/**
    @brief 無効なストレージハンドル定数
*/
const StorageHandle InvalidStorageHandle = {};

//! @name ライブラリアプレットのデータ渡しに使用するストレージ操作 API
//! @{

/**
    @brief ライブラリアプレットの引数などの受け渡しに使用できるストレージを作成します。

    @retresult
    @endretresult

    @param[out] pOut 作成されたストレージを表すストレージハンドルを格納するバッファポインタを指定します。
    @param[in] size ストレージのサイズを指定します。

    @details
     size で指定されたサイズのストレージをシステム内のメモリを使用して作成し *pOut に格納します。
     作成されたストレージ内のメモリは、ゼロ初期化されています。

     作成されたストレージに対し、必要に応じて WriteToStorage() などを呼んでデータを書き込み、
     applet ライブラリもしくは ae ライブラリの、

     - Push 系関数
     - Unpop 系関数

     に渡すことができます。
     これらに渡したストレージは、所有権が移り、以後アクセスできなくなります。

     上記関数に渡さなかったストレージで不要になったものは ReleaseStorage() に渡し、解放する必要があります。

     本 API で作成されるストレージの個数やサイズは、今後制限が行われる予定です。
     現時点では以下を満たすようにしてください。

     - size <= 1024 * 8
     - 一段のライブラリアプレット呼び出しにつき、同時に存在する数が 4 個以下
     - 一段のライブラリアプレット呼び出しにつき、同時に存在するサイズ合計が、1024 単位で 32 以下

     ニュースからのアプリ起動時の起動パラメータなど、特定の機能仕様において
     これ以上のサイズを必要とするケースにおいては別途ご相談ください。

     なお、CreateStorage() で作成されたストレージが(渡された先での呼び出しも含め) ReleaseStorage() されるまでの期間を「存在する」とします。
*/
Result CreateStorage(StorageHandle* pOut, size_t size) NN_NOEXCEPT;

/**
    @brief メモリ共有の仕組みを使用して大容量ストレージを作成します。

    @retresult
    @endretresult

    @param[out] pOut 作成されたストレージを表すストレージハンドルを格納するバッファポインタを指定します。
    @param[in] buffer データメモリアドレスを指定します。
    @param[in] size データサイズを指定します。
    @param[in] isWritable 書き込み可能であるかどうかを指定します。

    @pre buffer と size が nn::os::MemoryPageSize でアラインされている
    @pre [buffer, buffer + size) の領域が読み書き可能

    @details
     サイズ size を持ち先頭が buffer であるメモリ領域を使って、大容量ストレージを作成し、*pOut に受け取ります。
     メモリ領域に書き込まれていたデータはそのまま維持されます。

     isWritable = false で作成されたストレージは読み込み専用となり、 WriteToStorage() などを呼ぶことはできません。
     このため、この関数を呼ぶ前に buffer で示される領域に必要なデータを書き込んでおく必要があります。

     この関数に渡したメモリは、作成されたストレージが解放されるまでの間、直接のアクセスはできません。

     この関数で作成されたストレージは、作成したアプレット(またはアプリケーション)よりも呼び出し側に
     Push しないようにしてください。
     すなわち、ライブラリアプレットで大きなデータを返したいような場合は以下のような手順を経る必要があります。

     - 呼び出し元で本関数を isWritable = true で呼んで大容量ストレージを作成する
     - InChannel 経由で(データが空の)大容量ストレージを呼び出し先ライブラリアプレットに渡す
     - 呼び出し先ライブラリアプレットで、大容量ストレージにデータを書き込む
     - OutChannel 経由で呼び出し元に大容量ストレージを返却する

     Push などの挙動や解放責任に関しては CreateStorage() と同様です。
     解放する必要がある場合には ReleaseStorage() を呼んでください。

     本 API で作成されるストレージの個数やサイズは、今後制限が行われる予定です。
     一段のライブラリアプレット呼び出しにつき、同時に存在する数が 2 個以下になるようにしてください。
*/
Result CreateLargeStorage(StorageHandle* pOut, void* buffer, size_t size, bool isWritable) NN_NOEXCEPT;

/**
    @brief メモリ共有の仕組みを使用して転送ストレージを作成します。

    @retresult
    @endretresult

    @param[out] pOut 作成されたストレージを表すストレージハンドルを格納するバッファポインタを指定します。
    @param[in] buffer データメモリアドレスを指定します。
    @param[in] size データサイズを指定します。

    @pre buffer と size が nn::os::MemoryPageSize でアラインされている
    @pre [buffer, buffer + size) の領域が読み書き可能

    @details
     サイズ size を持ち先頭が buffer であるメモリ領域を使って、転送ストレージを作成し、*pOut に受け取ります。
     メモリ領域に書き込まれていたデータはそのまま維持されます。

     転送ストレージは、呼出先のライブラリアプレットで nn::ae::MapTransferStorage() を使ってメモリ空間上にマップして利用します。
     nn::applet::ReadFromStorage() や nn::applet::WriteToStorage() は使用できません。

     この関数に渡したメモリは、作成されたストレージが解放されるまでの間、直接のアクセスはできません。
     この関数で作成されたストレージは、作成したアプレット(またはアプリケーション)よりも呼び出し側に Push しないようにしてください。

     Push などの挙動や解放責任に関しては CreateStorage() と同様です。
     解放する必要がある場合には ReleaseStorage() を呼んでください。

     本 API で作成されるストレージの個数やサイズは、今後制限が行われる予定です。
     一段のライブラリアプレット呼び出しにつき、同時に存在する数が 2 個以下になるようにしてください。
*/
Result CreateTransferStorage(StorageHandle* pOut, void* buffer, size_t size) NN_NOEXCEPT;

/**
    @brief ストレージを解放します。

    @param[in] handle 対象となるストレージのハンドルを指定します。

    @pre handle が有効でまだ解放されていないストレージのハンドルである。

    @details
     handle で指定されたストレージを解放します。
     以後 handle へのアクセスはできなくなります。
*/
void ReleaseStorage(StorageHandle handle) NN_NOEXCEPT;

/**
    @brief ストレージのサイズを取得します。

    @param[in] handle 対象となるストレージのハンドルを指定します。

    @pre handle が CreateStorage() もしくは CreateLargeStorage() で生成されたものである。
    @pre handle が有効でまだ解放されていないストレージのハンドルである。

    @return ストレージのサイズを返します。

    @details
     handle で指定されたストレージのサイズを取得します。
*/
size_t GetStorageSize(StorageHandle handle) NN_NOEXCEPT;

/**
    @brief ストレージにデータを書き込みます。

    @param[in] handle 対象となるストレージのハンドルを指定します。
    @param[in] offset 書き込む位置を指定します。
    @param[in] buffer 書き込むバッファを指定します。
    @param[in] size 書き込むサイズを指定します。

    @retresult
        @handleresult{nn::applet::ResultNotMemoryStorage}
    @endretresult

    @pre 0 <= offset
    @pre offset + size <= GetStorageSize(handle)
    @pre [buffer, buffer + size) の領域が読み込み可能

    @details
     handle で指定されたストレージの offset バイト目からの領域に、
     buffer を先頭とする size バイトのデータを書き込みます。
*/
Result WriteToStorage(StorageHandle handle, int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT;

/**
    @brief ストレージからデータを読み込みます。

    @param[in] handle 対象となるストレージのハンドルを指定します。
    @param[in] offset 読み込む位置を指定します。
    @param[out] buffer 読み込むバッファを指定します。
    @param[in] size 読み込むサイズを指定します。

    @retresult
        @handleresult{nn::applet::ResultNotMemoryStorage}
    @endretresult

    @pre 0 <= offset
    @pre offset + size <= GetStorageSize(handle)
    @pre [buffer, buffer + size) の領域が書き込み可能

    @details
     handle で指定されたストレージの offset バイト目からの size バイトのデータを、
     buffer を先頭とするメモリに読み出します。
*/
Result ReadFromStorage(StorageHandle handle, int64_t offset, void* buffer, size_t size) NN_NOEXCEPT;

//! @}

}}
