﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   無操作状態の制御に関する API の宣言
 *
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_TimeSpan.h>
#include <nn/applet/applet_Types.h>

namespace nn { namespace applet {

//--------------------------------------------------------------------------
/**
 * @brief   無操作状態検知における検知対象入力を示す列挙型です。
 *
 * @details
 *  デフォルトでは InputSource_All で、全ての入力が検知対象です。
 *
 */
enum InputSource
{
    InputSource_Pad                  = 1u << 0,     //!< Pad (ボタン, アナログスティックボタン) 入力を検知対象とする
    InputSource_Sensor               = 1u << 1,     //!< センサー (加速度, 角速度)入力を検知対象とする
    InputSource_DebugPad             = 1u << 2,     //!< デバッグコントローラー入力を検知対象とする
    InputSource_TouchScreen          = 1u << 3,     //!< タッチスクリーン入力を検知対象とする
    InputSource_Keyboard             = 1u << 4,     //!< キーボード入力を検知対象とする
    InputSource_Mouse                = 1u << 5,     //!< マウス入力を検知対象とする
    InputSource_UsbPort              = 1u << 6,     //!< USB 接続端子を検知対象とする
    InputSource_SdCardSlot           = 1u << 7,     //!< SD カードスロットを検知対象とする
    InputSource_GameCardSlot         = 1u << 8,     //!< ゲームカードスロットを検知対象とする
    InputSource_PadConnection        = 1u << 9,     //!< Pad 接続を検知対象とする
    InputSource_AnalogStick          = 1u << 10,    //!< アナログスティック入力を検知対象とする
    InputSource_ConsoleSixAxisSensor = 1u << 11,    //!< 本体 6 軸センサー入力を検知対象とする
    InputSource_All                  = 0xffffffff   //!< 全入力を検知対象とする
};


//--------------------------------------------------------------------------

//! @name   無操作状態の制御に関する API
//! @{

//--------------------------------------------------------------------------
/**
 * @brief   無操作状態検出時の自動スリープや自動低輝度化の発動時間を指定します。
 *
 * @param[in]  autoSleepTimeInHandheld 携帯モードでの自動スリープ発動時間（秒）
 * @param[in]  autoSleepTimeInConsole  据置モードでの自動スリープ発動時間（秒）
 * @param[in]  dimmingTimeInHandheld   携帯モードでの自動低輝度化発動時間（秒）
 * @param[in]  dimmingTimeInConsole    据置モードでの自動低輝度化発動時間（秒）
 *
 * @pre
 *  - autoSleepTimeInHandheld >= 0
 *  - autoSleepTimeInConsole >= 0
 *  - dimmingTimeInHandheld >= 0
 *  - dimmingTimeInConsole >= 0
 *
 * @details
 *  無操作状態を検出した際に働く自動スリープや画面焼け軽減のための
 *  自動低輝度化機能の発動時間をまとめて上書き設定します。
 *
 *  単位は全て秒で指定し、0 を指定した場合は上書き設定を解除します。
 *
 *  自アプレットがオーバーレイアプレットの場合、本機能は設定値の登録のみを
 *  行ない、実際の適用は別途 nn::ae::EnableAutoSleepTimeAndDimmingTime() を
 *  発行するまで行われません。
 *
 */
void OverrideAutoSleepTimeAndDimmingTime(int autoSleepTimeInHandheld, int autoSleepTimeInConsole, int dimmingTimeInHandheld, int dimmingTimeInConsole) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 * @brief   メディア再生中か否かをシステムに通知します。
 *
 * @param[in]  isInMediaPlayback    メディア再生中か否かを bool で指定
 *
 * @details
 *  自プログラム上でメディア再生中か否かをシステムに通知します。
 *
 *  システムには、無操作状態を検知したときに自動スリープや画面の低輝度化を
 *  発動する機能がありますが、メディア再生中は通常とは異なったポリシーで
 *  これらの機能が発動するようになっています。
 *  本 API はそのメディア再生中か否かをシステムに通知するための機能です。
 *
 *  プログラム起動直後のデフォルトでは、false が設定されています。
 *
 *  なお、nn::oe::BeginMediaPlaybackSection() を使ってアプリケーション自身が
 *  メディア再生区間を独自に制御している場合、いずれか一つの API でも
 *  メディア再生中であればそのポリシーが適用され、全ての API でメディア再生を
 *  解除した時点で通常のポリシーが適用されるようになります。
 *
 */
void SetMediaPlaybackState(bool isInMediaPlayback) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 * @brief   無操作状態検出機能における検知対象入力を指定します。
 *
 * @param[in] inputSources  検知対象とする入力デバイス
 *
 * @pre
 *  - inputSources != 0
 *
 * @details
 *  無操作状態検出機能において、その検知対象とする入力デバイスを指定します。
 *  入力デバイスは、nn::applet::InputSource の列挙子を複数論理和したものを
 *  指定することができます。
 *
 *  デフォルトでは nn::applet::InputSource_All となっています。
 *
 */
void SetInputDetectionSourceSet(Bit32 inputSources) NN_NOEXCEPT;

//! @}

//! @name   自動スリープの無効化に関する API
//! @{

//--------------------------------------------------------------------------
/**
 * @brief   無操作状態時の自動スリープを無効化します。
 *
 * @details
 *  無操作状態時の自動スリープを無効化します。 @n
 *  @n
 *  本関数は、無操作状態時の画面焼け軽減機能の動作には影響しません。 @n
 *  @n
 *  本関数を呼び出された回数は内部でカウントされています。本関数を呼び出した回数と同じ回数だけ
 *  @ref nn::applet::EndAutoSleepDisabledSection() が呼び出されたとき、初めて自動スリープ無効化が解除されます。 @n
 *  @n
 *  本機能の効果は、本関数を呼び出したアプリケーションにフォーカスがある間のみ有効です。
 */
void BeginAutoSleepDisabledSection() NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
 * @brief   無操作状態時の自動スリープの無効化を解除します。
 *
 * @details
 *  無操作状態時の自動スリープの無効化を解除します。 @n
 *  @n
 *  本関数を呼び出された回数は内部でカウントされています。 @ref nn::applet::BeginAutoSleepDisabledSection() を
 *  呼び出した回数と同じ回数だけ本関数が呼び出されたとき、初めて自動スリープ無効化が解除されます。
 */
void EndAutoSleepDisabledSection() NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
 * @brief   無操作状態時の自動スリープが現在無効化されているかどうかを取得します。
 *
 * @return  自動スリープが現在無効化されているかどうかが返ります。
 *
 * @details
 *  無操作状態時の自動スリープが @ref nn::applet::BeginAutoSleepDisabledSection() の呼び出しによって無効化されているかどうかを取得します。
 */
bool IsAutoSleepDisabled() NN_NOEXCEPT;

//! @}

//-----------------------------------------------------------------------------

}} // namespace nn::applet

