﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   アプレット開発者向けの型や定数の宣言
 *
 */

#pragma once

#include <nn/nn_Common.h>

namespace nn { namespace applet {

//-----------------------------------------------------------------------------
/**
 * @brief   アプレットリソースのユーザを識別するための ID を示す型です。
 *
 */
struct AppletResourceUserId
{
    Bit64   lower;

    /**------------------------------------------------------------------
     * @brief  常に無効な AppletResourceUserId を返します。
     */
    static AppletResourceUserId GetInvalidId() NN_NOEXCEPT
    {
        AppletResourceUserId id = { 0 };
        return id;
    }

    /**------------------------------------------------------------------
     * @brief  AppletResourceUserId 同士が一致しているか否かを返します。
     */
    friend bool operator==(const AppletResourceUserId& lhs,
                           const AppletResourceUserId& rhs) NN_NOEXCEPT
    {
        return lhs.lower == rhs.lower;
    }

    /**------------------------------------------------------------------
     * @brief  AppletResourceUserId 同士が不一致か否かを返します。
     */
    friend bool operator!=(const AppletResourceUserId& lhs,
                           const AppletResourceUserId& rhs) NN_NOEXCEPT
    {
        return !(lhs == rhs);
    }
};


//-----------------------------------------------------------------------------
/**
 * @brief   各種アプレットの ID を定義する列挙体です。
 *
 * @details
 *  各種アプレット機能を示す ID 番号です。@n
 *  ライブラリアプレットを起動する場合などに使用します。
 *
 */
enum AppletId
{
    AppletId_None                       = 0,  //!< 特定のアプレットを指しません
    AppletId_Application                = 1,  //!< アプリケーション

    AppletId_OverlayApplet              = 2,  //!< オーバレイ表示アプレット
    AppletId_SystemAppletMenu           = 3,  //!< システムアプレットメニュー

    AppletId_SystemApplication          = 4,  //!< システムアプリケーション

    AppletId_LibraryAppletAuth          = 10, //!< 簡易認証アプレット
    AppletId_LibraryAppletCabinet       = 11, //!< amiibo 設定アプレット
    AppletId_LibraryAppletController    = 12, //!< コントローラーアプレット
    AppletId_LibraryAppletDataErase     = 13, //!< データ削除アプレット
    AppletId_LibraryAppletError         = 14, //!< エラービューアアプレット
    AppletId_LibraryAppletNetConnect    = 15, //!< ネット接続設定アプレット
    AppletId_LibraryAppletPlayerSelect  = 16, //!< プレイヤー選択アプレット
    AppletId_LibraryAppletSwkbd         = 17, //!< ソフトウェアキーボードアプレット
    AppletId_LibraryAppletMiiEdit       = 18, //!< Mii 編集アプレット
    AppletId_LibraryAppletWeb           = 19, //!< Web アプレット
    AppletId_LibraryAppletShop          = 20, //!< ショップアプレット
    AppletId_LibraryAppletPhotoViewer   = 21, //!< フォトビューアー
    AppletId_LibraryAppletSet           = 22, //!< 本体設定
    AppletId_LibraryAppletOfflineWeb    = 23, //!< Offline Web アプレット
    AppletId_LibraryAppletLoginShare    = 24, //!< ログインシェアアプレット
    AppletId_LibraryAppletWifiWebAuth   = 25, //!< WiFi WebAuth アプレット

    AppletId_LibraryAppletMyPage        = 26, //!< マイページアプレット

    AppletId_LibraryAppletGift          = 27, //!< おすそわけ通信アプレット
    AppletId_LibraryAppletUserMigration = 28, //!< ユーザ移行アプレット

    AppletId_LibraryAppletEncounter     = 29, //!< すれちがいアプレット
    AppletId_LibraryAppletStory         = 30, //!< セーブデータ同期アプレット
};


//-----------------------------------------------------------------------------
/**
 * @brief   ライブラリアプレットのモードを定義する列挙体です。
 *
 */
enum LibraryAppletMode
{
    LibraryAppletMode_AllForeground = 0, //!< フォアグラウンド全域を使って動作します
    LibraryAppletMode_PartialForeground = 1, //!< 部分的にフォアグラウンドを使って動作します
    LibraryAppletMode_NoUi = 2, //!< UI なしモードで動作します
    LibraryAppletMode_PartialForegroundWithIndirectDisplay = 3, //!< 部分的にフォアグラウンドを使って動作しますが、直接のディスプレイ表示を行いません
};


//-----------------------------------------------------------------------------
/**
 * @brief   各アプレットへの通知メッセージを定義する列挙体です。
 *
 */
enum Message
{
    // 通知メッセージ
    Message_None                            = 0,  //!< 空メッセージ
    Message_InFocus                         = 1,  //!< InFocus への遷移指示
    Message_OutOfFocus                      = 2,  //!< OutOfFocus への遷移指示

    Message_Exit                            = 4,  //!< プロセス終了指示

    Message_ApplicationExited               = 6,  //!< アプリが終了
    Message_ApplicationSuspendedByRightsError = 7,  //!< アプリが権利不足で中断

    Message_FocusStateChanged               = 15, //!< フォーカス状態の変更
    Message_RestartFromSuspend              = 16, //!< 一時中断からの再開

    Message_DetectShortPressingHomeButton   = 20, //!< HOME ボタン短押しを検知
    Message_DetectLongPressingHomeButton    = 21, //!< HOME ボタン長押しを検知
    Message_DetectShortPressingPowerButton  = 22, //!< POWER ボタン短押しを検知
    Message_DetectMiddlePressingPowerButton = 23, //!< POWER ボタン中押しを検知
    Message_DetectLongPressingPowerButton   = 24, //!< POWER ボタン長押しを検知

    // スリープ
    Message_RequestToReleaseSleepLock       = 25, //!< スリープロックのアンロック指示
    Message_FinishedSleepSequence           = 26, //!< スリープシーケンスの終了通知
    Message_SleepRequiredByHighTemperature  = 27, //!< 高温状態によるスリープ遷移推奨通知
    Message_SleepRequiredByLowBattery       = 28, //!< バッテリー不足によるスリープ遷移推奨通知
    Message_AutoPowerDown                   = 29, //!< 一定時間無操作によるスリープ遷移要求

    // 動作モード、性能モードの変更
    Message_OperationModeChanged            = 30, //!< 動作モードの変更通知
    Message_PerformanceModeChanged          = 31, //!< 性能モードの変更通知

    // 通知メッセージ
    Message_DetectReceivingCecSystemStandby = 32, //!< CEC システムスタンバイ受信を検知

    // SD カードが抜かれた
    Message_SdCardRemoved                   = 33, //!< SD カードが抜かれたことの通知

    // コントローラ F/W 更新期間の変更
    Message_ControllerFirmwareUpdateSectionChanged = 34, //!< コントローラ F/W 更新期間の変更通知

    // ライブラリアプレット制御用のメッセージを定義する列挙体です。
    Message_LibraryAppletLaunched           = 40, //!< LA が起動
    Message_LibraryAppletExited             = 41, //!< LA が正常終了
    Message_LibraryAppletExitedAbnormally   = 42, //!< LA が異常終了
    Message_LibraryAppletTerminated         = 43, //!< LA が強制終了
    Message_ReplyMessageArrived             = 44, //!< IPC 返答メッセージが到着

    // アプリケーション起動リクエスト
    Message_LaunchApplicationRequested      = 50, //!< アプリケーションの起動リクエストが存在する

    // 表示リクエスト
    Message_RequestToDisplay                = 51, //!< 表示リクエストが存在する

    // オーバレイへのアプリ起動・終了通知
    Message_ShowApplicationLogo             = 55, //!< オーバーレイ通知へのロゴ表示指示
    Message_HideApplicationLogo             = 56, //!< オーバーレイ通知へのロゴ非表示指示
    Message_ForceHideApplicationLogo        = 57, //!< オーバーレイ通知へのロゴ非表示指示(強制)

    // デバッグ用
    Message_FloatingApplicationDetected     = 60, //!< 外部起動したアプリを検知

    // VR モード関連
    Message_VrModeChanged                   = 70, //!< VR モードの変更通知
    Message_VrModeDisabledBySystem          = 71, //!< VR モードの強制終了
    Message_VrModeCurtainRequired           = 72, //!< VR モードのカーテン要求通知

    // LCD バックライト関連
    Message_LcdBacklightSwitchedOn          = 75, //!< LCD バックライトが点灯

    // システムアプレットへの要求
    Message_RequestToShutdown               = 80, //!< シャットダウン要求
    Message_RequestToReboot                 = 81, //!< 再起動要求
    Message_RequestToGoBackQuestMenu        = 82, //!< アプリ終了および試遊台メニュー復帰要求

    // アプリ向けのメッセージ
    Message_CaptureButtonPressedShortly     = 90, //!< CAPTURE ボタン短押し検知

    // SystemAppletControllerForDebug
    Message_RequestToLaunchApplicationForDebug = 1000, //!< SystemAppletControllerForDebug の RequestLaunch
};


//-----------------------------------------------------------------------------

/**
    @brief 起動時引数の種類を表すタグ型です。
*/
enum LaunchParameterKind
{
    LaunchParameterKind_Invalid = 0,
    LaunchParameterKind_User = 1,
    LaunchParameterKind_Account = 2,
    LaunchParameterKind_News = 3,
};


//-----------------------------------------------------------------------------
/**
 * @brief   スクリーンショット撮影の禁止／許可状態を示す列挙対です。
 *
 */
enum ScreenShotPermission
{
    ScreenShotPermission_Inherit    = 0,
    ScreenShotPermission_Permit     = 1,
    ScreenShotPermission_Forbid     = 2,
};


//-----------------------------------------------------------------------------
/**
 * @brief アプレット間の処理区間／禁止区間制御用の ReaderWriteLock インデックス
 *
 */
enum ReaderWriterLockIndex
{
    ReaderWriterLockIndex_HomeButton = 0,   // HomeButton 処理／禁止区間用
    ReaderWriterLockIndex_Entrance,         // エントランス処理／禁止区間用
    ReaderWriterLockIndex_InterruptScene,   // 割込みシーン処理／禁止区間用
    ReaderWriterLockIndex_Reserved,         // 予備
    ReaderWriterLockIndex_Max               // 配列定義用インデックス最大値
};

// キャプチャバッファインデックス
enum CaptureBufferIndex
{
    CaptureBufferIndex_LastApplication  = 0,
    CaptureBufferIndex_LastForeground   = 1,
    CaptureBufferIndex_CallerApplet     = 2,
};


inline bool CheckAppletResourceUserId(Bit64 processId, AppletResourceUserId aruid) NN_NOEXCEPT
{
    auto value = aruid.lower;
    // TORIAEZU: 0 を通すようにしておく
    if (value == 0)
    {
        return true;
    }
    return value == processId;
}

}} // namespace nn::applet

