﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief 性能モードと性能の定義。
 */

#pragma once

namespace nn { namespace apm {

/**
 * @brief   性能モードを表す列挙型です。
 *
 * @details ひとつもしくは幾つかの性能を選択可能なプラットフォームの内部状態を性能モードと呼称します。
 */
enum PerformanceMode
{
    PerformanceMode_Invalid = -1, //!< 無効な性能モードです。

    /**
     * 通常の性能モードです。@n
     * この性能モードの性能の初期設定は @ref nn::apm::PerformanceConfiguration_Cpu1020MhzGpu307Mhz1331Mhz です。
     */
    PerformanceMode_Normal,

    /**
     * より高い性能を使用できる性能モードです。@n
     * この性能モードの性能の初期設定は @ref nn::apm::PerformanceConfiguration_Cpu1020MhzGpu768Mhz です。
     */
    PerformanceMode_Boost,
};

/**
 * @brief   選択可能な性能を表す列挙型です。
 *
 * @details 今後のリリースで列挙値が変更される予定があります。
 */
enum PerformanceConfiguration
{
    PerformanceConfiguration_Invalid = 0x00000000, //!< 無効な設定です。

    /**
     * CPU 1020.0 MHz, GPU 384.0 MHz, EMC 1600.0 Mhz。@n
     * 全ての性能モードで特権開発者のみ使用可能です。
     */
    PerformanceConfiguration_Cpu1020MhzGpu384Mhz = 0x00010000,

    /**
     * CPU 1020.0 MHz, GPU 768.0 MHz, EMC 1600.0 Mhz。@n
     * @ref nn::apm::PerformanceMode_Boost のみで使用可能です。
     */
    PerformanceConfiguration_Cpu1020MhzGpu768Mhz = 0x00010001,

    /**
     * CPU 1224.0 MHz, GPU 691.2 MHz, EMC 1600.0 Mhz。@n
     * 現在サポートしていません。
     */
    PerformanceConfiguration_Cpu1224MhzGpu691Mhz = 0x00010002,

    /**
     * CPU 1020.0 MHz, GPU 230.4 MHz, EMC 1600.0 Mhz。@n
     * 現在サポートしていません。
     */
    PerformanceConfiguration_Cpu1020MhzGpu230Mhz = 0x00020000,

    /**
     * CPU 1020.0 MHz, GPU 307.2 MHz, EMC 1600.0 Mhz。@n
     * 全ての性能モードで特権開発者のみ使用可能です。
     */
    PerformanceConfiguration_Cpu1020MhzGpu307Mhz = 0x00020001,

    /**
     * CPU 1224.0 MHz, GPU 230.4 MHz, EMC 1600.0 Mhz。@n
     * 現在サポートしていません。
     */
    PerformanceConfiguration_Cpu1224MhzGpu230Mhz = 0x00020002,

    /**
     * CPU 1020.0 MHz, GPU 307.2 MHz, EMC 1331.2 Mhz。@n
     * 全ての性能モードで使用可能です。
     */
    PerformanceConfiguration_Cpu1020MhzGpu307MhzEmc1331Mhz = 0x00020003,

    /**
     * CPU 1020.0 MHz, GPU 384.0 MHz, EMC 1331.2 Mhz。@n
     * 全ての性能モードで特権開発者のみ使用可能です。
     */
    PerformanceConfiguration_Cpu1020MhzGpu384MhzEmc1331Mhz = 0x00020004,

    /**
     * CPU 1020.0 MHz, GPU 307.2 MHz, EMC 1065.6 Mhz。@n
     * 現在サポートしていません。
     */
    PerformanceConfiguration_Cpu1020MhzGpu307MhzEmc1065Mhz = 0x00020005,

    /**
     * CPU 1020.0 MHz, GPU 384.0 MHz, EMC 1065.6 Mhz。@n
     * 現在サポートしていません。
     */
    PerformanceConfiguration_Cpu1020MhzGpu384MhzEmc1065Mhz = 0x00020006,

    /**
     * CPU 1020.0 MHz, GPU 460.8 MHz, EMC 1600.0 Mhz。@n
     * 全ての性能モードで特権開発者のみ使用可能です。
     */
    PerformanceConfiguration_Cpu1020MhzGpu460MhzEmc1600Mhz = 0x92220007,

    /**
     * CPU 1020.0 MHz, GPU 460.8 MHz, EMC 1331.2 Mhz。@n
     * 全ての性能モードで特権開発者のみ使用可能です。
     */
    PerformanceConfiguration_Cpu1020MhzGpu460MhzEmc1331Mhz = 0x92220008,
};

}} // namespace nn::apm
