﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>

#include <nn/am/service/window/am_WindowControllerBase.h>

#include <nn/util/util_Exchange.h>

namespace nn { namespace am { namespace service { namespace window {

enum class SleepSequenceStage : int
{
    Initializing,
    Sleep,
    Finalizing,
};

class SleepWindowController
     : public WindowControllerBase<SleepWindowController>
{
public:

    // Status という名前でデータ構造を定義
    struct Status
    {
        bool lcdOff; // LCD の OFF を要求するかどうか

        uint64_t clockRequestedApprovalForSleep; // 認証リクエストされているスリープクロック(0 無効)
        bool lockingSleep; // ロックしているかどうか
        bool isSleeping;     // スリープ中

        SleepSequenceStage nextSleepSequenceStage;
    };

    // active でないときの Status を初期化する関数
    static void SetNotActive(Status* p) NN_NOEXCEPT
    {
        p->lcdOff = false;
        p->clockRequestedApprovalForSleep = 0;
        p->lockingSleep = false;
        p->isSleeping = false;
        p->nextSleepSequenceStage = SleepSequenceStage::Initializing;
    }

    // これが true を返す場合は Update 系関数は呼ばれない
    static bool AreEqual(const Status& x, const Status& y) NN_NOEXCEPT
    {
        return true
            && x.lcdOff == y.lcdOff
            && x.clockRequestedApprovalForSleep == y.clockRequestedApprovalForSleep
            && x.lockingSleep == y.lockingSleep
            && x.isSleeping == y.isSleeping
            && x.nextSleepSequenceStage == y.nextSleepSequenceStage
        ;
    }

    struct Property
    {
        bool requestsPowerDown;
        bool forcePowerDown;
        uint64_t clockForSleep; // スリープクロック(1 以上が有効)

        bool controlsLcd; // LCD を off にするかどうか (shutter window で true)

        bool needsApprovalForSleep; // スリープに承認が必要かどうか
        uint64_t approvedClockForSleep; // 承認されたスリープクロック

        bool drivesSleepSequence; // スリープを行うかどうか (スリープ処理ウインドウで true)
    };

    struct CalculationContext
    {
        uint64_t currentClockForSleep;
        bool powerDownRequested;
        bool forcePowerDown;
        bool canSleep;
    };

    void InitializeCalculationContext(CalculationContext* pContext) NN_NOEXCEPT
    {
        pContext->currentClockForSleep = 0;
        pContext->powerDownRequested = false;
        pContext->forcePowerDown = false;
        pContext->canSleep = true;
    }

    void CalculateStatus(Status* pOut, const CommonWindowProperty& commonProp, const Property& prop, CalculationContext* pContext) NN_NOEXCEPT
    {
        NN_UNUSED(commonProp);

        // リクエストコンテキストの設定
        if (prop.requestsPowerDown)
        {
            // スリープ開始ウインドウでスリープリクエストが出ていたらコンテキストに流す
            NN_SDK_ASSERT(!pContext->powerDownRequested); // パワーダウン要求は最大ひとつを仮定
            pContext->powerDownRequested = true;
            pContext->currentClockForSleep = prop.clockForSleep;
            if (prop.forcePowerDown)
            {
                pContext->forcePowerDown = true;
            }
        }

        // LCD 制御
        pOut->lcdOff = prop.controlsLcd;

        if (pContext->powerDownRequested)
        {
            // 通常ウインドウの制御
            bool approved;
            if (pContext->forcePowerDown)
            {
                approved = true;
            }
            else
            {
                approved = prop.needsApprovalForSleep ? pContext->currentClockForSleep == prop.approvedClockForSleep : true;
            }
            if (!approved)
            {
                pContext->canSleep = false;
                pOut->clockRequestedApprovalForSleep = pContext->currentClockForSleep;
            }
            pOut->lockingSleep = !approved;
            pOut->isSleeping = approved;

            // ロックが通っていたら次のステートへ
            if (prop.drivesSleepSequence && pContext->canSleep)
            {
                pOut->nextSleepSequenceStage = SleepSequenceStage::Sleep;
            }
        }
        else
        {
            // approval リクエストの取り下げ
            pOut->clockRequestedApprovalForSleep = 0;

            // approval の必要なものはロック
            pOut->lockingSleep = prop.needsApprovalForSleep;
            pOut->isSleeping = false;

            // 次の状態へ移行
            if (prop.drivesSleepSequence)
            {
                pOut->nextSleepSequenceStage = SleepSequenceStage::Finalizing;
            }
        }
    }

    // 必要なものを定義(不要なものはコメントアウトすること)
    //static void UpdateNegativeImmediately(Window* p, const Status& oldStatus, const Status& newStatus) NN_NOEXCEPT;
    static void UpdateNegative(Window* p, const Status& oldStatus, const Status& newStatus) NN_NOEXCEPT;
    //static UpdateImmediately(Window* p, const Status& oldStatus, const Status& newStatus) NN_NOEXCEPT;
    static void UpdatePositive(Window* p, const Status& oldStatus, const Status& newStatus) NN_NOEXCEPT;
    //static UpdatePositiveImmediately(Window* p, const Status& oldStatus, const Status& newStatus) NN_NOEXCEPT;

};

}}}}
