﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>

#include <nn/am/service/window/am_WindowControllerBase.h>

#include <nn/util/util_Exchange.h>

namespace nn { namespace am { namespace service { namespace window {

class HidWindowController
     : public WindowControllerBase<HidWindowController>
{
public:
    enum HidInputTakingMode
    {
        // hid 入力を受け取らない。
        HidInputTakingMode_Never = 0,

        // foregroundMode によって hid 入力を受け取れるかを決定する。
        HidInputTakingMode_TakeOnForeground,

        // foregroundMode を無視して hid 入力を受け取る。
        // 下位のアプレットも hid 入力を受け取ることが出来る。
        // IrSensor, Vibration は対象外。
        HidInputTakingMode_TakeForcedShared,

        // foregroundMode を無視して hid 入力を独占する。
        // 下位のアプレットは hid 入力を受け取ることができない。
        // IrSensor, Vibration は対象外。
        HidInputTakingMode_TakeForcedExclusive,
    };

    // Status という名前でデータ構造を定義
    struct Status
    {
        // needHidInput は hid 入力を必要とするアプレットで true とする。
        // 複数のアプレットが同時に true となってよい。
        bool needHidInput;

        // npadPolicyOwner は Npad のポリシーオーナーとなるべきアプレットで
        // true とする。１つのアプレットのみが true となれる。
        // 基本、OA もしくは All-FG のみが true となり、Partial-FG は除外する。
        bool npadPolicyOwner;

        // needIrSensorInput は IrSensor 入力を必要とするアプレットで
        // true とする。複数のアプレットが同時に true となってよい。
        bool needIrSensorInput;

        // irSensorPolicyOwner は IrSensor のポリシーオーナーとなるべき
        // アプレットで true とする。１つのアプレットのみが true となれる。
        // 基本、All-FG のみが true となり、OA や Partial-FG は除外する。
        bool irSensorPolicyOwner;

        // hidbusPolicyOwner は hidbus のポリシーオーナーとなるべき
        // アプレットで true とする。１つのアプレットのみが true となれる。
        // 基本、All-FG のみが true となり、OA や Partial-FG は除外する。
        bool hidbusPolicyOwner;

        // vibrationOwner は振動子の制御権を持つべきアプレットで true とする。
        // １つのアプレットのみが true となるべき。
        // 基本、All-FG のみが true となり、OA や Partial-FG は除外する。
        bool vibrationOwner;

        // needSixAxisSensorInput は６軸センサー入力を必要とするアプレットで
        // true とする。実質的に単一のアプレットのみを true とする。
        // 基本、All-FG のみが true となり、OA や Partial-FG は除外する。
        bool needSixAxisSensorInput;

        // bleConnectionOwner は BLE の通信オーナーとなるべきアプレットで
        // true とする。１つの All-FG なアプレットのみが true となれる。
        // 基本、OA や Partial-FG は除外され、常に false となる。
        bool bleConnectionOwner;
    };

    // active でないときの Status を初期化する関数
    static void SetNotActive(Status* p) NN_NOEXCEPT
    {
        p->needHidInput = false;
        p->npadPolicyOwner = false;

        p->needIrSensorInput = false;
        p->irSensorPolicyOwner = false;
        p->hidbusPolicyOwner = false;

        p->vibrationOwner = false;
        p->needSixAxisSensorInput = false;
        p->bleConnectionOwner = false;
    }

    // これが true を返す場合は Update 系関数は呼ばれない
    static bool AreEqual(const Status& x, const Status& y) NN_NOEXCEPT
    {
        return true
            && x.needHidInput == y.needHidInput
            && x.npadPolicyOwner == y.npadPolicyOwner
            && x.needIrSensorInput == y.needIrSensorInput
            && x.irSensorPolicyOwner == y.irSensorPolicyOwner
            && x.hidbusPolicyOwner == y.hidbusPolicyOwner
            && x.vibrationOwner == y.vibrationOwner
            && x.needSixAxisSensorInput == y.needSixAxisSensorInput
            && x.bleConnectionOwner == y.bleConnectionOwner
            ;
    }

    struct Property
    {
        // 現在のアプレットが hid 入力を受け取る条件を指定する。
        HidInputTakingMode hidInputTakingMode;
    };

    struct CalculationContext
    {
        // 上位のアプレットによって hid 入力が独占されているか。
        bool alreadyTakenHidInputExclusively;

        bool shouldNotifyToHid;
    };

    void InitializeCalculationContext(CalculationContext* pContext) NN_NOEXCEPT
    {
        pContext->alreadyTakenHidInputExclusively = false;
        pContext->shouldNotifyToHid = true;
    }

    void CalculateStatus(Status* pOut, const CommonWindowProperty& commonProp, const Property& prop, CalculationContext* pContext) NN_NOEXCEPT
    {
        auto needHidInput = false;
        auto npadPolicyOwner = false;
        auto needIrSensorInput = false;
        auto irSensorPolicyOwner = false;
        auto hidbusPolicyOwner = false;
        auto vibrationOwner = false;
        auto needSixAxisSensorInput = false;
        auto bleConnectionOwner = false;
        if (pContext->shouldNotifyToHid)
        {
            switch(prop.hidInputTakingMode)
            {
            case HidInputTakingMode_Never:
                break;
            case HidInputTakingMode_TakeForcedShared:
                {
                    // FG 状態は無視するが、上位のアプレットが独占していたら入力を取れない
                    needHidInput = !pContext->alreadyTakenHidInputExclusively;
                    break;
                }
            case HidInputTakingMode_TakeForcedExclusive:
                {
                    // FG 状態は無視するが、上位のアプレットが独占していたら入力を取れない
                    needHidInput = !pContext->alreadyTakenHidInputExclusively;
                    pContext->alreadyTakenHidInputExclusively = true;
                    break;
                }
            case HidInputTakingMode_TakeOnForeground:
                {
                    switch (commonProp.foregroundMode)
                    {
                        case ForegroundMode::All:
                        {
                            needHidInput = !pContext->alreadyTakenHidInputExclusively;
                            npadPolicyOwner = true;
                            needIrSensorInput = true;
                            irSensorPolicyOwner = true;
                            hidbusPolicyOwner = true;
                            vibrationOwner = true;
                            needSixAxisSensorInput = true;
                            bleConnectionOwner = true;
                            pContext->shouldNotifyToHid = false;
                            break;
                        }
                        case ForegroundMode::Partial:
                        {
                            needHidInput = !pContext->alreadyTakenHidInputExclusively;
                            needIrSensorInput = true;
                            // pContext->shouldNotifyToHid は true を維持する
                            break;
                        }
                        default:
                        {
                            // nop
                            break;
                        }
                    }
                    break;
                }
            default: NN_UNEXPECTED_DEFAULT;
            }
        }
        pOut->needHidInput = needHidInput;
        pOut->npadPolicyOwner = npadPolicyOwner;
        pOut->needIrSensorInput = needIrSensorInput;
        pOut->irSensorPolicyOwner = irSensorPolicyOwner;
        pOut->hidbusPolicyOwner = hidbusPolicyOwner;
        pOut->vibrationOwner = vibrationOwner;
        pOut->needSixAxisSensorInput = needSixAxisSensorInput;
        pOut->bleConnectionOwner = bleConnectionOwner;
    }

    // 必要なものを定義(不要なものはコメントアウトすること)
    //static void UpdateNegativeImmediately(Window* p, const Status& oldStatus, const Status& newStatus) NN_NOEXCEPT;
    static void UpdateNegative(Window* p, const Status& oldStatus, const Status& newStatus) NN_NOEXCEPT;
    //static void UpdateImmediately(Window* p, const Status& oldStatus, const Status& newStatus) NN_NOEXCEPT;
    static void UpdatePositive(Window* p, const Status& oldStatus, const Status& newStatus) NN_NOEXCEPT;
    //static void UpdatePositiveImmediately(Window* p, const Status& oldStatus, const Status& newStatus) NN_NOEXCEPT;

};

}}}}
