﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>

#include <nn/am/service/window/am_WindowControllerBase.h>
#include <nn/am/service/am_GpuResourceControl.h>
#include <nn/am/service/am_CommonTypes.h>
#include <nn/nn_TimeSpan.h>
#include <algorithm>
#include <nn/util/util_Optional.h>
#include <nn/os/os_Types.h>

namespace nn { namespace am { namespace service { namespace window {

class ForegroundWindowController
     : public WindowControllerBase<ForegroundWindowController>
{
public:

    // Status という名前でデータ構造を定義
    struct Status
    {
        WindowFocusState focusState;
        bool canUseApplicationCoreAsRequested;
        bool canRun = true;
        util::optional<WirelessPriorityMode> pWirelessPriorityMode;
        util::optional<GpuResourceControlInfo> pGpuResourceControlInfo;
    };

    // active でないときの Status を初期化する関数
    static void SetNotActive(Status* p) NN_NOEXCEPT
    {
        p->focusState = WindowFocusState::Background;
        p->canUseApplicationCoreAsRequested = true;
        p->canRun = true;
        p->pWirelessPriorityMode = util::nullopt;
        p->pGpuResourceControlInfo = util::nullopt;
    }

    // これが true を返す場合は Update 系関数は呼ばれない
    static bool AreEqual(const Status& x, const Status& y) NN_NOEXCEPT
    {
        return true
            && x.focusState == y.focusState
            && x.canUseApplicationCoreAsRequested == y.canUseApplicationCoreAsRequested
            && x.canRun == y.canRun
            && x.pWirelessPriorityMode == y.pWirelessPriorityMode
            && x.pGpuResourceControlInfo == y.pGpuResourceControlInfo;
    }

    struct Property
    {
        CoreUsageRequest applicationCoreUsageRequest{CoreUsageRequest::None};
        bool controlsGpuResource{false};
        bool canRun = true;
        util::optional<GpuResourceGroupId> pGpuResourceGroupId{util::nullopt};
        TimeSpan gpuTimeSliceInForeground{0};
        TimeSpan gpuTimeSliceInBackground{0};
        TimeSpan gpuTimeSliceBoost{0};
        uint8_t gpuTimeSlicePriority{255};
        util::optional<WirelessPriorityMode> pWirelessPriorityModeRequest{util::nullopt};
    };

    struct CalculationContext
    {
        CoreAvailability applicationCoreAvailability;
        bool canGetFocus;
        bool mustBeBackground;
        bool canSetWirelessPriorityMode;
        TimeSpan restGpuTimeSlice;
        GpuResourceControlInfo gpuResourceInfo;
    };

    void InitializeCalculationContext(CalculationContext* pContext) NN_NOEXCEPT
    {
        pContext->applicationCoreAvailability = CoreAvailability::All;
        pContext->canGetFocus = true;
        pContext->mustBeBackground = false;
        pContext->canSetWirelessPriorityMode = true;
        pContext->restGpuTimeSlice = GetGpuScheduleTimeSpan();
    }

    void CalculateStatus(Status* pOut, const CommonWindowProperty& commonProp, const Property& prop, CalculationContext* pContext) NN_NOEXCEPT
    {
        if (pContext->canSetWirelessPriorityMode && prop.pWirelessPriorityModeRequest)
        {
            pOut->pWirelessPriorityMode = prop.pWirelessPriorityModeRequest;
            pContext->canSetWirelessPriorityMode = false;
        }
        else
        {
            pOut->pWirelessPriorityMode = util::nullopt;
        }

        auto focusState = [&]() -> WindowFocusState
        {
            if (pContext->mustBeBackground)
            {
                if (pContext->canGetFocus)
                {
                    if (commonProp.foregroundMode == ForegroundMode::All)
                    {
                        pContext->canGetFocus = false;
                    }
                    return WindowFocusState::Background;
                }
                else
                {
                    return WindowFocusState::BackgroundOutOfFocus;
                }
            }
            else
            {
                switch (commonProp.foregroundMode)
                {
                    case ForegroundMode::All:
                    case ForegroundMode::Partial:
                    {
                        auto ret = pContext->canGetFocus ? WindowFocusState::InFocus : WindowFocusState::OutOfFocus;
                        if (commonProp.foregroundMode == ForegroundMode::All)
                        {
                            pContext->canGetFocus = false;
                        }
                        if (commonProp.isMainAppletWindow)
                        {
                            pContext->mustBeBackground = true;
                            pContext->canGetFocus = true;
                        }
                        return ret;
                    }
                    default: return WindowFocusState::Background;
                }
            }
        }();
        auto canUseApplicationCoreAsRequested = [&]() -> bool
        {
            switch (prop.applicationCoreUsageRequest)
            {
                case CoreUsageRequest::Exclusive:
                {
                    if (pContext->applicationCoreAvailability == CoreAvailability::All)
                    {
                        pContext->applicationCoreAvailability = CoreAvailability::None;
                        return true;
                    }
                    else
                    {
                        return false;
                    }
                }
                case CoreUsageRequest::Shared:
                {
                    if (pContext->applicationCoreAvailability == CoreAvailability::None)
                    {
                        return false;
                    }
                    else
                    {
                        pContext->applicationCoreAvailability = CoreAvailability::Partial;
                        return true;
                    }
                }
                case CoreUsageRequest::None:
                {
                    return true;
                }
                default: NN_UNEXPECTED_DEFAULT;
            }
        }();
        pOut->focusState = focusState;
        pOut->canUseApplicationCoreAsRequested = canUseApplicationCoreAsRequested;
        auto gpuTimeSliceRequested = (focusState == WindowFocusState::InFocus) ? prop.gpuTimeSliceInForeground : prop.gpuTimeSliceInBackground;
        gpuTimeSliceRequested += prop.gpuTimeSliceBoost;
        auto gpuTimeSlice = std::min(gpuTimeSliceRequested, pContext->restGpuTimeSlice);
        pContext->restGpuTimeSlice -= gpuTimeSlice;
        if (prop.pGpuResourceGroupId)
        {
            pContext->gpuResourceInfo.Add(*prop.pGpuResourceGroupId, gpuTimeSlice);
        }
        if (prop.controlsGpuResource)
        {
            pOut->pGpuResourceControlInfo = pContext->gpuResourceInfo;
        }
        pOut->canRun = prop.canRun;
    }   // NOLINT(impl/function_size)

    // 必要なものを定義(不要なものはコメントアウトすること)
    //static void UpdateNegativeImmediately(Window* p, const Status& oldStatus, const Status& newStatus) NN_NOEXCEPT;
    static void UpdateNegative(Window* p, const Status& oldStatus, const Status& newStatus) NN_NOEXCEPT;
    //static UpdateImmediately(Window* p, const Status& oldStatus, const Status& newStatus) NN_NOEXCEPT;
    static void UpdatePositive(Window* p, const Status& oldStatus, const Status& newStatus) NN_NOEXCEPT;
    //static UpdatePositiveImmediately(Window* p, const Status& oldStatus, const Status& newStatus) NN_NOEXCEPT;

    struct EventType
    {
        bool onFatal;
        util::optional<os::ProcessId> pExceptionProcessId;
    };

    void HandleEvent(Window* p, const EventStatus&, const EventType& e) NN_NOEXCEPT;

};

}}}}
