﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>

#include <nn/am/service/window/am_WindowControllerBase.h>

namespace nn { namespace am { namespace service { namespace window {

class DisplayWindowController
     : public WindowControllerBase<DisplayWindowController>
{
public:

    // Status という名前でデータ構造を定義
    struct Status
    {
        int32_t zOrder;
        bool staysDisplayOnInvisible;

        bool isVrMode;
        bool isVrModeChangedByApplication;
        bool needsRequestVrModeCurtain;

        bool isLcdBacklightOff;
        bool needsSwitchLcdBacklightOnByForce;

        uint64_t clockOnRequestToDisplay{0};
        uint64_t clockOnDisplay{0};
        uint64_t clockOnExpired{0};

        bool IsVisible() const NN_NOEXCEPT
        {
            return clockOnDisplay > clockOnExpired;
        }

        bool IsRequestedToDisplay() const NN_NOEXCEPT
        {
            return !IsVisible() && (clockOnExpired < clockOnRequestToDisplay);
        }

        void Expire(uint64_t clock) NN_NOEXCEPT
        {
            this->clockOnExpired = clock;
            this->clockOnDisplay = 0;
            this->clockOnRequestToDisplay = 0;
        }

        void Display(uint64_t clock) NN_NOEXCEPT
        {
            this->clockOnDisplay = clock;
        }

        void RequestToDisplay(uint64_t clock) NN_NOEXCEPT
        {
            this->clockOnRequestToDisplay = clock;
        }

        void SetFixedParameter(const CommonWindowProperty& commonProp) NN_NOEXCEPT
        {
            if (commonProp.isOverlayAppletWindow)
            {
                this->zOrder = 3;
                this->staysDisplayOnInvisible = false;
                return;
            }
            else
            {
                switch (commonProp.foregroundMode)
                {
                    case ForegroundMode::All:
                    {
                        this->zOrder = 1;
                        this->staysDisplayOnInvisible = true;
                        return;
                    }
                    case ForegroundMode::Partial:
                    {
                        this->zOrder = 2;
                        this->staysDisplayOnInvisible = false;
                        return;
                    }
                    default:
                    {
                        this->zOrder = 0;
                        this->staysDisplayOnInvisible = false;
                        return;
                    }
                }
            }
        }
    };

    // active でないときの Status を初期化する関数
    static void SetNotActive(Status* p) NN_NOEXCEPT
    {
        p->clockOnRequestToDisplay = 0;
        p->clockOnDisplay = 0;
        p->zOrder = 0;
        p->isVrMode = false;
        p->isVrModeChangedByApplication = false;
        p->needsRequestVrModeCurtain = false;
        p->isLcdBacklightOff = false;
        p->needsSwitchLcdBacklightOnByForce = false;
    }

    // これが true を返す場合は Update 系関数は呼ばれない
    static bool AreEqual(const Status& x, const Status& y) NN_NOEXCEPT
    {
        return true
            && x.IsVisible() == y.IsVisible()
            && x.IsRequestedToDisplay() == y.IsRequestedToDisplay()
            && x.zOrder == y.zOrder
            && x.isVrMode == y.isVrMode
            && x.needsRequestVrModeCurtain == y.needsRequestVrModeCurtain
            && x.isLcdBacklightOff == y.isLcdBacklightOff
            && x.needsSwitchLcdBacklightOnByForce == y.needsSwitchLcdBacklightOnByForce
        ;
    }

    struct Property
    {
        bool isVrMode{false};
        bool isVrModeChangedByApplication{false};
        bool isVrModeDualScreenSupported{false};
        bool handlesRequestToVrModeCurtain{false};

        bool requestsToSwitchLcdBacklightOff{false};

        bool displayIsValid{true};
        bool handlesRequestToDisplay{true};
        uint64_t clockOnDisplay{0};

        void Display(uint64_t clock) NN_NOEXCEPT
        {
            if (clockOnDisplay >= clock)
            {
                return;
            }
            this->clockOnDisplay = clock;
        }
    };

    struct CalculationContext
    {
        bool isVrMode;
        bool isVrModeChangedByApplication;
        bool needsVrModeCurtain;

        bool foregroundVisible;
        uint64_t currentClock;
    };

    uint64_t currentClock{0};

    void InitializeCalculationContext(CalculationContext* pContext) NN_NOEXCEPT
    {
        pContext->isVrMode = false;
        pContext->isVrModeChangedByApplication = false;
        pContext->needsVrModeCurtain = false;

        pContext->foregroundVisible = true;
        pContext->currentClock = ++this->currentClock;
    }

    void CalculateStatus(Status* pOut, const CommonWindowProperty& commonProp, const Property& prop, CalculationContext* pContext) NN_NOEXCEPT
    {
        // OA Window の Property 値を pContext に反映させる
        if (commonProp.isOverlayAppletWindow)
        {
            pContext->isVrMode = prop.isVrMode;
            pContext->isVrModeChangedByApplication = prop.isVrModeChangedByApplication;
        }

        // OA 以下の Window にも反映させる
        pOut->isVrMode = pContext->isVrMode;
        pOut->isVrModeChangedByApplication = pContext->isVrModeChangedByApplication;

        // カーテン通知
        if (prop.handlesRequestToVrModeCurtain)
        {
            pOut->needsRequestVrModeCurtain = pContext->needsVrModeCurtain;
        }

        pOut->SetFixedParameter(commonProp);
        if (commonProp.isOverlayAppletWindow)
        {
            pOut->Display(1);
            return;
        }
        else
        {
            switch (commonProp.foregroundMode)
            {
                case ForegroundMode::All:
                case ForegroundMode::Partial:
                {
                    if (pContext->foregroundVisible)
                    {
                        pOut->isLcdBacklightOff = prop.requestsToSwitchLcdBacklightOff;
                        pOut->needsSwitchLcdBacklightOnByForce = false;

                        if (pContext->isVrMode && !prop.isVrModeDualScreenSupported)
                        {
                            pContext->needsVrModeCurtain = true;
                        }

                        if (prop.displayIsValid)
                        {
                            pOut->RequestToDisplay(pContext->currentClock);
                            pOut->Display(prop.handlesRequestToDisplay ? prop.clockOnDisplay : pContext->currentClock);
                        }
                        else
                        {
                            pOut->Expire(pContext->currentClock);
                        }

                        if (commonProp.foregroundMode == ForegroundMode::All)
                        {
                            pContext->foregroundVisible = false;
                        }
                        return;
                    }
                    else
                    {
                        pOut->isLcdBacklightOff = false;
                        pOut->needsSwitchLcdBacklightOnByForce = prop.requestsToSwitchLcdBacklightOff;

                        pOut->Expire(pContext->currentClock);
                        return;
                    }
                }
                default:
                {
                    pOut->Expire(pContext->currentClock);
                    return;
                }
            }
        }
    }

    // 必要なものを定義(不要なものはコメントアウトすること)
    //static void UpdateNegativeImmediately(Window* p, const Status& oldStatus, const Status& newStatus) NN_NOEXCEPT;
    static void UpdateNegative(Window* p, const Status& oldStatus, const Status& newStatus) NN_NOEXCEPT;
    //static void UpdateImmediately(Window* p, const Status& oldStatus, const Status& newStatus) NN_NOEXCEPT;
    static void UpdatePositive(Window* p, const Status& oldStatus, const Status& newStatus) NN_NOEXCEPT;
    //static void UpdatePositiveImmediately(Window* p, const Status& oldStatus, const Status& newStatus) NN_NOEXCEPT;

};

}}}}
