﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os.h>
#include <nn/applet/applet_FundamentalTypes.h>
#include <nn/vi/sf/vi_ServiceTypes.h>
#include <nn/vi/fbshare/vi_SharedBufferHandle.h>
#include <nn/vi/fbshare/vi_SharedLayerHandle.h>

namespace nn{ namespace am{ namespace service{ namespace display{

    namespace detail{
        class IntegratedDisplayLayerPool;

        enum IntegratedDisplayZOrder
        {
            IntegratedDisplayZOrder_Default = 0,
            IntegratedDisplayZOrder_SystemSharedFullscreen = 0,
            IntegratedDisplayZOrder_SystemSharedPartial = 2,
            IntegratedDisplayZOrder_ApplicationCopyright = 3,
            IntegratedDisplayZOrder_OperationMode = 5,
        };

        enum IntegratedDisplayLayerState
        {
            // 未使用
            IntegratedDisplayLayerState_Free = 0,

            // 未初期化
            IntegratedDisplayLayerState_NotInitialized,

            // パラメータが未確定
            IntegratedDisplayLayerState_Undetermined,

            // 確定済:
            IntegratedDisplayLayerState_Determined,
        };

        enum IntegratedDisplayLayerUserClass
        {
            IntegratedDisplayLayerUserClass_Unspecified = 0,
            IntegratedDisplayLayerUserClass_Application,
            IntegratedDisplayLayerUserClass_FullscreenApplet,
            IntegratedDisplayLayerUserClass_PartialApplet,
            IntegratedDisplayLayerUserClass_Overlay,
            IntegratedDisplayLayerUserClass_NoDisplay,
        };

        enum IntegratedDisplayLayerBufferOwner
        {
            IntegratedDisplayLayerBufferOwner_Unspecified = 0,
            IntegratedDisplayLayerBufferOwner_OwnedByApplet,
            IntegratedDisplayLayerBufferOwner_SystemShared,
        };

        enum IntegratedDisplayLayerDestinationLayer
        {
            IntegratedDisplayLayerDestinationLayer_Unspecified = 0,
            IntegratedDisplayLayerDestinationLayer_DisplayLayer,
            IntegratedDisplayLayerDestinationLayer_IndirectLayer,
        };

        class IntegratedDisplayLayer
        {
            friend class IntegratedDisplayLayerPool;
        public:
            IntegratedDisplayLayer() NN_NOEXCEPT;

            // @pre いつでも
            IntegratedDisplayLayerState GetState() const NN_NOEXCEPT;
            // @pre いつでも
            IntegratedDisplayLayerUserClass GetUserClass() const NN_NOEXCEPT;
            // @pre いつでも
            IntegratedDisplayLayerBufferOwner GetBufferOwner() const NN_NOEXCEPT;
            // @pre いつでも
            IntegratedDisplayLayerDestinationLayer GetDestinationLayer() const NN_NOEXCEPT;
            // @pre いつでも
            nn::applet::AppletResourceUserId GetAruid() const NN_NOEXCEPT;
            // @pre いつでも
            nn::vi::IndirectLayerHandleType GetIndirectLayerHandle() const NN_NOEXCEPT;

            // @pre GetState() != NotInitialized
            nn::vi::LayerId GetDirectLayerId() const NN_NOEXCEPT;
            // @pre GetState() != NotInitialized
            nn::vi::IndirectProducerHandleType GetIndirectProducerHandle() const NN_NOEXCEPT;
            // @pre GetState() != NotInitialized
            nn::vi::fbshare::SharedLayerHandle GetSharedLayerHandle() const NN_NOEXCEPT;
            // @pre !GetSharedLayerHandle().IsInvalid()
            nn::os::SystemEventType* GetSharedLayerDetachReadyEvent() NN_NOEXCEPT;
            // @pre GetState() != NotInitialized
            nn::vi::LayerId GetIndirectProducerHandleForSharedLayer() const NN_NOEXCEPT;

            // @pre GetState() != NotInitialized
            bool IsVisible() const NN_NOEXCEPT;
            // @pre GetState() != NotInitialized
            int32_t GetZOrder() const NN_NOEXCEPT;
            // @pre GetState() != NotInitialized
            bool IsCaptureTarget() const NN_NOEXCEPT;
            // @pre GetState() != NotInitialized
            bool IsTransitionLayerRequested() const NN_NOEXCEPT;
            // @pre GetState() != NotInitialized
            bool IsConductorIfForeground() const NN_NOEXCEPT;
            // @pre GetState() != NotInitialized
            bool IsRecordingLayerStackRequested() const NN_NOEXCEPT;
            // @pre GetState() != NotInitialized
            bool IsDebugLayerStackRequested() const NN_NOEXCEPT;
            // @pre GetState() != NotInitialized
            bool IsCallerAppletCaptureRequested() const NN_NOEXCEPT;
            // @pre GetState() != NotInitialized
            bool IsCallerAppletCaptureScreenshotPermitted() const NN_NOEXCEPT;

            // @pre GetState() != NotInitialized
            bool IsSharedBufferExported() const NN_NOEXCEPT;

            // @pre GetState() != NotInitialized
            bool InvokeScreenShotPermissionGetterFunction() NN_NOEXCEPT;

            // @pre GetState() == NotInitialized
            // @post GetState() == Undetermined
            // @post GetUserClass() == Unspecified
            // @post GetBufferOwner() == Unspecified
            // @post GetDestinationLayer() == Unspecified
            // @post GetAruid() == Invalid
            void Initialize() NN_NOEXCEPT;

            // @pre GetState() != NotInitialized
            // @post GetState() == NotInitialized
            // @post GetUserClass() == Unspecified
            // @post GetBufferOwner() == Unspecified
            // @post GetDestinationLayer() == Unspecified
            // @post GetAruid() == Invalid
            void Finalize() NN_NOEXCEPT;

            // @pre GetState() == Undetermined
            // @pre GetUserClass() == Unspecified
            // @post GetUserClass() == value
            // @post GetAruid() == aruid
            // @post result == true の場合、 GetState() == Determined
            // @post result == false の場合、 GetState() == Undetermined
            // @details
            // この関数の呼出の結果、全パラメータが揃った場合、レイヤの構築を行う。
            bool SpecifyUserClass(IntegratedDisplayLayerUserClass value, applet::AppletResourceUserId aruid) NN_NOEXCEPT;

            // @param [in] hBuffer value == SystemShared の場合に有効な値を指定する。それ以外の場合は無効値で OK。
            // @pre GetState() == Undetermined
            // @pre GetBufferOwner() == Unspecified
            // @post GetBufferOwner() == value
            // @post result == true の場合、 GetState() == Determined
            // @post result == false の場合、 GetState() == Undetermined
            // @details
            // この関数の呼出の結果、全パラメータが揃った場合、レイヤの構築を行う。
            bool SpecifyBufferOwner(
                IntegratedDisplayLayerBufferOwner value,
                nn::vi::fbshare::SharedBufferHandle hBuffer
            ) NN_NOEXCEPT;

            // @param [in] indirectLayerHandle     value == Indirect の場合に有効な値を指定する。それ以外の場合は 0 で OK。
            // @param [in] indirectLayerFlipOffset value == Indirect の場合に有効な値を指定する。それ以外の場合は 0 で OK。
            // @pre GetState() == Undetermined
            // @pre GetDestinationLayer() == Unspecified
            // @post GetDestinationLayer() == value
            // @post result == true の場合、 GetState() == Determined
            // @post result == false の場合、 GetState() == Undetermined
            // @details
            // この関数の呼出の結果、全パラメータが揃った場合、レイヤの構築を行う。
            bool SpecifyDestinationLayer(
                IntegratedDisplayLayerDestinationLayer value,
                nn::vi::IndirectLayerHandleType indirectLayerHandle,
                nn::TimeSpan indirectLayerFlipOffset
                ) NN_NOEXCEPT;

            // @pre GetState() != NotInitilaized
            void SetScreenShotPermissionGetterFunction(bool (*func)(void*), void* userPtr) NN_NOEXCEPT;

            // @pre GetState() != NotInitilaized
            // @details
            // 可視状態を設定する。
            // レイヤの構築前にも事前に設定しておくことが可能。
            void SetVisibility(bool isVisible, int32_t zOrder, bool isCaptureTarget) NN_NOEXCEPT;

            // @pre GetState() != NotInitialized
            void SetTransitionLayerRequested(bool value) NN_NOEXCEPT;
            // @pre GetState() != NotInitialized
            void SetConductorIfForeground(bool value) NN_NOEXCEPT;
            // @pre GetState() != NotInitialized
            void SetRecordingLayerStackRequested(bool value) NN_NOEXCEPT;
            // @pre GetState() != NotInitialized
            void SetDebugLayerStackRequested(bool value) NN_NOEXCEPT;
            // @pre GetState() != NotInitialized
            void SetCallerAppletCaptureRequested(bool value, bool isScreenshotPermitted) NN_NOEXCEPT;

            // @pre GetState() == Determined
            // @pre !IsSharedBufferExported()
            void ExportSharedBuffer(nn::vi::fbshare::SharedBufferHandle hBuffer) NN_NOEXCEPT;

        private:
            // @brief 全パラメータが指定されているか調べて、指定されていればレイヤを構築する。
            // @pre GetState() == Undetermined
            bool SetupIfParameterDeterminedImpl() NN_NOEXCEPT;

        private:
            IntegratedDisplayLayerState m_State;

        // @name パラメータ
        // @{
            IntegratedDisplayLayerUserClass m_UserClass;
            IntegratedDisplayLayerBufferOwner m_BufferOwner;
            IntegratedDisplayLayerDestinationLayer m_DestinationLayer;
            applet::AppletResourceUserId m_Aruid;

            // BufferOwner == SystemShared の場合。
            nn::vi::fbshare::SharedBufferHandle m_SharedBufferHandle;

            // Destination == Indirect の場合。
            // 所有権は IntegratedApplet が持っている
            nn::vi::IndirectLayerHandleType m_IndirectLayerHandle;
            // Destination == Indirect の場合。
            nn::TimeSpan m_IndirectProducerFlipOffset;

            bool (*m_pScreenShotPermissionGetterFunction)(void*);
            void* m_pScreenShotPermissionGetterUserPtr;
        // @}

        // @name 状態
        // @{

            // SetVisibility
            bool m_IsVisible;
            int32_t m_ZOrder;
            bool m_IsCaptureTarget;

            bool m_IsTransitionLayerRequested;
            bool m_IsConductorIfForeground;
            bool m_IsRecordingLayerStackRequested;
            bool m_IsDebugLayerStackRequested;
            bool m_IsCallerAppletCaptureRequested;
            bool m_IsCallerAppletCaptureScreenshotPermitted;
        // @}

        // @name オブジェクト
        // @{
            nn::vi::LayerId m_DirectLayerId;
            nn::vi::IndirectProducerHandleType m_IndirectProducerHandle;
            nn::vi::fbshare::SharedLayerHandle m_SharedLayerHandle;
            nn::os::SystemEventType m_SharedLayerDetachReadyEvent;

            nn::vi::fbshare::SharedBufferHandle m_ExportedSharedBufferHandle;
            nn::vi::IndirectProducerHandleType m_IndirectProducerHandleForSharedLayer;
        // @}

        };
    }
}}}}

