﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_TimeSpan.h>
#include <nn/ncm/ncm_ContentMetaId.h>
#include <nn/account/account_Types.h>
#include <nn/os/os_Tick.h>
#include <nn/util/util_Optional.h>
#include <nn/util/util_Span.h>
#include <nn/ns/ns_ApplicationRightsTypes.h>

namespace nn { namespace am { namespace service { namespace rightsManagement {

struct AllUserTag {};
constexpr AllUserTag AllUser = {};

struct RightsEnvironmentStatus
{
    bool isAvailable;
    os::Tick expirationTick;
};

RightsEnvironmentStatus GetInitialRightsEnvironmentStatus() NN_NOEXCEPT;

class RightsEnvironment
{
public:

    RightsEnvironment() NN_NOEXCEPT = default;
    RightsEnvironment(const RightsEnvironment&) = delete;
    RightsEnvironment(RightsEnvironment&&) = delete;
    RightsEnvironment& operator=(const RightsEnvironment&) = delete;
    RightsEnvironment& operator=(RightsEnvironment&&) = delete;

    virtual ~RightsEnvironment() NN_NOEXCEPT = default;

    virtual Result SetUsers(util::Span<const account::Uid> uids) NN_NOEXCEPT = 0;
    virtual Result SetUsers(AllUserTag) NN_NOEXCEPT = 0;

    virtual void Start() NN_NOEXCEPT = 0;
    virtual void Stop() NN_NOEXCEPT = 0;

    virtual RightsEnvironmentStatus GetStatus() const NN_NOEXCEPT = 0;

    virtual void RequestExpandExpirationDate() NN_NOEXCEPT = 0;

    virtual Result GetNsHandle(Bit64* pOut) NN_NOEXCEPT = 0;

    // TORIAEZU
    virtual TimeSpan GetPollingInterval() const NN_NOEXCEPT = 0;

};

class NoCheckRightsEnvironment
    : public RightsEnvironment
{
public:

    NoCheckRightsEnvironment() = default;
    NoCheckRightsEnvironment(const NoCheckRightsEnvironment&) = delete;
    NoCheckRightsEnvironment(NoCheckRightsEnvironment&&) = delete;
    NoCheckRightsEnvironment& operator=(const NoCheckRightsEnvironment&) = delete;
    NoCheckRightsEnvironment& operator=(NoCheckRightsEnvironment&&) = delete;

    virtual Result SetUsers(util::Span<const account::Uid> uids) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result SetUsers(AllUserTag) NN_NOEXCEPT NN_OVERRIDE;

    virtual void Start() NN_NOEXCEPT NN_OVERRIDE;
    virtual void Stop() NN_NOEXCEPT NN_OVERRIDE;

    virtual RightsEnvironmentStatus GetStatus() const NN_NOEXCEPT NN_OVERRIDE;

    virtual void RequestExpandExpirationDate() NN_NOEXCEPT NN_OVERRIDE;

    virtual Result GetNsHandle(Bit64* pOut) NN_NOEXCEPT NN_OVERRIDE;

    // TORIAEZU
    virtual TimeSpan GetPollingInterval() const NN_NOEXCEPT NN_OVERRIDE
    {
        return TimeSpan::FromDays(100); // 十分長い時間
    }

};

class ApplicationRightsEnvironment
    : public RightsEnvironment
{
public:

    ApplicationRightsEnvironment() = default;
    ApplicationRightsEnvironment(const ApplicationRightsEnvironment&) = delete;
    ApplicationRightsEnvironment(ApplicationRightsEnvironment&&) = delete;
    ApplicationRightsEnvironment& operator=(const ApplicationRightsEnvironment&) = delete;
    ApplicationRightsEnvironment& operator=(ApplicationRightsEnvironment&&) = delete;

    Result Initialize(ncm::ApplicationId applicationId) NN_NOEXCEPT;
    virtual ~ApplicationRightsEnvironment() NN_NOEXCEPT NN_OVERRIDE;

    virtual Result SetUsers(util::Span<const account::Uid> uids) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result SetUsers(AllUserTag) NN_NOEXCEPT NN_OVERRIDE;

    virtual void Start() NN_NOEXCEPT NN_OVERRIDE;
    virtual void Stop() NN_NOEXCEPT NN_OVERRIDE;

    virtual RightsEnvironmentStatus GetStatus() const NN_NOEXCEPT NN_OVERRIDE;

    virtual void RequestExpandExpirationDate() NN_NOEXCEPT NN_OVERRIDE;

    virtual Result GetNsHandle(Bit64* pOut) NN_NOEXCEPT NN_OVERRIDE;

    // TORIAEZU
    virtual TimeSpan GetPollingInterval() const NN_NOEXCEPT NN_OVERRIDE
    {
        return TimeSpan::FromSeconds(10);
    }

private:

    ncm::ApplicationId m_ApplicationId{};
    ns::RightsEnvironmentHandle m_Handle{};

    void AddTargetImpl(ncm::ApplicationId applicationId) NN_NOEXCEPT;

    bool IsInitialized() const NN_NOEXCEPT
    {
        return m_Handle != static_cast<decltype(m_Handle)>(0);
    }

    void FinalizeImpl() NN_NOEXCEPT;

};

}}}}
