﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <utility>
#include <memory>

#include <nn/applet/applet_Types.h>
#include <nn/am/service/am_IntegratedApplet.h>
#include <nn/am/service/am_Foundation.sfdl.h>
#include <nn/am/service/am_Functions.sfdl.h>
#include <nn/am/service/am_AppletProxy.h>
#include <nn/am/service/am_Storage.h>
#include <nn/sf/sf_Types.h>
#include <nn/os/os_Mutex.h>

namespace nn { namespace am { namespace service {

class IntegratedLibraryApplet
    : public IntegratedApplet
{
public:

    explicit IntegratedLibraryApplet(AppletSystem* pAppletSystem, applet::AppletId appletId, applet::LibraryAppletMode mode) NN_NOEXCEPT;

    // for AppletSystem
    sf::SharedPointer<ILibraryAppletAccessor> CreateAccessor() NN_NOEXCEPT;

    virtual void SendPlayLog(pdm::AppletEventType eventType) NN_NOEXCEPT NN_OVERRIDE final;

    NN_AM_SERVICE_DIAGNOSTICS_DEFINE_APPLET_KIND_STRING("LA")

private:

    typedef StorageChannel Channel;

    // 基本情報
    applet::AppletId m_AppletId;
    applet::LibraryAppletMode m_LibraryAppletMode;
    bool m_HasMainAppletWindow{false};
    CoreUsageRequest m_ApplicationCoreUsageRequest = CoreUsageRequest::None;

    // 入出力
    std::shared_ptr<Channel> m_InChannel;
    std::shared_ptr<Channel> m_OutChannel;
    std::shared_ptr<Channel> m_InteractiveInChannel;
    std::shared_ptr<Channel> m_InteractiveOutChannel;
    std::shared_ptr<Channel> m_ExtraChannel;
    std::shared_ptr<Channel> m_Context;

    // チャネルクリア
    os::Mutex m_ChannelClearMutex{false};
    int m_ChannelRefCount{0};
    void IncrementChannelRef() NN_NOEXCEPT;
    void DecrementChannelRef() NN_NOEXCEPT;

    // Return
    os::Mutex m_ReturnMutex{false};
    bool m_ReturnIsValid{false};

    // overrides(option)
    //virtual Result Before() NN_NOEXCEPT NN_OVERRIDE;
    //virtual Result After() NN_NOEXCEPT NN_OVERRIDE;
    virtual void Cleanup() NN_NOEXCEPT NN_OVERRIDE;
    //virtual void RunBehindProcess(process::NsProcess* pProcess) NN_NOEXCEPT NN_OVERRIDE;
    virtual void OnProcessBegin(os::ProcessId processId) NN_NOEXCEPT NN_OVERRIDE;

    uint64_t m_ProcessBeginCount{0};

    // overrides IntegratedApplet
    virtual applet::AppletIdentityInfo GetAppletIdentityInfo() NN_NOEXCEPT NN_OVERRIDE final;
    virtual Result CreateProcessImpl(std::shared_ptr<process::NsProcess>* pOut) NN_NOEXCEPT NN_OVERRIDE final;
    virtual std::shared_ptr<process::NsProcess> MakeAttachedProcess(os::ProcessId processId) NN_NOEXCEPT NN_OVERRIDE final;

    // overrides IntegratedApplet
    virtual AppletProxyInfo DoCreateAppletProxy(os::ProcessId processId) NN_NOEXCEPT NN_OVERRIDE;

    // for impl
    std::shared_ptr<IntegratedLibraryApplet> SharedFromThis() NN_NOEXCEPT
    {
        return std::static_pointer_cast<IntegratedLibraryApplet>(this->shared_from_this());
    }

    class LibraryAppletAccessorImpl;
    class LibraryAppletSelfProxyImpl;

    virtual window::ForegroundMode GetForegroundMode() const NN_NOEXCEPT NN_OVERRIDE
    {
        switch (m_LibraryAppletMode)
        {
            case applet::LibraryAppletMode_AllForeground: return window::ForegroundMode::All;
            case applet::LibraryAppletMode_PartialForeground: return window::ForegroundMode::Partial;
            case applet::LibraryAppletMode_PartialForegroundWithIndirectDisplay: return window::ForegroundMode::Partial;
            default: return window::ForegroundMode::None;
        }
    }

    virtual GpuResourceGroupId GetGpuResourceGroupId() const NN_NOEXCEPT NN_OVERRIDE
    {
        return GpuResourceGroupId_System;
    }

    // overrieds (window)
    virtual window::WindowProperty GetInitialWindowProperty() NN_NOEXCEPT NN_OVERRIDE
    {
        auto ret = IntegratedApplet::GetInitialWindowProperty();
        ret.isMainAppletWindow = m_HasMainAppletWindow;
        ret.applicationCoreUsageRequest = m_ApplicationCoreUsageRequest;
        switch (GetForegroundMode())
        {
            case window::ForegroundMode::All:
            {
                ret.gpuTimeSliceInForeground = GetGpuTimeSliceOf(GpuTimeKind_FgLa);
                ret.gpuTimeSliceInBackground = GetGpuTimeSliceOf(GpuTimeKind_BgLa);
                ret.determinesHomeButtonLongPressingTime = true;
                break;
            }
            case window::ForegroundMode::Partial:
            {
                ret.gpuTimeSliceInForeground = GetGpuTimeSliceOf(GpuTimeKind_PartialFgLa);
                ret.gpuTimeSliceInBackground = GetGpuTimeSliceOf(GpuTimeKind_BgLa);
                break;
            }
            default:
            {
                ret.gpuTimeSliceInForeground = 0;
                ret.gpuTimeSliceInBackground = 0;
                break;
            }
        }
        ret.gpuTimeSlicePriority = 1;
        return ret;
    }

};

}}}
