﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <utility>
#include <memory>
#include <mutex>

#include <nn/applet/applet_Types.h>
#include <nn/am/service/am_CommonTypes.h>
#include <nn/am/service/am_CallingWindableApplet.h>
#include <nn/am/service/am_Foundation.sfdl.h>
#include <nn/am/service/am_Functions.sfdl.h>
#include <nn/sf/sf_Types.h>
#include <nn/sf/sf_NativeHandle.h>

#include <nn/os/os_Mutex.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_Tick.h>
#include <nn/util/util_Optional.h>
#include <nn/util/util_IntrusiveList.h>

#include <nn/ns/ns_ApplicationControlDataApi.h>
#include <nn/ncm/ncm_ContentMetaId.h>
#include <nn/idle/idle_SystemTypes.h>
#include <nn/am/service/am_AppletProxy.h>
#include <nn/am/service/am_AppletSystem.h>
#include <nn/am/service/window/am_Window.h>
#include <nn/am/service/window/am_IWindowTransiter.h>
#include <nn/am/service/am_DisplayLayerControl.h>
#include <nn/am/service/am_RightsEnvironment.h>
#include <nn/arp/arp_Api.h>
#include <nn/pdm/pdm_SystemTypes.h>

namespace nn { namespace am { namespace service {

/**
    @brief 機能統合されたアプレットの共通クラス

    @details
     コンストラクタで与えられる AppletSystem を参照して動く。
     親を持つことができ(親は nullptr でも可)、この場合 StackableApplet の仕組みで管理される。

     プロセス作成時(OnProcessBegin)に CreateAppletProxy でプロキシを作成し、
     これによって得られた AppletProxy を AppletSystem に登録する。
     プロセス終了時(OnProcessEnd)に、登録を解除する。
*/
class IntegratedApplet
    : public CallingWindableApplet
    , public std::enable_shared_from_this<IntegratedApplet>
{
protected:

    explicit IntegratedApplet(AppletSystem* pAppletSystem, bool hasWindow = true) NN_NOEXCEPT;

public:

    void SetParent(std::shared_ptr<IntegratedApplet> pParent) NN_NOEXCEPT;
    void SetFatalResultOnExit(Result result) NN_NOEXCEPT;
    bool IsDormant() NN_NOEXCEPT;

    virtual window::ForegroundMode GetForegroundMode() const NN_NOEXCEPT = 0;
    virtual GpuResourceGroupId GetGpuResourceGroupId() const NN_NOEXCEPT = 0;

    void PushMessageToProcess(applet::Message message) NN_NOEXCEPT
    {
        am::AppletMessage m = {};
        m.message[0] = message;
        this->PushMessageToProcess(m);
    }
    void NotifyControllerFirmwareUpdateSectionChanged() NN_NOEXCEPT;
    void NotifyVrModeCurtainRequired() NN_NOEXCEPT;

    virtual void SendPlayLog( pdm::AppletEventType eventType ) NN_NOEXCEPT;

    // pure
    virtual applet::AppletIdentityInfo GetAppletIdentityInfo() NN_NOEXCEPT = 0;

    IntegratedApplet* GetParent() NN_NOEXCEPT
    {
        return m_pParent.get();
    }

    bool IsReservedWithoutUnwinding() const NN_NOEXCEPT
    {
        return m_IsReservedWithoutUnwinding;
    }

    bool IsSelfProxyAvailable() NN_NOEXCEPT
    {
        return !!m_pSelfProxy.lock();
    }

    // 自アプレットの情報を Submit する。
    virtual Result SubmitApplicationInfo() NN_NOEXCEPT;

    // キーボード配列
    void SetDesirableKeyboardLayoutImpl(Bit32 layout) NN_NOEXCEPT;
    bool GetDesirableKeyboardLayoutImpl(Bit32* pOutLayout) NN_NOEXCEPT;

protected:

    // overrides(adding)
    virtual Result Before() NN_NOEXCEPT NN_OVERRIDE;
    virtual Result After() NN_NOEXCEPT NN_OVERRIDE;
    virtual void Cleanup() NN_NOEXCEPT NN_OVERRIDE;
    virtual void OnProcessBegin(os::ProcessId processId) NN_NOEXCEPT NN_OVERRIDE;
    virtual void OnProcessEnd(os::ProcessId processId) NN_NOEXCEPT NN_OVERRIDE;
    virtual void RunBehindProcess(process::NsProcess* pProcess) NN_NOEXCEPT NN_OVERRIDE;

    // virtual
    virtual void BeforeActivateProxy(os::ProcessId) NN_NOEXCEPT {}
    virtual void AfterDeactivateProxy(os::ProcessId) NN_NOEXCEPT {}

    // for derived
    void PushMessageToProcess(const am::AppletMessage& message) NN_NOEXCEPT;
    void OnGamePlayRecordingStateChanged() NN_NOEXCEPT;

    // request exit
    void RequestExit() NN_NOEXCEPT;
    bool IsExitRequested() NN_NOEXCEPT;
    void TerminateWithResult(Result terminateResult) NN_NOEXCEPT;

    // reservation
    Result ReserveToStartAndWaitAndUnwindThis(sf::SharedPointer<am::service::ILibraryAppletAccessor> p) NN_NOEXCEPT;
    Result ReserveToStartAndWait(sf::SharedPointer<am::service::ILibraryAppletAccessor> p) NN_NOEXCEPT;
    sf::SharedPointer<am::service::ILibraryAppletAccessor> GetReserved() NN_NOEXCEPT;

    AppletSystem* GetAppletSystem() NN_NOEXCEPT
    {
        return m_pAppletSystem;
    }

    IntegratedApplet* GetMainApplet() NN_NOEXCEPT
    {
        auto p = this;
        for (;;)
        {
            auto pParent = p->GetParent();
            if (!pParent)
            {
                return p;
            }
            p = pParent;
        }
    }

    class ProcessGateway;
    class AppletAccessorImpl;
    class AppletSelfProxyImpl;

    void LockExit() NN_NOEXCEPT;
    void UnlockExit() NN_NOEXCEPT;
    void UpdateExit() NN_NOEXCEPT;

    // idle
    idle::HandlingContext* GetIdleHandlingContext() NN_NOEXCEPT
    {
        return &m_IdleHandlingContext;
    }

private:

    AppletSystem* const m_pAppletSystem;
    std::shared_ptr<IntegratedApplet> m_pParent;
    Result m_FatalResultOnExit{ResultSuccess()};
    bool m_IsFatalResultOnExitValid{false};

    bool m_IsReservedWithoutUnwinding{false};
    sf::SharedPointer<am::service::ILibraryAppletAccessor> m_ReservedLibraryAppletAccessor;

    mutable os::Mutex m_StateMutex{true}; // recursive
    Result m_TerminateResult{ResultSuccess()};
    bool m_ExitLocked{false};
    bool m_ExitRequested{false};
    bool m_IsCompleted{false};
    void UpdateExitImpl(std::unique_lock<decltype(m_StateMutex)> lock) NN_NOEXCEPT;
    util::IntrusiveList<AppletAccessorImpl, util::IntrusiveListBaseNodeTraits<AppletAccessorImpl>> m_List;
    idle::HandlingContext m_IdleHandlingContext = idle::HandlingContext();

    Result GetIntegratedResult() NN_NOEXCEPT;

    bool IsExitRequested() const NN_NOEXCEPT;
    bool IsExitLocked() const NN_NOEXCEPT;
    void AddAccessor(AppletAccessorImpl* p) NN_NOEXCEPT;
    void RemoveAccessor(AppletAccessorImpl* p) NN_NOEXCEPT;
    bool IsCompleted() const NN_NOEXCEPT;

protected:
    // self proxy (m_StateMutex で保護)
    std::weak_ptr<AppletSelfProxyImpl> m_pSelfProxy;
    std::shared_ptr<AppletSelfProxyImpl> GetSelfProxy() NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_StateMutex)> lk(m_StateMutex);
        return m_pSelfProxy.lock();
    }

private:
    // for impl
    std::shared_ptr<IntegratedApplet> SharedFromThis() NN_NOEXCEPT
    {
        return std::static_pointer_cast<IntegratedApplet>(this->shared_from_this());
    }

    // キーボード配列
    os::Mutex m_DesirableKeyboardLayoutMutex{false};
    util::optional<Bit32> m_pDesirableKeyboardLayout;

public:

    Result CreateAppletProxy(std::shared_ptr<AppletProxy>* pOut, os::ProcessId processId) NN_NOEXCEPT;
    void SetProcessHandle(sf::NativeHandle processHandle) NN_NOEXCEPT;
    void SetAppletAttribute(const AppletAttribute* pAttribute) NN_NOEXCEPT;
    void ActivateWindow(const ProxyOption& proxyOption) NN_NOEXCEPT;

protected:

    typedef std::tuple<std::shared_ptr<AppletProxy>, std::shared_ptr<AppletSelfProxyImpl>> AppletProxyInfo;

    template <template <typename> class ProxyImpl, typename Impl>
    static AppletProxyInfo MakeAppletProxyInfo(std::shared_ptr<Impl> p) NN_NOEXCEPT
    {
        auto pAppletSelfProxyImpl = p;
        return std::make_tuple(MakeShared<ProxyImpl<decltype(p)>>(std::move(p)), std::move(pAppletSelfProxyImpl));
    }

private:

    // pure
    virtual AppletProxyInfo DoCreateAppletProxy(os::ProcessId processId) NN_NOEXCEPT = 0;

protected:

    // for impl utility
    Result MakeSystemProgramIdFromAppletId(ncm::SystemProgramId* pOut, applet::AppletId appletId) NN_NOEXCEPT;
    Result MakeProgramIdFromAppletIdentityInfo(ncm::ProgramId* pOut, const applet::AppletIdentityInfo& info) NN_NOEXCEPT;
    Result CreateAppletProcessImpl(std::shared_ptr<process::NsProcess>* pOut, applet::AppletId appletId) NN_NOEXCEPT;
    Result CreateApplicationProcessImpl(std::shared_ptr<process::NsProcess>* pOut, ncm::ApplicationId applicationId, uint8_t programIndex) NN_NOEXCEPT;
    Result CreateSystemApplicationProcessImpl(std::shared_ptr<process::NsProcess>* pOut, ncm::SystemApplicationId systemApplicationId) NN_NOEXCEPT;

public:

    void AttachProcessAndStart(os::ProcessId processId) NN_NOEXCEPT;

private:

    virtual Result CreateProcess(std::shared_ptr<process::NsProcess>* pOut) NN_NOEXCEPT NN_OVERRIDE final;
    virtual Result CreateProcessImpl(std::shared_ptr<process::NsProcess>* pOut) NN_NOEXCEPT = 0;
    virtual std::shared_ptr<process::NsProcess> MakeAttachedProcess(os::ProcessId processId) NN_NOEXCEPT = 0;

    std::shared_ptr<process::NsProcess> m_pAttachedProcess;

    void ReportProcessCreationFailure(nn::Result result) NN_NOEXCEPT;
    Result MakeCreateProcessFailureErrorReport(nn::Result result) NN_NOEXCEPT;

public:

    // for TORIAEZU
    void SetAutoMoveToTop() NN_NOEXCEPT
    {
        this->m_AutoMoveToTop = true;
    }

private:

    bool m_AutoMoveToTop{false};

private:

    // window support (これらの実装は am_IntegratedAppletWindow.cpp)

    os::Mutex m_WindowDeletionMutex{false};
    bool m_WindowShouldCreatedAsActive{true};
    window::Window* m_pWindow{nullptr};

    std::shared_ptr<window::WindowGroup> m_pWindowGroup;
    uint32_t m_WindowSubOrder{0};

    window::WindowManager::Updater BeginWindowUpdate() NN_NOEXCEPT;
    void CreateWindow() NN_NOEXCEPT;
    void DestroyWindow() NN_NOEXCEPT;
    void SetWindowTansiter(window::IWindowTransiter* pTransiter) NN_NOEXCEPT;
    void ClearWindowTansiter() NN_NOEXCEPT;

    // audio window
    void SetEffectiveVolumeImpl(float volume, TimeSpan fadeTime) NN_NOEXCEPT;

    // for audio controller
    os::Mutex m_ExpectedVolumeMutex{false};
    float m_ExpectedVolumeForMainApplet{0.25f};
    float m_ExpectedVolumeForLibraryApplet{1.f};
    void SetNextVolumeChangeFadeTime(float volume, TimeSpan fadeTime) NN_NOEXCEPT;
    void SetMainAppletWindowVolume(float volume) NN_NOEXCEPT;
    void SetTransparentVolumeRate(float transparentVolumeRate) NN_NOEXCEPT;

    // for capture
    virtual void OnLostForeground() NN_NOEXCEPT {}

    // for omm (HDCP)
    virtual void OnGetForeground() NN_NOEXCEPT {}

protected:
    virtual bool IsScreenShotEnabled() NN_NOEXCEPT;

    // 各アプレットの撮影許可禁止属性の設定
    // ただし、アプリだけはこれを使用せず、AppletSystem 側にだけ属性を保存
    virtual Result SetScreenShotPermissionImpl(applet::ScreenShotPermission permission) NN_NOEXCEPT;

    virtual bool IsCaptureTarget() NN_NOEXCEPT
    {
        return this->GetForegroundMode() == window::ForegroundMode::All;
    }

    void SetVideoCaptureValue(ns::VideoCapture videoCapture) NN_NOEXCEPT
    {
        this->m_VideoCaptureValue = videoCapture;
    }
    bool IsVideoCaptureAllowed() const NN_NOEXCEPT
    {
        return m_VideoCaptureValue != ns::VideoCapture::Disable;
    }
    bool IsVideoCaptureWorkBufferAutoAllocation() const NN_NOEXCEPT
    {
        return m_VideoCaptureValue == ns::VideoCapture::Enable;
    }
    void SetGamePlayRecordingForbiddenByBlackList(bool isFobidden) NN_NOEXCEPT
    {
        this->m_IsGamePlayRecordingForbiddenByBlackList = isFobidden;
    }
    bool IsGamePlayRecordingForbiddenByBlackList() const NN_NOEXCEPT
    {
        return m_IsGamePlayRecordingForbiddenByBlackList;
    }

private:
    applet::ScreenShotPermission m_ScreenShotPermission{applet::ScreenShotPermission_Inherit};
    ns::VideoCapture m_VideoCaptureValue = ns::VideoCapture::Manual;
    bool m_IsGamePlayRecordingForbiddenByBlackList = false;

protected:

    // virtual
    virtual window::WindowProperty GetInitialWindowProperty() NN_NOEXCEPT
    {
        window::WindowProperty ret = {};
        ret.globalOrder = window::WindowGlobalOrder::Normal;
        ret.isMainWindowVolumeRateValid = false;
        ret.handlesHomeButtonShortPressed = false;
        ret.handlesHomeButtonLongPressed = false;
        ret.isMainAppletWindow = true;
        ret.hidInputTakingMode = window::HidWindowController::HidInputTakingMode_TakeOnForeground;
        ret.handlesPerformanceModeChanged = false;
        ret.transparentVolumeRate = 1.f;
        ret.needsSetHandlingContext = false;
        return ret;
    }

    window::Window* GetWindow() NN_NOEXCEPT
    {
        return m_pWindow;
    }

public:

    // for Derived
    Result MoveWindowToTop() NN_NOEXCEPT;
    Result MoveWindowToScreenLock() NN_NOEXCEPT;
    Result MoveWindowToNormal() NN_NOEXCEPT;

private:

    // last Created LA (m_StateMutex で保護, CreateLibraryApplet() 時に登録)
    std::weak_ptr<IntegratedApplet> m_pLastCreatedLibraryApplet;

    // last LA (m_StateMutex で保護, StartLibraryApplet() 時に登録)
    std::weak_ptr<IntegratedApplet> m_pLastLibraryApplet;

public:

    void SetLastCreatedLibraryApplet(std::shared_ptr<IntegratedApplet> p) NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_StateMutex)> lk(m_StateMutex);
        this->m_pLastCreatedLibraryApplet = std::move(p);
    }

    std::shared_ptr<IntegratedApplet> GetLastCreatedLibraryApplet() NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_StateMutex)> lk(m_StateMutex);
        return this->m_pLastCreatedLibraryApplet.lock();
    }

    void SetLastLibraryApplet(std::shared_ptr<IntegratedApplet> p) NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_StateMutex)> lk(m_StateMutex);
        this->m_pLastCreatedLibraryApplet = p;
        this->m_pLastLibraryApplet = std::move(p);
    }

    std::shared_ptr<IntegratedApplet> GetLastLibraryApplet() NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_StateMutex)> lk(m_StateMutex);
        return this->m_pLastLibraryApplet.lock();
    }

private:

    virtual void OnGameCardStateChanged() NN_NOEXCEPT
    {
    }

private:

    // IndirectLayer
    class IndirectLayer
    {
    private:
        vi::IndirectLayerHandleType m_Handle{0};
    public:
        Result Create() NN_NOEXCEPT;
        ~IndirectLayer() NN_NOEXCEPT;
        vi::IndirectLayerHandleType GetHandle() const NN_NOEXCEPT
        {
            return m_Handle;
        }
    };
    IndirectLayer m_IndirectLayer;

    bool m_IsDestinationLayerIndirect{false};

protected:

    Result CreateIndirectLayer() NN_NOEXCEPT
    {
        return m_IndirectLayer.Create();
    }

    vi::IndirectLayerHandleType GetIndirectLayerHandle() NN_NOEXCEPT
    {
        return m_IndirectLayer.GetHandle();
    }

    void SetDestinationLayerIndirect(bool value) NN_NOEXCEPT
    {
        m_IsDestinationLayerIndirect = value;
    }

    bool IsDestinationLayerIndirect() const NN_NOEXCEPT
    {
        return m_IsDestinationLayerIndirect;
    }

private:

    virtual void OnHomeButtonLongPressed() NN_NOEXCEPT;

private:

    std::shared_ptr<service::rightsManagement::RightsEnvironment> m_pRightsEnvironment;
    mutable os::SdkMutex m_RightsEnvironmentStatusMutex;
    service::rightsManagement::RightsEnvironmentStatus m_RightsEnvironmentStatus = service::rightsManagement::GetInitialRightsEnvironmentStatus();
    void UpdateRightsEnvironmentStatus() NN_NOEXCEPT;
    bool CheckRightsEnvironmentAvailable() NN_NOEXCEPT;
    void RunBehindProcessForRights(process::NsProcess* pProcess) NN_NOEXCEPT;

    // pure
    virtual void OnRightsActivated(os::Tick expirationTick) NN_NOEXCEPT;
    virtual void OnRightsDeactivated() NN_NOEXCEPT;
    virtual void OnRightsAvailableTimeChanged(os::Tick expirationTick) NN_NOEXCEPT;

protected:

    void OverrideRightsEnvironment(std::shared_ptr<service::rightsManagement::RightsEnvironment> p) NN_NOEXCEPT;

};

/**
    @brief (他アプレットからの)アクセッサ実装


*/
class IntegratedApplet::AppletAccessorImpl
    : public util::IntrusiveListBaseNode<AppletAccessorImpl>
{
    friend IntegratedApplet;
protected:

    explicit AppletAccessorImpl(std::shared_ptr<IntegratedApplet> p) NN_NOEXCEPT;
    ~AppletAccessorImpl() NN_NOEXCEPT;

    IntegratedApplet* GetApplet() const NN_NOEXCEPT
    {
        return m_P.get();
    }

    virtual void OnStartByAccessor() NN_NOEXCEPT
    {
    }

public:

    // IAppletAccessor
    Result GetAppletStateChangedEvent(sf::Out<sf::NativeHandle> pOut) NN_NOEXCEPT;
    Result IsCompleted(sf::Out<bool> pState) NN_NOEXCEPT;
    Result Start() NN_NOEXCEPT;
    Result RequestExit() NN_NOEXCEPT;
    Result Terminate() NN_NOEXCEPT;
    Result GetResult() NN_NOEXCEPT;
    Result GetIntegratedAppletPointer(sf::Out<IntegratedAppletPointer> p) NN_NOEXCEPT;
    Result SetUsers(bool allUser, const sf::InArray<account::Uid>& uids) NN_NOEXCEPT;
    Result CheckRightsEnvironmentAvailable(sf::Out<bool> pOut) NN_NOEXCEPT;
    Result GetNsRightsEnvironmentHandle(sf::Out<Bit64> pOut) NN_NOEXCEPT;

private:

    std::shared_ptr<IntegratedApplet> m_P;
    os::SystemEvent m_Event{os::EventClearMode_ManualClear, true};

private:

    vi::IndirectConsumerHandleType m_IndirectLayerConsumerHandle{0};

protected:

    Result GetIndirectLayerConsumerHandle(sf::Out<vi::IndirectConsumerHandleType> pOut, applet::AppletResourceUserId aruid) NN_NOEXCEPT;

};

enum HomeButtonShortPressedHandleType : uint8_t
{
    HomeButtonShortPressedHandleType_EnqueueMessage,
    HomeButtonShortPressedHandleType_NotifyBlockingHomeButton,
    HomeButtonShortPressedHandleType_DoNothing,
};

class IntegratedApplet::AppletSelfProxyImpl
    : public std::enable_shared_from_this<AppletSelfProxyImpl>
{
    friend class IntegratedApplet;
public:

    explicit AppletSelfProxyImpl(std::shared_ptr<IntegratedApplet> p, os::ProcessId processId) NN_NOEXCEPT;
    ~AppletSelfProxyImpl() NN_NOEXCEPT;
    void SetProcessHandle(sf::NativeHandle processHandle) NN_NOEXCEPT;
    void SetHandlesHomeButtonShortPressed(HomeButtonShortPressedHandleType type) NN_NOEXCEPT;
    void SetHandlesHomeButtonLongPressed(bool handlesHomeButtonLongPressed) NN_NOEXCEPT;
    void SetHandlesPowerButtonShortPressed(bool handlesPowerButtonShortPressed) NN_NOEXCEPT;
    void ActivateWindow(const ProxyOption& proxyOption) NN_NOEXCEPT;
    void NotifyControllerFirmwareUpdateSectionChangedImpl() NN_NOEXCEPT;
    void NotifyVrModeCurtainRequiredImpl() NN_NOEXCEPT;
    void OnGamePlayRecordingStateChangedImpl() NN_NOEXCEPT;

    // GetCommonStateGetter
    Result GetEventHandle(sf::Out<sf::NativeHandle> pOut) NN_NOEXCEPT;
    Result ReceiveMessage(sf::Out<am::AppletMessage> pOut) NN_NOEXCEPT;
    Result GetCurrentFocusState(sf::Out<Bit8> focusState) NN_NOEXCEPT;
    Result GetThisAppletKind(sf::Out<am::service::AppletKind> pOut) NN_NOEXCEPT;
    Result AllowToEnterSleep() NN_NOEXCEPT;
    Result DisallowToEnterSleep() NN_NOEXCEPT;
    Result GetOperationMode(sf::Out<Bit8> pOut) NN_NOEXCEPT;
    Result GetDefaultDisplayResolution(sf::Out<int> pOutWidth, sf::Out<int> pOutHeight) NN_NOEXCEPT;
    Result GetDefaultDisplayResolutionChangeEvent(sf::Out<sf::NativeHandle> outValue) NN_NOEXCEPT;
    Result GetHdcpAuthenticationState(sf::Out<int> outValue) NN_NOEXCEPT;
    Result GetHdcpAuthenticationStateChangeEvent(sf::Out<sf::NativeHandle> outValue) NN_NOEXCEPT;
    Result SetTvPowerStateMatchingMode(int mode) NN_NOEXCEPT;
    Result IsVrModeEnabled(sf::Out<bool> pOut) NN_NOEXCEPT;
    Result IsInControllerFirmwareUpdateSection(sf::Out<bool> pOut) NN_NOEXCEPT;
    Result SetVrModeEnabled(bool isEnabled) NN_NOEXCEPT;
    Result SetLcdBacklighOffEnabled(bool isLcdBacklightOff) NN_NOEXCEPT;
    Result GetCradleStatus(sf::Out<Bit8> pOut) NN_NOEXCEPT;
    Result GetCradleFwVersion(nn::sf::Out<std::uint32_t> pOutPdcH, nn::sf::Out<std::uint32_t> pOutPdcA, nn::sf::Out<std::uint32_t> pOutMcu, nn::sf::Out<std::uint32_t> pOutDp2Hdmi) NN_NOEXCEPT;
    Result GetPerformanceMode(sf::Out<Bit32> pOut) NN_NOEXCEPT;
    Result GetBootMode(sf::Out<Bit8> pOut) NN_NOEXCEPT;
    Result RequestToAcquireSleepLock() NN_NOEXCEPT;
    Result ReleaseSleepLock() NN_NOEXCEPT;
    Result ReleaseSleepLockTransiently() NN_NOEXCEPT;
    Result GetAcquiredSleepLockEvent(sf::Out<sf::NativeHandle> pOut) NN_NOEXCEPT;
    virtual Result PushToGeneralChannel(sf::SharedPointer<am::service::IStorage> storage) NN_NOEXCEPT;
    virtual Result GetApplicationIdByContentActionName(sf::Out<Bit64> pOut, const sf::InArray<char>& name) NN_NOEXCEPT;
    std::shared_ptr<AppletSelfProxyImpl> GetCommonStateGetter() NN_NOEXCEPT
    {
        return std::static_pointer_cast<AppletSelfProxyImpl>(shared_from_this());
    }

    // GetSelfController
    virtual Result Exit() NN_NOEXCEPT;
    Result LockExit() NN_NOEXCEPT;
    Result UnlockExit()NN_NOEXCEPT;
    Result EnterFatalSection() NN_NOEXCEPT;
    Result LeaveFatalSection() NN_NOEXCEPT;
    Result SetRequiresCaptureButtonShortPressedMessage(bool isRequired) NN_NOEXCEPT;
    Result SetScreenShotAppletIdentityInfo(AppletIdentityInfo info) NN_NOEXCEPT;
    Result SetScreenShotPermission(int permission) NN_NOEXCEPT;
    Result SetAlbumImageOrientation(int orientation) NN_NOEXCEPT;
    Result SetOperationModeChangedNotification(bool needNotify) NN_NOEXCEPT;
    Result SetPerformanceModeChangedNotification(bool needNotify) NN_NOEXCEPT;
    Result SetNotifiesFocusStateChanged(bool notifiesFocusStateChanged) NN_NOEXCEPT;
    Result SetFocusHandlingMode(bool focus, bool bg, bool pause) NN_NOEXCEPT;
    Result SetOutOfFocusSuspendingEnabled(bool isEnabled) NN_NOEXCEPT;
    Result SetRestartMessageEnabled(bool enabled) NN_NOEXCEPT;
    Result CreateManagedDisplayLayer(sf::Out<Bit64> pOutLayerId) NN_NOEXCEPT;
    Result IsSystemBufferSharingEnabled() NN_NOEXCEPT;
    Result GetSystemSharedLayerHandle(sf::Out<vi::fbshare::SharedBufferHandle> pOutBufferHandle, sf::Out<vi::fbshare::SharedLayerHandle> pOutLayerHandle) NN_NOEXCEPT;
    Result GetSystemSharedBufferHandle(sf::Out<vi::fbshare::SharedBufferHandle> pOutBufferHandle) NN_NOEXCEPT;
    Result SetHandlesRequestToDisplay(bool handlesRequestToDisplay) NN_NOEXCEPT;
    Result ApproveToDisplay() NN_NOEXCEPT;
    Result OverrideAutoSleepTimeAndDimmingTime(int autoSleepTimeInHandheld, int autoSleepTimeInConsole, int dimmingTimeInHandheld, int dimmingTimeInConsole) NN_NOEXCEPT;
    virtual Result SetMediaPlaybackState(bool mediaPlaybackState) NN_NOEXCEPT;
    Result IsGamePlayRecordingSupported(sf::Out<bool> pOut) NN_NOEXCEPT;
    Result InitializeGamePlayRecording(sf::NativeHandle transferMemoryHandle, uint64_t transferMemorySize) NN_NOEXCEPT;
    Result SetGamePlayRecordingState(int state) NN_NOEXCEPT;
    Result RequestFlushGamePlayingMovieForDebug() NN_NOEXCEPT;
    Result SetControllerFirmwareUpdateSection(bool isInSection) NN_NOEXCEPT;
    Result SetIdleTimeDetectionExtension(uint32_t extentionType) NN_NOEXCEPT;
    Result GetIdleTimeDetectionExtension(sf::Out<uint32_t> pOut) NN_NOEXCEPT;
    Result GetLibraryAppletLaunchableEvent(sf::Out<sf::NativeHandle> pOut) NN_NOEXCEPT;
    Result SetInputDetectionSourceSet(Bit32 inputSources) NN_NOEXCEPT;
    Result SetAutoSleepDisabled(bool isDisabled) NN_NOEXCEPT;
    Result IsAutoSleepDisabled(sf::Out<bool> pOut) NN_NOEXCEPT;
    Result ReportUserIsActive() NN_NOEXCEPT;
    Result GetCurrentIlluminance(sf::Out<float> pOut) NN_NOEXCEPT;
    Result GetCurrentIlluminanceEx(sf::Out<float> pOut, sf::Out<bool> pOutOverflowAlert) NN_NOEXCEPT;
    Result IsIlluminanceAvailable(sf::Out<bool> pOut) NN_NOEXCEPT;
    Result ReportMultimediaError(nn::Bit32 resultValue, const nn::sf::InBuffer& multimediaTelemetry) NN_NOEXCEPT;
    std::shared_ptr<AppletSelfProxyImpl> GetSelfController() NN_NOEXCEPT
    {
        return std::static_pointer_cast<AppletSelfProxyImpl>(shared_from_this());
    }
    Result SetDesirableKeyboardLayout(Bit32 layout) NN_NOEXCEPT;
    Result SetWirelessPriorityMode(int mode) NN_NOEXCEPT;

    // GetLibraryAppletCreator
    Result CreateLibraryApplet(sf::Out<sf::SharedPointer<am::service::ILibraryAppletAccessor>> pOut, Bit32 appletId, uint32_t libraryAppletMode) NN_NOEXCEPT;
    Result TerminateAllLibraryApplets() NN_NOEXCEPT;
    Result AreAnyLibraryAppletsLeft(sf::Out<bool> pOut) NN_NOEXCEPT;
    Result CreateStorage(sf::Out<sf::SharedPointer<am::service::IStorage>> pOut, std::int64_t size) NN_NOEXCEPT;
    Result CreateTransferMemoryStorage(sf::Out<sf::SharedPointer<am::service::IStorage>> pOut, sf::NativeHandle transferMemory, int64_t size, bool isWritable) NN_NOEXCEPT;
    Result CreateHandleStorage(sf::Out<sf::SharedPointer<am::service::IStorage>> pOut, sf::NativeHandle transferMemory, int64_t size) NN_NOEXCEPT;
    std::shared_ptr<AppletSelfProxyImpl> GetLibraryAppletCreator() NN_NOEXCEPT
    {
        return std::static_pointer_cast<AppletSelfProxyImpl>(shared_from_this());
    }

    // GetWindowController
    Result CreateWindow(sf::Out<sf::SharedPointer<am::service::IWindow>> pOut, am::service::WindowCreationOption option) NN_NOEXCEPT;
    Result GetAppletResourceUserId(sf::Out<am::AppletResourceUserId> pOut) NN_NOEXCEPT;
    Result GetAppletResourceUserIdOfCallerApplet(sf::Out<am::AppletResourceUserId> pOut) NN_NOEXCEPT;
    Result AcquireForegroundRights() NN_NOEXCEPT;
    Result ReleaseForegroundRights() NN_NOEXCEPT;
    Result RejectToChangeIntoBackground() NN_NOEXCEPT;
    std::shared_ptr<AppletSelfProxyImpl> GetWindowController() NN_NOEXCEPT
    {
        return std::static_pointer_cast<AppletSelfProxyImpl>(shared_from_this());
    }

    // GetAudioController
    Result SetExpectedMasterVolume(float mainAppletVolume, float libraryAppletVolume) NN_NOEXCEPT;
    Result GetMainAppletExpectedMasterVolume(sf::Out<float> pOutVolume) NN_NOEXCEPT;
    Result GetLibraryAppletExpectedMasterVolume(sf::Out<float> pOutVolume) NN_NOEXCEPT;
    Result ChangeMainAppletMasterVolume(float volume, std::int64_t fadeTimeInNanoSeconds) NN_NOEXCEPT;
    Result SetTransparentVolumeRate(float transparentVolumeRate) NN_NOEXCEPT;
    std::shared_ptr<AppletSelfProxyImpl> GetAudioController() NN_NOEXCEPT
    {
        return std::static_pointer_cast<AppletSelfProxyImpl>(shared_from_this());
    }

    // GetDisplayController
    Result GetLastApplicationCaptureImageEx(sf::Out<bool> pOutIsScreenShotEnabled, sf::OutBuffer pOutBuffer) NN_NOEXCEPT;
    Result GetLastForegroundCaptureImageEx(sf::Out<bool> pOutIsScreenShotEnabled, sf::OutBuffer pOutBuffer) NN_NOEXCEPT;
    Result GetCallerAppletCaptureImageEx(sf::Out<bool> pOutIsScreenShotEnabled, sf::OutBuffer pOutBuffer) NN_NOEXCEPT;

    Result GetLastApplicationCaptureImage(sf::OutBuffer pOutBuffer) NN_NOEXCEPT;
    Result GetLastForegroundCaptureImage(sf::OutBuffer pOutBuffer) NN_NOEXCEPT;
    Result UpdateLastForegroundCaptureImage() NN_NOEXCEPT;
    Result GetCallerAppletCaptureImage(sf::OutBuffer pOutBuffer) NN_NOEXCEPT;
    Result UpdateCallerAppletCaptureImage() NN_NOEXCEPT;
    Result TakeScreenShotOfOwnLayer(int index, bool isScreenShotPermitted) NN_NOEXCEPT;
    Result TakeScreenShotOfOwnLayerEx(int index, bool isScreenShotPermitted, bool isImmediate) NN_NOEXCEPT;
    Result CopyBetweenCaptureBuffers(int dstIndex, int srcIndex) NN_NOEXCEPT;

    Result AcquireLastApplicationCaptureBufferEx(sf::Out<bool> pOutIsScreenShotEnabled, sf::Out<sf::NativeHandle> pOut) NN_NOEXCEPT;
    Result AcquireLastForegroundCaptureBufferEx(sf::Out<bool> pOutIsScreenShotEnabled, sf::Out<sf::NativeHandle> pOut) NN_NOEXCEPT;
    Result AcquireCallerAppletCaptureBufferEx(sf::Out<bool> pOutIsScreenShotEnabled, sf::Out<sf::NativeHandle> pOut) NN_NOEXCEPT;

    Result AcquireLastApplicationCaptureBuffer(sf::Out<sf::NativeHandle> pOut) NN_NOEXCEPT;
    Result ReleaseLastApplicationCaptureBuffer() NN_NOEXCEPT;
    Result AcquireLastForegroundCaptureBuffer(sf::Out<sf::NativeHandle> pOut) NN_NOEXCEPT;
    Result ReleaseLastForegroundCaptureBuffer() NN_NOEXCEPT;
    Result AcquireCallerAppletCaptureBuffer(sf::Out<sf::NativeHandle> pOut) NN_NOEXCEPT;
    Result ReleaseCallerAppletCaptureBuffer() NN_NOEXCEPT;

    Result AcquireLastApplicationCaptureSharedBuffer(sf::Out<bool> pOutIsScreenShotEnabled, sf::Out<int> pOut) NN_NOEXCEPT;
    Result ReleaseLastApplicationCaptureSharedBuffer() NN_NOEXCEPT;
    Result AcquireLastForegroundCaptureSharedBuffer(sf::Out<bool> pOutIsScreenShotEnabled, sf::Out<int> pOut) NN_NOEXCEPT;
    Result ReleaseLastForegroundCaptureSharedBuffer() NN_NOEXCEPT;
    Result AcquireCallerAppletCaptureSharedBuffer(sf::Out<bool> pOutIsScreenShotEnabled, sf::Out<int> pOut) NN_NOEXCEPT;
    Result ReleaseCallerAppletCaptureSharedBuffer() NN_NOEXCEPT;

    Result ClearCaptureBuffer(int index, bool isScreenShotPermitted, Bit32 clearColor) NN_NOEXCEPT;
    Result ClearAppletTransitionBuffer(Bit32 clearColor) NN_NOEXCEPT;

    Result StartContinuousRecordingFlushForDebug(sf::Out<sf::NativeHandle> pOut, int64_t maxTimeInNanoSeconds) NN_NOEXCEPT;

    std::shared_ptr<AppletSelfProxyImpl> GetDisplayController() NN_NOEXCEPT
    {
        return std::static_pointer_cast<AppletSelfProxyImpl>(shared_from_this());
    }

    // GetDebugFunctions
    AppletSystem* GetDebugFunctions() NN_NOEXCEPT
    {
        return GetApplet()->GetAppletSystem();
    }

    // to ProcessProxy
    bool IsSuspended() NN_NOEXCEPT;
    void RequestExit() NN_NOEXCEPT;
    void PushMessage(const AppletMessage& message) NN_NOEXCEPT;
    void SetNextVolumeChangeFadeTime(float volume, TimeSpan fadeTime) NN_NOEXCEPT;

    Result GetHomeButtonWriterLockAccessor(sf::Out<sf::SharedPointer<ILockAccessor>> pOut) NN_NOEXCEPT;
    Result GetHomeButtonReaderLockAccessor(sf::Out<sf::SharedPointer<ILockAccessor>> pOut) NN_NOEXCEPT;
    Result GetWriterLockAccessorEx(sf::Out<sf::SharedPointer<ILockAccessor>> pOut, int index) NN_NOEXCEPT;
    Result GetReaderLockAccessorEx(sf::Out<sf::SharedPointer<ILockAccessor>> pOut, int index) NN_NOEXCEPT;

protected:

    IntegratedApplet* GetApplet() NN_NOEXCEPT
    {
        return m_P.get();
    }

    void InvalidateDisplayLayer() NN_NOEXCEPT;

    Result GetIndirectLayerProducerHandle(sf::Out<vi::IndirectProducerHandleType> pOut) NN_NOEXCEPT;

private:

    std::shared_ptr<IntegratedApplet> m_P;
    CleanupLock m_CleanupLock;

    std::shared_ptr<ProcessGateway> m_pProcessGateway;
    bool    m_IsLastApplicationCaptureBufferOwner{false};
    bool    m_IsLastForegroundCaptureBufferOwner{false};
    bool    m_IsCallerAppletCaptureBufferOwner{false};
    os::Mutex m_MediaPlaybackStateMutex{false};

    os::Mutex m_FatalSectionCountMutex{false};
    uint64_t m_FatalSectionCount{0};

};

class IntegratedMainApplet
    : public IntegratedApplet
{
protected:

    virtual window::ForegroundMode GetForegroundMode() const NN_NOEXCEPT NN_OVERRIDE
    {
        return window::ForegroundMode::All;
    }

    explicit IntegratedMainApplet(AppletSystem* pAppletSystem) NN_NOEXCEPT
        : IntegratedApplet(pAppletSystem)
    {
    }

    class MainAppletAccessorImpl
        : public AppletAccessorImpl
    {
    protected:

        explicit MainAppletAccessorImpl(std::shared_ptr<IntegratedApplet> p) NN_NOEXCEPT
            : AppletAccessorImpl(std::move(p))
        {
        }

    };

    class MainAppletSelfProxyImpl
        : public AppletSelfProxyImpl
    {
    protected:

        explicit MainAppletSelfProxyImpl(std::shared_ptr<IntegratedApplet> p, os::ProcessId processId) NN_NOEXCEPT
            : AppletSelfProxyImpl(std::move(p), processId)
        {
        }

    };

};

}}}
