﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>

#include <map>
#include <nn/util/util_Optional.h>
#include <nn/sf/sf_NativeHandle.h>
#include <nn/am/service/am_ServiceStaticAllocator.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/am/service/am_AppletProxy.h>
#include <nn/am/service/core/am_Applet.h>
#include <nn/am/service/process/am_NsProcess.h>
#include <nn/am/service/window/am_WindowSystem.h>
#include <nn/am/service/am_GlobalStateManager.h>
#include <nn/am/service/am_Storage.h>
#include <nn/am/service/am_Lock.h>
#include <nn/applet/applet_Types.h>
#include <nn/omm/omm_Api.h>

#include <list>
#include <atomic>

namespace nn { namespace am { namespace service {

class AppletSystem;
class IntegratedApplet;
class IntegratedApplication;
class IntegratedSystemApplet;
class IntegratedOverlayApplet;

namespace detail {

class AppletProxyManager
{
public:

    // for applet implementation
    void RegisterAppletProxy(nn::Bit64 processId, IntegratedApplet* p) NN_NOEXCEPT;
    void UnregisterAppletProxy(nn::Bit64 processId) NN_NOEXCEPT;

    // for AppletService
    Result GetAppletProxy(std::shared_ptr<AppletProxy>* pOut, nn::Bit64 processId, sf::NativeHandle processHandle, const ProxyOption& proxyOption, const AppletAttribute* pAttribute) NN_NOEXCEPT;

private:

    struct Entry
    {
        IntegratedApplet* p{nullptr};
    };

    typedef std::pair<const Bit64, Entry> KeyValue;
    typedef std::map<KeyValue::first_type, KeyValue::second_type, std::less<KeyValue::first_type>, StaticAllocator<KeyValue>> Map;
    os::Mutex m_MapMutex{false};
    Map m_Map;

};

class AppletCleaner
{
public:

    void RegisterApplet(std::shared_ptr<core::Applet> pApplet) NN_NOEXCEPT;
    void CleanApplets() NN_NOEXCEPT;

private:

    os::Mutex m_Mutex{false};
    std::list<std::shared_ptr<core::Applet>, StaticAllocator<std::shared_ptr<core::Applet>>> m_Applets;

};

class FloatingApplicationList
{
    friend AppletSystem;
public:

    void PushApplication(std::shared_ptr<IntegratedApplication> pApplication) NN_NOEXCEPT;
    sf::SharedPointer<am::service::IApplicationAccessor> PopApplication() NN_NOEXCEPT;
    void CleanApplications() NN_NOEXCEPT;

private:

    void ClearAllImpl() NN_NOEXCEPT;
    void RegisterObserver(std::shared_ptr<IntegratedSystemApplet> pObserver) NN_NOEXCEPT;

    os::Mutex m_Mutex{false};
    std::list<std::weak_ptr<IntegratedApplication>, StaticAllocator<std::weak_ptr<IntegratedApplication>>> m_List;
    std::weak_ptr<IntegratedSystemApplet> m_pObserver;

};

class ApplicationObserver
{
    friend AppletSystem;
public:

    void NotifyApplicationExit() NN_NOEXCEPT;

private:

    void RegisterObserver(std::shared_ptr<IntegratedSystemApplet> pObserver) NN_NOEXCEPT;

    os::Mutex m_Mutex{false};
    std::weak_ptr<IntegratedSystemApplet> m_pObserver;
};

class GlobalStateManagerForAppletSystem
    : public GlobalStateManager
{
public:

    explicit GlobalStateManagerForAppletSystem(window::WindowManager* pWindowManager) NN_NOEXCEPT
        : GlobalStateManager(pWindowManager)
    {
    }

    void SetNotifier(std::shared_ptr<IntegratedApplet> pApplet) NN_NOEXCEPT
    {
        this->m_pApplet = pApplet;
    }

private:

    std::weak_ptr<IntegratedApplet> m_pApplet;

    virtual void OnFinishedSleepSequence() NN_NOEXCEPT NN_OVERRIDE;

};

class ResourceReservationWindowManager
{
private:

    window::WindowManager* m_pWindowManager;
    window::Window* m_pWindow;

public:

    explicit ResourceReservationWindowManager(window::WindowManager* pWindowManager) NN_NOEXCEPT;
    ~ResourceReservationWindowManager() NN_NOEXCEPT;

};

class BackgroundWindowManager
    : public window::IWindowTransiter
{
private:

    AppletSystem* m_pAppletSystem;
    window::WindowManager* m_pWindowManager;
    window::Window* m_pWindow;

    virtual void OnHomeButtonLongPressed() NN_NOEXCEPT NN_OVERRIDE;
    virtual void OnCaptureButtonShortPressed() NN_NOEXCEPT NN_OVERRIDE;
    virtual void OnCaptureButtonLongPressed() NN_NOEXCEPT NN_OVERRIDE;
    virtual void OnVrModeCurtainRequired() NN_NOEXCEPT NN_OVERRIDE;

public:

    explicit BackgroundWindowManager(AppletSystem* pAppletSystem, window::WindowManager* pWindowManager) NN_NOEXCEPT;
    ~BackgroundWindowManager() NN_NOEXCEPT;

    void SetDefaultHomeButtonLongPressTime(TimeSpan time) NN_NOEXCEPT;

};

}

class AppletSystem
{
    friend detail::BackgroundWindowManager;
public:

    // プロキシの登録と削除
    // for IntegratedApplet
    void BeginProcessCreation() NN_NOEXCEPT;
    void EndProcessCreation() NN_NOEXCEPT;
    void RegisterAppletProxy(nn::Bit64 processId, IntegratedApplet* p) NN_NOEXCEPT;
    void UnregisterAppletProxy(nn::Bit64 processId) NN_NOEXCEPT;

    // プロキシの取得
    // for AppletService
    Result GetAppletProxy(std::shared_ptr<AppletProxy>* pOut, nn::Bit64 processId, sf::NativeHandle handle, const ProxyOption& proxyOption, const AppletAttribute* pAttribute) NN_NOEXCEPT;

    // アプレットの作成
    Result CreateLibraryApplet(sf::SharedPointer<am::service::ILibraryAppletAccessor>* pOut, Bit32 appletId, applet::LibraryAppletMode libraryAppletMode, std::shared_ptr<IntegratedApplet> pParent) NN_NOEXCEPT;
    Result CreateApplication(sf::SharedPointer<am::service::IApplicationAccessor>* pOut, ncm::ApplicationId id, std::shared_ptr<IntegratedApplet> pParent, const applet::ApplicationLaunchRequestInfo& launchRequestInfo) NN_NOEXCEPT;
    Result CreateApplicationForReserved(std::shared_ptr<IntegratedApplication>* pOut, ncm::ApplicationId id, const applet::ApplicationLaunchRequestInfo& launchRequestInfo) NN_NOEXCEPT;
    Result CreateSystemApplication(sf::SharedPointer<am::service::IApplicationAccessor>* pOut, ncm::SystemApplicationId id, std::shared_ptr<IntegratedApplet> pParent) NN_NOEXCEPT;

    // アプリメモリを拝借してシステムメモリをブーストし、メモリヒープに設定
    Result AllocateMemoryFromApplicationResource(uintptr_t* pOutAddress, size_t size) NN_NOEXCEPT;
    void FreeMemoryToApplicationResource(uintptr_t address) NN_NOEXCEPT;

    // アプレットの作成(内部用)
    Result CreateSystemApplet(std::shared_ptr<core::Applet>* pOut, Bit32 appletId) NN_NOEXCEPT;
    Result CreateOverlayApplet(std::shared_ptr<core::Applet>* pOut, Bit32 appletId) NN_NOEXCEPT;

    // アプリケーション起動要求
    void PushLaunchRequestedApplication(std::shared_ptr<IntegratedApplication> pApplication) NN_NOEXCEPT;

    // 試遊台関連
    void NotifyGoBackQuestMenu() NN_NOEXCEPT;

    // 汎用チャンネル
    StorageChannel* GetGeneralChannel() NN_NOEXCEPT
    {
        return &m_GeneralChannel;
    }

    // アプリケーションの撮影禁止状態（他アプレット分は含まれない）
    bool IsApplicationScreenShotEnabled() NN_NOEXCEPT
    {
        return m_ApplicationScreenShotPermission;
    }
    void SetApplicationScreenShotPermission(bool isEnabled) NN_NOEXCEPT
    {
        m_ApplicationScreenShotPermission = isEnabled;
    }

    // コントローラ F/W 更新期間
    bool IsInControllerFirmwareUpdateSection() NN_NOEXCEPT
    {
        return m_InControllerFirmwareUpdateSection;
    }
    Result SetControllerFirmwareUpdateSection(bool isInSection) NN_NOEXCEPT;

    // アプリケーションによる VR モード切替え
    bool IsVrModeEnabledImpl() NN_NOEXCEPT
    {
        return m_VrModeEnabled;
    }
    void SetVrModeEnabledImpl(bool isEnabled, bool fromOe) NN_NOEXCEPT;

    // 動作モード
    void SetOperationMode(omm::OperationMode mode) NN_NOEXCEPT;
    omm::OperationMode GetOperationMode() NN_NOEXCEPT;

    void SetDefaultHomeButtonLongPressTime(TimeSpan time) NN_NOEXCEPT
    {
        m_BackgroundWindowManager.SetDefaultHomeButtonLongPressTime(time);
    }

    // アプレット実装用(プロセス作成)
    // 外部用
    process::NsProcessManager* GetNsProcessManager() NN_NOEXCEPT
    {
        return &m_NsProcessManager;
    }

    window::WindowManager* GetWindowManager() NN_NOEXCEPT
    {
        return &m_WindowManager;
    }

    detail::ApplicationObserver* GetApplicationObserver() NN_NOEXCEPT
    {
        return &m_ApplicationObserver;
    }

    detail::GlobalStateManagerForAppletSystem* GetGlobalStateManager() NN_NOEXCEPT
    {
        return &m_GlobalStateManager;
    }

    sf::SharedPointer<am::service::IApplicationAccessor> PopFloatingApplication() NN_NOEXCEPT
    {
        return m_FloatingApplicationList.PopApplication();
    }

    void CleanApplets() NN_NOEXCEPT
    {
        m_AppletCleaner.CleanApplets();
    }

    // プロセスが先に起動したアプレットの作成(開発用)
    Result CreateApplicationForRunningProcess(std::shared_ptr<core::Applet>* pOut, os::ProcessId processId, const applet::ApplicationLaunchRequestInfo& launchRequestInfo) NN_NOEXCEPT;
    Result CreateSystemApplicationForRunningProcess(std::shared_ptr<core::Applet>* pOut, os::ProcessId processId) NN_NOEXCEPT;
    Result CreateLibraryAppletForRunningProcess(std::shared_ptr<core::Applet>* pOut, Bit32 appletId, os::ProcessId processId) NN_NOEXCEPT;
    Result CreateSystemAppletForRunningProcess(std::shared_ptr<core::Applet>* pOut, Bit32 appletId, os::ProcessId processId) NN_NOEXCEPT;
    Result CreateOverlayAppletForRunningProcess(std::shared_ptr<core::Applet>* pOut, Bit32 appletId, os::ProcessId processId) NN_NOEXCEPT;

    // 自分をプロキシとする LA を作成するような LACreator を作成する
    Result CreateSelfLibraryAppletCreatorForDevelop(sf::Out<sf::SharedPointer<ILibraryAppletCreator>> pOut, os::ProcessId processId) NN_NOEXCEPT;

    // ReaderWriterLock の取得
    ReaderWriterLock& GetHomeButtonReaderWriterLock() NN_NOEXCEPT
    {
        return *m_pReaderWriterLockArray[applet::ReaderWriterLockIndex_HomeButton];
    }
    ReaderWriterLock& GetReaderWriterLockEx(int index) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(index >= 0 && index < applet::ReaderWriterLockIndex_Max);
        return *m_pReaderWriterLockArray[index];
    }

    // IDebugFunctions
    Result NotifyMessageToHomeMenuForDebug(am::AppletMessage message) NN_NOEXCEPT;
    Result OpenMainApplication(sf::Out<sf::SharedPointer<am::service::IApplicationAccessor>> pOut) NN_NOEXCEPT;
    Result EmulateButtonEvent(EmulatedButtonEvent e) NN_NOEXCEPT;
    Result InvalidateTransitionLayer() NN_NOEXCEPT;

    // システム制御
    Result RequestToShutdown() NN_NOEXCEPT;
    Result RequestToReboot() NN_NOEXCEPT;

    // 通知
    void NotifyApplicationStarted(const ncm::ApplicationId& applicationId) NN_NOEXCEPT;
    void NotifyApplicationRunning(const ncm::ApplicationId& applicationId) NN_NOEXCEPT;
    void NotifyApplicationExited(const ncm::ApplicationId& applicationId) NN_NOEXCEPT;
    void NotifyApplicationSuspendedByRightsError() NN_NOEXCEPT;
    void NotifyControllerFirmwareUpdateSectionChanged() NN_NOEXCEPT;
    void NotifyHomeButtonLongPressedToOverlayApplet() NN_NOEXCEPT;

    // アプリケーション制御
    void TerminateCurrentApplicationAndSetResult(Result result) NN_NOEXCEPT;
    bool AreAnyApplicationActive() NN_NOEXCEPT;

    std::shared_ptr<IntegratedSystemApplet> GetTheSystemApplet() NN_NOEXCEPT
    {
        return m_pTheSystemApplet.lock();
    }

private:

    detail::AppletProxyManager m_AppletProxyManager;
    detail::AppletCleaner m_AppletCleaner;
    detail::FloatingApplicationList m_FloatingApplicationList;
    detail::ApplicationObserver m_ApplicationObserver;
    process::NsProcessManager m_NsProcessManager;
    window::WindowManager m_WindowManager;
    std::weak_ptr<IntegratedApplication> m_pLastCreatedApplication;
    std::weak_ptr<IntegratedSystemApplet> m_pTheSystemApplet;
    std::weak_ptr<IntegratedOverlayApplet> m_pTheOverlayApplet;
    bool m_ApplicationScreenShotPermission{true};
    bool m_VrModeEnabled{false};
    bool m_InControllerFirmwareUpdateSection{false};
    os::Mutex m_VrModeMutex{false};
    os::Mutex m_InControllerFirmwareUpdateSectionMutex{false};
    os::SdkMutex m_OperationModeMutex;
    util::optional<omm::OperationMode> m_pOperationMode;
    detail::GlobalStateManagerForAppletSystem m_GlobalStateManager{&m_WindowManager};
    detail::ResourceReservationWindowManager m_ResourceReservationWindowManager{&m_WindowManager};
    detail::BackgroundWindowManager m_BackgroundWindowManager{this, &m_WindowManager};
    StorageChannel m_GeneralChannel{true};

    std::atomic<uint32_t> m_ProcessCreationCount{0};

    template <typename Applet, typename... Args>
    std::shared_ptr<Applet> MakeAppletImpl(Args&&... args) NN_NOEXCEPT;

    void RegisterSystemApplet(std::shared_ptr<IntegratedSystemApplet> pObserver) NN_NOEXCEPT;

    Result CreateApplicationImpl(sf::SharedPointer<am::service::IApplicationAccessor>* pOut, const applet::AppletIdentityInfo& appletIdentityInfo, std::shared_ptr<IntegratedApplet> pParent, const applet::ApplicationLaunchRequestInfo& launchRequestInfo) NN_NOEXCEPT;

    void TerminateAllAppletsForDebug() NN_NOEXCEPT;
    os::Mutex m_AppletListForDebugMutex{false};
    std::list<std::weak_ptr<IntegratedApplet>, StaticAllocator<std::weak_ptr<IntegratedApplet>>> m_AppletListForDebug;
    void CleanAppletListForDebug() NN_NOEXCEPT;

    struct SharedReaderWriterLock : public std::shared_ptr<ReaderWriterLock>
    {
        SharedReaderWriterLock() NN_NOEXCEPT
            : std::shared_ptr<ReaderWriterLock>(MakeShared<ReaderWriterLock>())
        {
        }
    };
    SharedReaderWriterLock m_pReaderWriterLockArray[applet::ReaderWriterLockIndex_Max];

    class SelfLibraryAppletCreatorImpl;

    os::SdkMutex m_MemoryHeapMutex;
    size_t m_CurrentMemoryHeapSize = 0;

    std::shared_ptr<IntegratedOverlayApplet> GetTheOverlayApplet() NN_NOEXCEPT
    {
        return m_pTheOverlayApplet.lock();
    }

    std::shared_ptr<IntegratedApplication> GetLastCreatedApplication() NN_NOEXCEPT
    {
        return m_pLastCreatedApplication.lock();
    }

};

}}}
