﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   システムプロセスの状態管理に関する型や定数の宣言
 *
 * @details
 *  本ヘッダでは、システムアプレットが使用する型や定数などを定義しています。
 *
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/applet/applet_Types.h>

namespace nn { namespace ae { namespace detail {
    struct ApplicationHandleObject;
}}}

namespace nn { namespace am { namespace service {
    class IStorage;
}}}

namespace nn { namespace ae {

//-----------------------------------------------------------------------------
/**
 * @brief   起動時パラメータの最大サイズを示す定数値です。
 *
 */
const size_t StartupParamSizeMax = 8192;


//-----------------------------------------------------------------------------
/**
 * @brief   終了コードの最大サイズを示す定数値です。
 *
 */
const size_t ReturnValueSizeMax = 8192;


//-----------------------------------------------------------------------------
/**
 * @brief   出画画像の横解像度を示す定数値です。
 *
 */
const size_t CaptureBufferWidth = 1280;


//-----------------------------------------------------------------------------
/**
 * @brief   出画画像の縦解像度を示す定数値です。
 *
 */
const size_t CaptureBufferHeight = 720;

//-----------------------------------------------------------------------------
/**
 * @brief   出画画像の 1 ピクセルあたりのバイト数を示す定数値です。
 *
 */
const size_t CaptureBufferBytesPerPixel = 4;


//-----------------------------------------------------------------------------
/**
 * @brief   出画画像のデータサイズを示す定数値です。
 *
 */
const size_t CaptureBufferSize = CaptureBufferWidth * CaptureBufferHeight * CaptureBufferBytesPerPixel;


//-----------------------------------------------------------------------------
/**
 * @brief   アプリケーションハンドルを扱う型です。
 *
 */
struct ApplicationHandle
{
    detail::ApplicationHandleObject*  _p;

    /**
     * @brief   無効な ApplicationHandle 値を返します。
     * @return  無効な ApplicationHandle 値です。
     */
    static ApplicationHandle GetInvalidHandle() NN_NOEXCEPT
    {
        ApplicationHandle handle = { NULL };
        return handle;
    }

    /**
     * @brief   ApplicationHandle が有効か否かを判定します。
     * @return  ApplicationHandle 値が有効なら true を、無効なら false を返します。
     */
    bool IsValid() const NN_NOEXCEPT
    {
        return !(_p == NULL);
    }

    friend bool operator==(const ApplicationHandle& lhs,
                           const ApplicationHandle& rhs) NN_NOEXCEPT
    {
        return lhs._p == rhs._p;
    }

    friend bool operator!=(const ApplicationHandle& lhs,
                           const ApplicationHandle& rhs) NN_NOEXCEPT
    {
        return !(lhs == rhs);
    }
};


//-----------------------------------------------------------------------------
/**
 * @brief   アプレット ID を定義する列挙体です。
 *
 */
enum AppletId
{
    AppletId_None           = applet::AppletId_None,            //!< 特定のアプレットを指しません
    AppletId_SystemAppletMenu   = applet::AppletId_SystemAppletMenu,    //!< システムアプレット（メニュー）

    AppletId_LibraryAppletAuth          = applet::AppletId_LibraryAppletAuth,           //!< 簡易認証アプレット
    AppletId_LibraryAppletCabinet       = applet::AppletId_LibraryAppletCabinet,        //!< amiibo 設定アプレット
    AppletId_LibraryAppletController    = applet::AppletId_LibraryAppletController,     //!< コントローラーアプレット
    AppletId_LibraryAppletDataErase     = applet::AppletId_LibraryAppletDataErase,      //!< データ削除アプレット
    AppletId_LibraryAppletError         = applet::AppletId_LibraryAppletError,          //!< エラービューアアプレット
    AppletId_LibraryAppletNetConnect    = applet::AppletId_LibraryAppletNetConnect,     //!< ネット接続設定アプレット
    AppletId_LibraryAppletPlayerSelect  = applet::AppletId_LibraryAppletPlayerSelect,   //!< プレイヤー選択アプレット
    AppletId_LibraryAppletSwkbd         = applet::AppletId_LibraryAppletSwkbd,          //!< ソフトウェアキーボードアプレット
    AppletId_LibraryAppletMiiEdit       = applet::AppletId_LibraryAppletMiiEdit,        //!< Mii 編集アプレット
    AppletId_LibraryAppletWeb           = applet::AppletId_LibraryAppletWeb,            //!< Web アプレット
    AppletId_LibraryAppletShop          = applet::AppletId_LibraryAppletShop,           //!< ショップアプレット
    AppletId_LibraryPhotoViewer         = applet::AppletId_LibraryAppletPhotoViewer,    //!< フォトビューアー
    AppletId_LibrarySet                 = applet::AppletId_LibraryAppletSet,            //!< 本体設定
    AppletId_LibraryAppletMyPage        = applet::AppletId_LibraryAppletMyPage,         //!< マイページアプレット
    AppletId_LibraryAppletOfflineWeb    = applet::AppletId_LibraryAppletOfflineWeb,     //!< Offline Web アプレット
    AppletId_LibraryAppletLoginShare    = applet::AppletId_LibraryAppletLoginShare,     //!< ログインシェアアプレット
    AppletId_LibraryAppletWifiWebAuth   = applet::AppletId_LibraryAppletWifiWebAuth,    //!< WiFi WebAuth アプレット

    AppletId_LibraryAppletGift          = applet::AppletId_LibraryAppletGift,           //!< おすそわけ通信アプレット
    AppletId_LibraryAppletUserMigration = applet::AppletId_LibraryAppletUserMigration,  //!< ユーザ移行アプレット

    AppletId_LibraryAppletEncounter     = applet::AppletId_LibraryAppletEncounter,      //!< すれちがいアプレット
    AppletId_LibraryAppletStory         = applet::AppletId_LibraryAppletStory,          //!< セーブデータ同期アプレット

    AppletId_OverlayApplet              = applet::AppletId_OverlayApplet,               //!< オーバレイアプレット
};


//-----------------------------------------------------------------------------
/**
 * @brief   アプレットマネージャからの通知メッセージを定義する列挙体です。
 *
 * @details
 *  アプレットマネージャからの通知メッセージを定義しています。
 *  通知メッセージは、 GetNotificationMessage() もしくは、
 *  WaitForNotificationMessage() で取得できます。
 *
 */
enum Message
{
    Message_None                    = applet::Message_None,                             //!< 通知メッセージが届いていない
    Message_ChangeIntoForeground    = applet::Message_InFocus,                          //!< Foreground へ遷移して下さい。
    Message_ChangeIntoBackground    = applet::Message_OutOfFocus,                       //!< Background へ遷移して下さい。
    Message_Exit                    = applet::Message_Exit,                             //!< 自プロセスを終了して下さい。

    Message_ApplicationExited           = applet::Message_ApplicationExited,            //!< アプリケーションが正常終了しました。
    Message_ApplicationSuspendedByRightsError = applet::Message_ApplicationSuspendedByRightsError,  //!< アプリが権利不足で中断した

    Message_RequestToPrepareSleep   = applet::Message_RequestToReleaseSleepLock,        //!< スリープシーケンスの開始
    Message_FinishedSleepSequence   = applet::Message_FinishedSleepSequence,            //!< スリープシーケンスの終了

    Message_SleepRequiredByHighTemperature  = applet::Message_SleepRequiredByHighTemperature, //!< 高温状態によるスリープ遷移推奨通知
    Message_SleepRequiredByLowBattery       = applet::Message_SleepRequiredByLowBattery,      //!< バッテリー不足によるスリープ遷移推奨通知
    Message_AutoPowerDown                   = applet::Message_AutoPowerDown,                  //!< 一定時間無操作によるスリープ遷移要求

    Message_OperationModeChanged   = applet::Message_OperationModeChanged,              //!< 動作モードの変更通知
    Message_PerformanceModeChanged = applet::Message_PerformanceModeChanged,            //!< 性能モードの変更通知

    Message_SdCardRemoved = applet::Message_SdCardRemoved,            //!< SD カードが抜かれたことの通知

    Message_ControllerFirmwareUpdateSectionChanged = applet::Message_ControllerFirmwareUpdateSectionChanged,    //!< コントローラ F/W 更新期間の変更通知

    Message_DetectShortPressingHomeButton   = applet::Message_DetectShortPressingHomeButton,    //!< HOME ボタン短押しを検知しました。
    Message_DetectLongPressingHomeButton    = applet::Message_DetectLongPressingHomeButton,     //!< HOME ボタン長押しを検知しました。

    Message_DetectShortPressingPowerButton  = applet::Message_DetectShortPressingPowerButton,   //!< POWER ボタン短押しを検知しました。
    Message_DetectMiddlePressingPowerButton = applet::Message_DetectMiddlePressingPowerButton,  //!< POWER ボタン中押しを検知しました。
    Message_DetectLongPressingPowerButton   = applet::Message_DetectLongPressingPowerButton,    //!< POWER ボタン長押しを検知しました。

    Message_DetectReceivingCecSystemStandby = applet::Message_DetectReceivingCecSystemStandby,  //!< CEC システムスタンバイ受信を検知しました。

    Message_LaunchApplicationRequested = applet::Message_LaunchApplicationRequested, //!< アプリケーションの起動リクエストが存在する

    Message_FloatingApplicationDetected = applet::Message_FloatingApplicationDetected, //!< 外部起動したアプリを検知

    Message_ShowApplicationLogo             = applet::Message_ShowApplicationLogo, //!< オーバーレイ通知へのロゴ表示指示
    Message_HideApplicationLogo             = applet::Message_HideApplicationLogo, //!< オーバーレイ通知へのロゴ非表示指示
    Message_ForceHideApplicationLogo        = applet::Message_ForceHideApplicationLogo, //!< オーバーレイ通知へのロゴ非表示指示(強制)

    Message_RequestToDisplay = applet::Message_RequestToDisplay, //!< 出画リクエスト

    Message_VrModeChanged                   = applet::Message_VrModeChanged, //!< VR モードの変更通知
    Message_VrModeCurtainRequired           = applet::Message_VrModeCurtainRequired, //!< VR モードのカーテン要求通知

    Message_RequestToShutdown               = applet::Message_RequestToShutdown,    //!< シャットダウン要求
    Message_RequestToReboot                 = applet::Message_RequestToReboot,      //!< 再起動要求
    Message_RequestToGoBackQuestMenu        = applet::Message_RequestToGoBackQuestMenu, //!< アプリ終了および試遊台メニュー復帰要求

    Message_RequestToLaunchApplicationForDebug = applet::Message_RequestToLaunchApplicationForDebug //!< デバッグ用アプリケーション起動要求
};


//-----------------------------------------------------------------------------
/**
 * @brief   無線通信のプライオリティを表す列挙型です。
 */
enum WirelessPriorityMode
{
    WirelessPriorityMode_Default = 0,            //!< 各アプレットごとのデフォルト設定
    WirelessPriorityMode_OptimizedForWlan = 1,   //!< Wlan を優先するように最適化。その他の無線通信（例：コントローラの無線通信）は品質低下
};


//-----------------------------------------------------------------------------
/**
 * @brief   本体の起動理由を示す列挙体です。
 *
 */
enum BootReason
{
    BootReason_ColdBoot     = 1,        //!< コールドブートです。
};


//-----------------------------------------------------------------------------
/**
 * @brief   スクリーンショット撮影の禁止／許可状態を示す列挙対です。
 *
 */
enum ScreenShotPermission
{
    ScreenShotPermission_Inherit    = applet::ScreenShotPermission_Inherit, //!< 呼び出し元の状態を継承します。
    ScreenShotPermission_Permit     = applet::ScreenShotPermission_Permit,  //!< 撮影を許可します。
    ScreenShotPermission_Forbid     = applet::ScreenShotPermission_Forbid,  //!< 撮影を禁止します。
};


//-----------------------------------------------------------------------------

}} // namespace nn::ae
