﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   スリープの制御に関する API の宣言
 *
 * @details
 *  本ヘッダでは、スリープの制御に関する API を定義しています。
 *
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/ae/ae_Types.h>

#include <nn/nn_Result.h>

namespace nn { namespace ae {

//--------------------------------------------------------------------------

//! @name 各種アプレット開発用スリープ関連 API
//! @{

/**
    @brief スリープシーケンスを開始します。

    @retresult
    @endretresult

    @details
     スリープシーケンスを開始します。スリープシーケンスは非同期に進行するため、本関数の呼び出しは短時間で返ります。
     スリープシーケンスの終了は Message_FinishedSleepSequence によって通知されます。

     スリープシーケンス実行中は、スリープシーケンスを開始することはできません。

     スリープシーケンスは以下のように進行します。

     - LCD を OFF にする (未実装・仕様変更あり)
     - FG 状態だったアプレットやアプリケーションが BG 状態に移行する
     - ※ アプリケーションが FG 状態から BG 状態に移行した場合には、アプリケーション画面キャプチャが更新される
     - 全てのスリープのハンドリングを行うアプレットに対し Message_RequestToPrepareSleep を通知する
     - 全てのスリープのハンドリングを行うアプレットから AllowToEnterSleepAndWait() または EndSleepHandling() が呼ばれるのを待機する
     - スリープに入り、起きるのを待つ (未実装)
     - 各アプレットで呼び出し中だった AllowToEnterSleepAndWait() が返る
     - LCD を ON にする (未実装・仕様変更あり)
     - 起動元アプレットに Message_FinishedSleepSequence を通知する
*/
Result StartSleepSequence() NN_NOEXCEPT;

/**
    @brief スリープの事前ハンドリングを行うことを宣言します。

    @pre (BeginSleepHandling() 呼び出し回数) == (EndSleepHandling() の呼び出し回数)

    @details
     この関数を呼んでから次の EndSleepHandling() までの間、スリープのハンドリングを行うことを宣言します。
     この間 Message_RequestToPrepareSleep の通知に対し、適切にスリープ前の準備を行い、
     速やかに AllowToEnterSleepAndWait() または EndSleepHandling() を呼ぶ必要があります。

     スリープシーケンスがすでに実行中の場合に本関数を呼んだ場合、
     スリープシーケンスの進行具合によって、長時間ブロックするときと、
     短時間で返ってきた上で Message_RequestToPrepareSleep が通知されるときとの、いずれの可能性もあります。
     長時間ブロックしてもよい状況で呼ぶようにしてください。

     スリープ復帰後に特定の画面出すように実装するにあたっては、以下のようなシーケンスが想定されています。

     - プログラム初期化時に BeginSleepHandling() を呼び、スリープのハンドリング行うことを宣言する
     - スリープトリガを検知したら StartSleepSequence() を呼び、スリープシーケンスを開始する
     - Message_RequestToPrepareSleep を待機する
     - スリープ復帰後の表示準備を行う
     - RequestToGetForeground() を呼んで、スリープ復帰時に FG 状態に移行するようにする(スリープから復帰するまでは Message_ChangeIntoForeground は通知されない)
     - AllowToEnterSleepAndWait() を呼び、スリープ状態への遷移を許可するとともに、スリープからの復帰を待つ
     - Message_ChangeIntoForeground が通知される

    @see AllowToEnterSleepAndWait(), EndSleepHandling(), Message_RequestToPrepareSleep
*/
void BeginSleepHandling() NN_NOEXCEPT;

/**
    @brief スリープ前の処理が終了したことを通知し、スリープから起きるまでブロックします。

    @pre (BeginSleepHandling() 呼び出し回数) == (EndSleepHandling() の呼び出し回数) + 1
    @pre 前回の BeginSleepHandling() 呼び出し以後、Message_RequestToPrepareSleep が通知されており、その後まだ AllowToEnterSleepAndWait() が呼ばれていない

    @details
     本関数は Message_RequestToPrepareSleep に対する返答として使用することを想定されています。
     Message_RequestToPrepareSleep が通知される前に呼ぶことはできません。
*/
void AllowToEnterSleepAndWait() NN_NOEXCEPT;

/**
    @brief スリープの事前ハンドリングを行わないことを宣言します。

    @pre (BeginSleepHandling() 呼び出し回数) == (EndSleepHandling() の呼び出し回数) + 1

    @details
     呼ぶ場合には、必ず BeginSleepHandling() と対で呼ぶ必要があります。
     本関数は、Message_RequestToPrepareSleep の通知のタイミングとは独立して呼ぶことが可能です。

     BeginSleepHandling() の呼び出し後、本関数が呼び出されずにプログラムが終了した場合には、
     本関数と同等の処理が自動で行われます。
     このため、プログラムの終了時に必ずしも本関数を呼ぶ必要はありません。
*/
void EndSleepHandling() NN_NOEXCEPT;

//! @}

//! @name 各種アプレット開発用電源操作関連 API
//! @{

/**
    @brief  システムをシャットダウンします。

    @retresult
    @endretresult

    @details
     システムをシャットダウンします。
     実際のシャットダウンシーケンスは非同期に進行するため、
     本関数の呼び出しは短時間で帰ります。

     他のスリープシーケンスや再起動シーケンスが進行中の場合には、
     エラーを示す Result が返ります。

     本 API は各アプレットの動作状態に関係なく、
     即座に後述のシャットダウンシーケンスを開始しますので、
     必要に応じて事前にアプリケーションやライブラリアプレットを
     終了させておくことを推奨します。

     シャットダウンシーケンスは以下のように進行します。

     - LCD を OFF にする (未実装・仕様変更あり)
     - FG 状態だったアプレットやアプリケーションが BG 状態に移行する
     - シャットダウンシーケンスを開始する

*/
Result StartShutdownSequence() NN_NOEXCEPT;

/**
    @brief  システムを再起動します。

    @retresult
    @endretresult

    @details
     システムを再起動します。
     実際の再起動シーケンスは非同期に進行するため、
     本関数の呼び出しは短時間で帰ります。

     他のスリープシーケンスやシャットダウンシーケンスが進行中の場合には、
     エラーを示す Result が返ります。

     本 API は各アプレットの動作状態に関係なく、
     即座に後述の再起動シーケンスを開始しますので、
     必要に応じて事前にアプリケーションやライブラリアプレットを
     終了させておくことを推奨します。

     再起動シーケンスは以下のように進行します。

     - LCD を OFF にする (未実装・仕様変更あり)
     - FG 状態だったアプレットやアプリケーションが BG 状態に移行する
     - 再起動シーケンスを開始する

*/
Result StartRebootSequence() NN_NOEXCEPT;

//! @}

//-----------------------------------------------------------------------------

}} // namespace nn::ae

