﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   オーバレイアプレットの制御に関する API の宣言
 *
 * @details
 *  本ヘッダでは、オーバレイアプレットの制御に関する API を宣言しています。
 *
 *  オーバレイアプレットには他のアプレットにはない以下の内部状態を持ちます。
 *  これらは、オーバレイアプレットが何かしらのユーザインタラクション処理を
 *  行なっている状態を示します。デフォルトは非ユーザインタラクション状態です。
 *
 *  - 非ユーザインタラクション状態
 *  - ユーザインタラクション状態
 *
 *  ユーザインタラクション状態中は、通常はシステムアプレットに通知される
 *  HOME ボタン短押しのメッセージ通知が、システムアプレットには通知されず、
 *  オーバレイアプレットに対してのみ通知されるようになります。
 *
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/ae/ae_Types.h>
#include <nn/nn_TimeSpan.h>
#include <nn/nn_Result.h>

namespace nn { namespace ae {

//-----------------------------------------------------------------------------
/**
 * @brief   オーバレイアプレットの起動パラメータを格納する構造体です。
 *
 * @details
 *  オーバレイアプレット起動時に呼出し元から渡された起動パラメータなどの情報を
 *  格納する構造体です。
 *
 */
struct OverlayAppletParameters
{
    char*   _noParameter;
};


//-----------------------------------------------------------------------------
/**
 * @brief   オーバレイアプレットのメイン関数のエントリを表す型です。
 */
typedef void (*OverlayAppletMainFunction)(OverlayAppletParameters*);


//--------------------------------------------------------------------------

//! @name オーバレイアプレット制御用 API
//! @{

//-----------------------------------------------------------------------------
/**
 * @brief   オーバレイアプレットとしての初期化を行ない、アプレットのメイン関数を呼び出します。
 *
 * @param[in]  appletId 自身のアプレット ID
 * @param[in]  function アプレットのメイン関数エントリ
 *
 * @pre
 *  - appletId == AppletId_OverlayApplet である
 *
 * @details
 *  オーバレイアプレットとしての初期化を行ない、その後に function で指定された
 *  アプレットのメイン関数を呼び出します。function は OverlayAppletMainFunction
 *  型の関数であり、引数で AppletParameter 構造体を受け取ります。
 *
 *  本関数の呼び出し中、オーバーレイアプレットとして実行中であることになります。
 *
 *  本関数は nnMain() の中から呼び出して下さい。
 *  また、本関数を呼ぶまで ae ライブラリで提供している他の機能は使用できません。
 *
 *  appletId には AppletId_OverlayApplet を指定して下さい。
 *
 *  オーバレイアプレットも他のアプレット同様に、アプレットマネージャからの
 *  メッセージ通知を受け取りますが、通知されるメッセージは限定的です。
 *  詳細は別途マニュアル等を参照して下さい。
 *
 */
void InvokeOverlayAppletMain(AppletId appletId, OverlayAppletMainFunction function) NN_NOEXCEPT;


//-----------------------------------------------------------------------------
/**
 * @brief   オーバレイアプレットのユーザインタラクション処理を開始します。
 *
 * @pre
 *  - オーバーレイアプレットとして実行中である
 *  - オーバレイアプレットが非ユーザインタラクション状態である
 *
 * @post
 *  - オーバレイアプレットがユーザインタラクション状態である
 *
 * @details
 *  オーバレイアプレットのユーザインタラクション処理を開始します。
 *  「クイック設定」や「電源メニュー」など、ユーザインターフェースを提供する
 *  ような処理を行なうには、本 API でユーザインタラクションを開始して下さい。
 *
 *  アプレットマネージャは本 API の受理後、
 *  オーバレイアプレットに対して以下の権限を優先的に付与します。
 *
 *  - 通常は SA にのみ通知される HOME ボタン短押しメッセージを、オーバレイアプレットのみが受理
 *
 *  ユーザインタラクション処理を終えた時には、
 *  EndOverlayUserInteraction() を発行して下さい。
 *
 */
void BeginOverlayUserInteraction() NN_NOEXCEPT;


//-----------------------------------------------------------------------------
/**
 * @brief   オーバレイアプレットのユーザインタラクション処理を終了します。
 *
 * @pre
 *  - オーバーレイアプレットとして実行中である
 *  - オーバレイアプレットがユーザインタラクション状態である
 *
 * @post
 *  - オーバレイアプレットが非ユーザインタラクション状態である
 *
 * @details
 *  オーバレイアプレットのユーザインタラクション処理を終了します。
 *  BeginOverlayUserInteraction() によって優先的に与えられていた各種権限は、
 *  BeginOverlayUserInteraction() 発行前の状態に戻ります。
 *
 */
void EndOverlayUserInteraction() NN_NOEXCEPT;


//-----------------------------------------------------------------------------
/**
 * @brief   オーバレイアプレットの APD 設定時間を有効にします。
 *
 * @pre
 *  - 自アプレットがオーバーレイアプレットである
 *
 * @details
 *  自アプレットで nn::applet::OverrideAutoSleepTimeAndDimmingTime() で
 *  設定済みの無操作状態検出時の自動スリープや自動低輝度化の発動時間の
 *  設定値を有効にします。デフォルトでは無効になっています。
 *
 *  オーバーレイアプレットは、クイック設定や電源メニューなどの
 *  ユーザインタラクションの種別によって、上記発動時間を有効にしたいケースと
 *  無効にしたいケースがあります。
 *
 *  そのため、これらの有効無効の設定は nn::ae::BeginOverlayUserInteraction()
 *  や nn::ae::EndOverlayUserInteraction() には連動せずに、本 API および
 *  nn::ae::DisableAutoSleepTimeAndDimmingTime() を使って明示的に有効無効を
 *  制御するようにして下さい。
 *
 */
void EnableAutoSleepTimeAndDimmingTime() NN_NOEXCEPT;


//-----------------------------------------------------------------------------
/**
 * @brief   オーバレイアプレットの APD 設定時間を無効にします。
 *
 * @pre
 *  - オーバーレイアプレットとして実行中である
 *
 * @details
 *  nn::ae::EnableAutoSleepTimeAndDimmingTime() で有効にしていた
 *  無操作状態検出時の自動スリープや自動低輝度化の発動時間の設定値を
 *  無効にします。
 *
 *  詳細は nn::ae::EnableAutoSleepTimeAndDimmingTime() を参照して下さい。
 *
 */
void DisableAutoSleepTimeAndDimmingTime() NN_NOEXCEPT;


//-----------------------------------------------------------------------------
/**
 * @brief   システムグローバルなスクリーンショット撮影禁止区間を開始します。
 *
 * @pre
 *  - 自アプレットがオーバーレイアプレットである
 *
 * @details
 *  キャプチャボタン短押しによるスクリーンショット撮影のシステムグローバルな
 *  撮影禁止区間を開始します。本 API は重ね掛けが可能です。
 *
 *  撮影禁止区間内では、撮影対象となる各アプレットごとの撮影禁止／許可状態の
 *  設定値によらず常に撮影禁止として扱われます。撮影禁止区間内で
 *  キャプチャ撮影が行われると、オーバーレイアプレットには撮影禁止を示す
 *  メッセージが通知されます。
 *
 */
void BeginScreenShotCaptureForbiddenSection() NN_NOEXCEPT;


//-----------------------------------------------------------------------------
/**
 * @brief   システムグローバルなスクリーンショット撮影禁止区間を終了します。
 *
 * @pre
 *  - 自アプレットがオーバーレイアプレットである
 *  - 現在がスクリーンショット撮影禁止区間内である
 *
 * @details
 *  キャプチャボタン短押しによるスクリーンショット撮影のシステムグローバルな
 *  撮影禁止区間を終了します。
 *  より具体的には、 nn::ae::BeginScreenShotCaptureForbiddenSection() と
 *  同じ回数だけ本 API を発行することで撮影禁止区間が解除されます。
 *
 *  撮影禁止区間外では、撮影対象となる各アプレットごとの撮影禁止／許可状態の
 *  設定値に従って撮影許可や禁止などが決定されます。
 *
 */
void EndScreenShotCaptureForbiddenSection() NN_NOEXCEPT;


/**
    @brief ロゴ表示対象のアプリケーションの ID を取得します。

    @return ロゴ表示対象のアプリケーションの ID を返します。

    @details
     Message_ShowApplicationLogo メッセージを通知する元になったアプリケーションの ID を取得します。
*/
ncm::ApplicationId GetApplicationIdForLogo() NN_NOEXCEPT;


/**
    @brief アプリケーションを強制終了し、必要に応じてその終了理由を登録します。

    @param[in] result アプリケーションの終了理由を示す Result コード

    @details
    アプリケーションを強制終了します。また、result の内容によっては、
    必要に応じてその終了理由をシステムに登録します。

    何等かの理由によりアプリケーションの実行を中止すべき時に利用して下さい。

*/
void TerminateApplicationAndSetReason(Result result) NN_NOEXCEPT;


/**
    @brief GPU 割り当て時間を一時的に増やします。

    @param[in] timeSpan 増加させる時間

    @details
     GPU 割り当て時間を、一時的に timeSpan で指定された量だけ増やします。
     増やす必要がなくなった際には、0 を指定して呼び出してください。
*/
void SetGpuTimeSliceBoost(TimeSpan timeSpan) NN_NOEXCEPT;

//! @}

//-----------------------------------------------------------------------------

}} // namespace nn::ae

