﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   メインアプレットの音声制御を行なう API の宣言
 *
 * @details
 *  本ヘッダでは、メインアプレットの音声に関する音量制御や停止／再開を、
 *  ライブラリアプレットから制御するための API を定義しています。
 *
 *  メインアプレットとは、ライブラリアプレット呼出しの大元のアプレットを指し、
 *  具体的にはアプリケーションやシステムアプレット（メニュー）を指します。
 *
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_TimeSpan.h>
#include <nn/ae/ae_Types.h>

namespace nn { namespace ae {

//--------------------------------------------------------------------------

//! @name   音声制御に関するシステムアプレット向け API
//! @{

//--------------------------------------------------------------------------
/**
 * @brief   システムアプレットおよびライブラリアプレットに対するマスターボリューム期待値を設定します。
 *
 * @param[in]   systemAppletVolume  システムアプレットに対するマスターボリューム期待値
 * @param[in]   libraryAppletVolume ライブラリアプレットに対するマスターボリューム期待値
 *
 * @pre
 *  - システムアプレットとして実行中である
 *  - nn::audio::ProcessMasterVolumeMin <= systemAppletVolume  && systemAppletVolume <= nn::audio::ProcessMasterVolumeMax
 *  - nn::audio::ProcessMasterVolumeMin <= libraryAppletVolume && libraryAppletVolume <= nn::audio::ProcessMasterVolumeMax
 *
 * @details
 *  システムアプレットから起動したライブラリアプレットが動作している最中の、
 *  システムアプレットおよびライブラリアプレットに対する
 *  マスターボリューム期待値を float 型で設定します。
 *
 *  本 API を呼んでも自分自身の音声マスターボリューム値が即座に変更される
 *  わけではありません。本 API は、システムアプレットからライブラリアプレットを
 *  起動した場合に、そのライブラリアプレットに対してシステムアプレットが期待する
 *  マスタボリューム設定値を伝えるための API です。
 *
 *  ただし、各ライブラリアプレット起動後にシステムアプレットや
 *  ライブラリアプレットのマスターボリューム値がどのような値に設定されるかは、
 *  各ライブラリアプレットの仕様に依存します。
 *
 *  本 API を使用しなかった場合のデフォルト値は、
 *  システムで定義されたデフォルトのマスターボリューム期待値が設定されています。
 *
 */
void SetExpectedMasterVolume(float systemAppletVolume, float libraryAppletVolume) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 * @brief   システムアプレットおよびライブラリアプレットに対するマスターボリューム期待値を取得します。
 *
 * @param[out]  pSystemAppletVolume  システムアプレットに対するマスターボリューム期待値の格納先
 * @param[out]  pLibraryAppletVolume ライブラリアプレットに対するマスターボリューム期待値の格納先
 *
 * @pre
 *  - システムアプレットとして実行中である
 *
 * @details
 *  システムアプレットおよびライブラリアプレットに対する
 *  マスターボリューム期待値をそれぞれ float 型で取得します。
 *
 *  マスターボリューム期待値については nn::ae::SetExpectedMasterVolume()
 *  を参照して下さい。
 *
 */
void GetExpectedMasterVolume(float* pSystemAppletVolume, float* pLibraryAppletVolume) NN_NOEXCEPT;


//! @}

//--------------------------------------------------------------------------

//! @name メインアプレットの音声制御を行なうライブラリアプレット向け API
//! @{

//--------------------------------------------------------------------------
/**
 * @brief   メインアプレット音声のマスターボリュームを変更する要求を出します。
 *
 * @param[in]   volume      メインアプレット音声のマスターボリューム値
 * @param[in]   fadeTime    音量が変化するときのフェード時間
 *
 * @pre
 *  - ライブラリアプレットとして実行中である
 *  - nn::audio::ProcessMasterVolumeMin <= volume && volume <= nn::audio::ProcessMasterVolumeMax
 *
 * @details
 *  メインアプレット音声のマスターボリューム値を volume に変更します。
 *  実際には、対象音声は現在のボリューム値から
 *  fadeTime で指定された時間をかけて volume 値にまで変化します。
 *
 */
void ChangeMainAppletMasterVolume(float volume, TimeSpan fadeTime) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 * @brief   メインアプレット向けのマスターボリューム期待値を取得します。
 *
 * @return  メインアプレット向けのマスターボリューム期待値を返します。
 *
 * @pre
 *  - ライブラリアプレットとして実行中である
 *
 * @details
 *  アプリケーションが nn::oe::SetExpectedMasterVolume() で設定した、もしくは、
 *  システムアプレットが nn::ae::SetExpectedMasterVolume() で設定した
 *  メインアプレット向けの マスターボリューム期待値を float 型で返します。
 *
 */
float GetMainAppletExpectedMasterVolume() NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 * @brief   ライブラリアプレット向けのマスターボリューム期待値を取得します。
 *
 * @return  ライブラリアプレット向けのマスターボリューム期待値を返します。
 *
 * @pre
 *  - ライブラリアプレットとして実行中である
 *
 * @details
 *  アプリケーションが nn::oe::SetExpectedMasterVolume() で設定した、もしくは、
 *  システムアプレットが nn::ae::SetExpectedMasterVolume() で設定した
 *  ライブラリアプレット向けのマスターボリューム期待値を float 型で返します。
 *
 */
float GetLibraryAppletExpectedMasterVolume() NN_NOEXCEPT;


//! @}

//! @name   音声制御に関するシステムアプレット向け API
//! @{

//--------------------------------------------------------------------------
/**
 * @brief   自アプレットより下位にある全アプレットの音量を絞ります。
 *
 * @param[in] transparentRate   下位アプレットに対する音量の透過率
 *
 * @pre
 *  - 0.0f <= transparentRate && transparentRate <= 1.0f
 *
 * @details
 *  自アプレットより下位にある全アプレット（アプリケーション含む）に対して、
 *  各アプレットの音量を transparentRate 値を乗算した音量値まで絞ります。
 *
 *  デフォルトの transparentRate は 1.0f で 減衰なし となります。
 *  transparentRate に 0.0f を指定するとボリューム値 0 になります。
 *
 *  通常この機能は、オーバーレイアプレットが一時的に他の全アプレットの
 *  音量を絞りたいときなどに使用します。
 *
 */
void SetTransparentVolumeRate(float transparentRate) NN_NOEXCEPT;


//! @}

//-----------------------------------------------------------------------------

}} // namespace nn::ae

