﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   アプリケーションの制御や管理を行なう API の宣言
 *
 * @details
 *  本ヘッダでは、アプリケーションの制御や管理を行なう API を定義しています。
 *
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/os/os_SystemEventTypes.h>
#include <nn/ae/ae_Types.h>
#include <nn/ncm/ncm_ContentMetaId.h>
#include <nn/ncm/ncm_SystemContentMetaId.h>
#include <nn/applet/applet_Storage.h>
#include <nn/applet/applet_Types.h>

#include <nn/nn_TimeSpan.h>

namespace nn { namespace ns {
    struct ApplicationControlProperty;
}}

namespace nn { namespace account {
    struct Uid;
}}

namespace nn { namespace ae {

//--------------------------------------------------------------------------

//! @name アプリケーションの制御や管理を行なう API
//! @{

//--------------------------------------------------------------------------
/**
 * @brief   アプリケーションを起動します。
 *
 * @param[in]   id      起動させるアプリケーションの ApplicationId
 * @param[in]   foreground アプリケーションを Foreground 状態にするかかどうかを指定します
 *
 * @return  対象アプリケーションを操作するためのハンドルが返ります。
 *
 * @pre
 *  - システムアプレットとして実行中である
 *
 * @details
 *  id で指定されたアプリケーションを起動します。
 *  ただし、本 API 発行後は対象アプリケーションの起動完了を待たずに
 *  即座にリターンします。
 *
 *  返値には対象アプリケーションを操作するためのハンドル値が返ります。
 *
 *  foreground が true のとき、アプリケーションを起動後すぐに Foreground 状態にします。
 *  foreground が false のとき、アプリケーションを Foreground 状態にはしません。
 *  この場合、アプリケーションを Foreground 状態にするには、
 *  別途 RequestApplicationGetForeground() を呼ぶ必要があります。
 *
 */
ApplicationHandle LaunchApplication(nn::ncm::ApplicationId id, bool foreground = true) NN_NOEXCEPT;

/**
    @brief 指定された ApplicationId のアプリケーションを作成します。

    @param[in] id 作成するアプリケーションの ApplicationId を指定します。

    @return 作成したアプリケーションを表すハンドルを返します。

    @details
     id で指定された ApplicationId のアプリケーションを作成し、そのハンドルを返します。
     LaunchApplication() と異なり、アプリケーションの開始はしません。
     開始させるには、別途 StartApplication() を呼ぶ必要があります。

     本関数から StartApplication() を呼ぶまでの間に PushApplicationLaunchParameter() を呼ぶことで、
     アプリケーションの起動時パラメータを指定することができます。
*/
ApplicationHandle CreateApplication(ncm::ApplicationId id) NN_NOEXCEPT;

/**
    @brief 指定された SystemApplicationId のシステムアプリケーションを作成します。

    @param[in] id 作成するシステムアプリケーションの SystemApplicationId を指定します。

    @return 作成したアプリケーションを表すハンドルを返します。

    @details
     作成対象が一般アプリケーションではなく、システムアプリケーションである点を除いて、
     CreateApplication() と同様です。
*/
ApplicationHandle CreateSystemApplication(ncm::SystemApplicationId id) NN_NOEXCEPT;

/**
    @brief 起動リクエストのあるアプリケーションのハンドルを取得します。

    @param[out] pOut 取得するハンドルを格納する領域を指定します。

    @return 取得できたとき true を返し、そうでないとき false を返します。

    @details
     アプリケーションの再起動や、別アプリケーションの起動要求が来た際に、
     その起動要求を受けたアプリケーションのハンドルを取得します。

     本関数は Message_LaunchApplicationRequested を受けたのちに呼ぶと、
     true が返り、アプリケーションのハンドルを取得することができます。
     そうでないときに呼ぶと取得できず、false が返ることがあります。

     取得したアプリケーションは本関数が返った時点では開始されていません。
     開始させるには、別途 StartApplication() を呼ぶ必要があります。

     本関数から StartApplication() を呼ぶまでの間に PushApplicationLaunchParameter() を呼ぶことで、
     アプリケーションの起動時パラメータを指定することができます。

     起動チェックなどのために、取得したアプリケーションの ApplicationId を取得するには、
     GetApplicationId() を呼んでください。

     起動チェックなどの結果、取得したアプリケーションの起動を行わない場合には、
     StartApplication() を呼ばずに CloseApplicationHandle() を呼んでください。
*/
bool TryPopLaunchRequestedApplication(ApplicationHandle* pOut) NN_NOEXCEPT;

/**
    @brief アプリケーションの起動時パラメータを設定します。

    @param[in] applicationHandle 対象アプリケーションのハンドル
    @param[in] launchParameterKind パラメータの種類
    @param[in] storageHandle 設定するデータの格納されたストレージ

    @pre 対象のアプリケーションが開始されていない

    @details
     applicationHandle で表されるアプリケーションの launchParameterKind で表されるパラメータとして、
     storageHandle で指定されたストレージを push します。

     ここで設定されたストレージは、起動されたアプリケーションプログラム内で、
     applet::TryPopFromApplicationParameterChannel() を使って取得することができます。
*/
void PushApplicationLaunchParameter(ApplicationHandle applicationHandle, applet::LaunchParameterKind launchParameterKind, applet::StorageHandle storageHandle) NN_NOEXCEPT;

/**
    @brief アプリケーションを起動します。

    @param[in] applicationHandle 対象アプリケーションのハンドル

    @details
     applicationHandle で指定されたアプリケーションを開始します。

     なお、本関数で開始されたアプリケーションは、自動では FG 状態には遷移しません。
     別途 RequestApplicationGetForeground() を呼ぶ必要があります。
*/
void StartApplication(ApplicationHandle applicationHandle) NN_NOEXCEPT;

/**
    @brief アプリケーションの ApplicationId を取得します。

    @param[in] applicationHandle 対象アプリケーションのハンドル

    @return 指定アプリケーションの ApplicationId を返します。
*/
ncm::ApplicationId GetApplicationId(ApplicationHandle applicationHandle) NN_NOEXCEPT;

/**
    @brief アプリケーションの管理データを取得します。

    @param[out] outValue 対象アプリケーションの管理データのポインタ
    @param[in]  applicationHandle 対象アプリケーションのハンドル

    @retresult
    @endretresult

    @details
     成功した場合、outValue が指すポインタに取得したデータが格納されます。

*/
Result GetApplicationControlProperty(nn::ns::ApplicationControlProperty* outValue, ApplicationHandle applicationHandle) NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
 * @brief   対象アプリケーションに終了するように要求します。
 *
 * @param[in]  handle   終了を要求する対象のアプリケーションハンドル
 *
 * @retresult
 *   @handleresult{nn::ae::ResultApplicationExited}
 * @endretresult
 *
 * @pre
 *  - システムアプレットとして実行中である
 *  - handle.IsValid() == true である
 *
 * @details
 *  handle で指定されたアプリケーションに対して終了するように要求を出します。
 *  ただし、本 API はアプリケーションの終了を待たずして即座にリターンします。
 *
 *  アプリケーションがライブラリアプレットを呼び出し中だった場合には、
 *  呼び出し中の全てのライブラリアプレットに対しても終了要求が出され、
 *  最終的にアプリケーションに対して終了要求が出されます。
 *
 *  アプリケーションに通知されるメッセージは nn::oe::MessageExitRequest で、
 *  ライブラリアプレットに通知されるメッセージは nn::ae::Message_Exit です。
 *  名前空間のみが異なり、意味は同じです。
 *
 *  アプリケーションやライブラリアプレットは終了要求を拒絶することはできません。
 *  アプリケーションが終了すると、 AppletId_SystemAppletMenu アプレットに対して
 *  Message_ApplicationExited メッセージが通知されます。
 *
 *  アプリケーションが一向に終了しない場合、いつまで経っても上記メッセージが
 *  通知されないことがあり得ます。そのような場合は、適当なタイムアウト時間を
 *  設けて TerminateApplication() にてアプリケーションを強制終了することが
 *  できます。
 *
 *  なお、対応アプリケーションが何かしらの理由で既に終了済みだった場合には、
 *  それらを示す Result が返されます。
 *
 */
Result RequestApplicationExit(ApplicationHandle handle) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 * @brief   指定したアプリケーションを強制終了させます。
 *
 * @param[in]  handle   強制終了する対象のアプリケーションハンドル
 *
 * @retresult
 *   @handleresult{nn::ae::ResultApplicationExited}
 * @endretresult
 *
 * @pre
 *  - システムアプレットとして実行中である
 *  - handle.IsValid() == true である
 *
 * @details
 *  handle で指定されたアプリケーションを強制終了させます。
 *  ただし、本 API はアプリケーションの終了を待たずして即座にリターンします。
 *
 *  通常、アプリケーションを終了させる場合には RequestApplicationExit() にて
 *  終了要求を出し、対象アプリケーションが自発的に終了するのを待機します。
 *  ただし、上記のような場合でも、一向にアプリケーションが終了しない場合には
 *  本 API にて強制終了させることができます。
 *
 *  LaunchApplication() にて handle を生成後、対象アプリケーションが何かしらの
 *  理由で終了していた場合、それに応じた Result が返ります。
 *
 */
Result TerminateApplication(ApplicationHandle handle) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 * @brief   対象アプリケーションが Foreground へ遷移するように要求を出します。
 *
 * @param[in]  handle   対象アプリケーションを指すアプリケーションハンドル
 *
 * @retresult
 *   @handleresult{nn::ae::ResultApplicationExited}
 *   @handleresult{nn::ae::ResultAlreadyForeground}
 * @endretresult
 *
 * @pre
 *  - システムアプレットとして実行中である
 *  - handle.IsValid() == true である
 *  - 自アプレットが AppletId_SystemAppletMenu である
 *  - 自アプレットが Foreground 状態である
 *
 * @details
 *  handle で指定されたアプリケーションが Foreground 状態に遷移するように
 *  要求を出します。本 API 自身は要求だけ出して即座にリターンします。
 *
 *  もし、対象アプリケーションが何らかの理由で終了していた場合は、
 *  それに応じた Result が返ります。
 *  また、すでにアプリケーションが Foreground 状態だった場合には、
 *  nn::ae::ResultAlreadyForeground が返ります。
 *
 *  対象アプリケーションがライブラリアプレットを呼び出し中だった場合には、
 *  最後に Foreground 状態だったライブラリアプレットが Foreground 状態に
 *  なるように要求が出されます。
 *
 *  本 API 呼び出し後、自アプレットには
 *  Message_ChangeIntoBackground が通知されます。
 *
 */
Result RequestApplicationGetForeground(ApplicationHandle handle) NN_NOEXCEPT;


//--------------------------------------------------------------------------

/**
    @brief アプリケーションが終了したらシグナルされるイベントを取得します。

    @param[in] handle 対象のアプリケーションハンドル

    @return アプリケーションが終了したらシグナルされるイベントへのポインタを返します。
*/
os::SystemEventType* GetApplicationExitEvent(ApplicationHandle handle) NN_NOEXCEPT;

/**
    @brief アプリケーションの結果を表す型です。
*/
enum ApplicationResult
{
    ApplicationResult_NormallyExited = 0,  //!< 正常終了
    ApplicationResult_AbnormallyExited = 1, //!< 異常終了
    ApplicationResult_TerminatedManually = 2, //!< TerminateApplication() による強制終了
    ApplicationResult_TerminatedByMediaLost = 3, //!< データメディアの喪失による強制終了(未実装)
    ApplicationResult_VoluntarilyExited = 4,  //!< 自発終了
    ApplicationResult_Unexpected = 100, //!< 想定外の要因による終了
};

/**
    @brief アプリケーションの結果を取得します。

    @param[in] handle 対象のアプリケーションハンドル

    @return アプリケーションの結果を返します。

    @pre StartApplication(handle) が呼ばれている

    @details
     handle で指定されたアプリケーションの結果を取得します。
     アプリケーションがまだ終了していなかった場合には、終了するまでブロックします。

     ApplicationResult_NormallyExited が返った場合には、アプリケーションの正常終了を表します。
     なお、システム的に正常終了したことを表すものであり、
     想定外のタイミングでの正常終了を異常系とみなすことも可能です。
     例えば RequestApplicationExit() を呼ぶ前に(正常)終了した場合は、異常系であるとみなすなどです。

     ApplicationResult_AbnormallyExited が返った場合には、アプリケーションが異常終了したことを表します。
     これは所定の手続きを経ずにアプリケーションが終了した場合に返ります。

     ApplicationResult_TerminatedManually が返った場合には、アプリケーションが TerminateApplication() によって強制終了されたことを表します。
     RequestApplicationExit() を呼んでもなおアプリケーションが終了しなかったような場合に、
     TerminateApplication() によって強制終了するなどが考えられます。

     ApplicationResult_TerminatedByMediaLost が返った場合には、
     アプリケーションの処理に必要なメディアが抜かれるなどが検知されたことによって、
     システムがアプリケーションを強制終了したことを表します。

     ApplicationResult_Unexpected が返った場合、想定外の要因によってアプリケーションが終了したことを表します。
     例えば、内部的なアプリケーションプログラムの起動チェックなどによって起動できなかった場合などを含みます。
     製品においては、この返り値が返らないようにデバッグする必要があります。
     デバッグ用の詳細は GetApplicationResultRawForDebug() で取得することができます。
 */
ApplicationResult GetApplicationResult(ApplicationHandle handle) NN_NOEXCEPT;

/**
    @brief (デバッグ用) アプリケーションのデバッグ用の結果を取得します。

    @param[in] handle 対象のアプリケーションハンドル

    @return アプリケーションのデバッグ用の結果を返します。

    @pre StartApplication(handle) が呼ばれている

    @details
     デバッグ用に GetApplicationResult(handle) が ApplicationResult_Unexpected を返した際、
     より詳細な結果を取得します。

     本関数はデバッグ用です。本関数から返った値によって、プログラムの処理を分岐するなど、
     有意に利用しないでください。
*/
Result GetApplicationResultRawForDebug(ApplicationHandle handle) NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
 * @brief   (開発用) 外部起動されたアプリケーションを指すアプリケーションハンドルを取得します。
 *
 * @return  外部起動されたアプリケーションを指すアプリケーションハンドルを返します。
 *
 * @pre
 *  - システムアプレットとして実行中である
 *
 * @details
 *  本関数は開発専用です。
 *  ae ライブラリを経由せずに外部で起動されたアプリケーションを指すアプリケーションハンドルを取得します。
 *
 *  アプレットシステムの内部では、外部起動されたアプリケーションのハンドルのキューを持っており、
 *  本関数を呼ぶと、そのキューから一つハンドルを pop (取得・削除)し、返します。
 *  このキューが空のときには nn::ae::ApplicationHandle::GetInvalidHandle() と同じ値が返ります。
 *
 *  アプレットシステムが外部起動されたアプリケーションを検知し、ハンドルをキューに積んだ際には、
 *  Message_FloatingApplicationDetected が通知されます。
 *
 *  本関数で取得された有効なハンドルは、不要になったのちに CloseApplicationHandle() に渡して閉じる必要があります。
 *
 *  本関数で取得されたハンドルが指すアプリケーションは、自動的には Foreground 状態にはなりません。
 *  必要に応じて適切なタイミングで RequestApplicationGetForeground() を呼ぶ必要があります。
 *  開発用に、デバッガなどからアプリケーションの外部起動を Message_FloatingApplicationDetected で待機し、本関数でハンドルを取得し、
 *  必要に応じてユーザに確認を取るなどしたのち、RequestApplicationGetForeground() を呼ぶことを想定しています。
 */
ApplicationHandle PopFloatingApplicationHandle() NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 * @brief   アプリケーションハンドルをクローズします。
 *
 * @param[in]  handle   クローズしたいアプリケーションハンドル
 *
 * @pre
 *  - システムアプレットとして実行中である
 *  - handle.IsValid() == true である
 *
 * @details
 *  handle で指定されたアプリケーションハンドルをクローズします。
 *  handle に nn::ae::ApplicationHandle::GetInvalidHandle() を渡した場合、
 *  何もせずにリターンします。
 *
 */
void CloseApplicationHandle(ApplicationHandle handle) NN_NOEXCEPT;

/**
    @brief アプリケーションを起点とするライブラリアプレットに終了リクエストを送信し、終了を待ちます。

    @param[in] handle 対象のアプリケーションハンドル

    @pre システムアプレットとして実行中である
    @pre handle が有効なハンドルである

    @details
     handle で指定されるアプリケーションから起動されているライブラリアプレット、および、そのライブラリアプレットから起動されているライブラリアプレットに対し、
     終了リクエストを送信し、終了を待機します。
*/
void RequestExitLibraryAppletOfApplication(ApplicationHandle handle) NN_NOEXCEPT;

/**
    @brief アプリケーションを起点とするライブラリアプレットに終了リクエストを送信し、指定時間待機し、それでも終了しなかった場合には、強制終了します。

    @param[in] handle 対象のライブラリアプレット
    @param[in] timeout 強制終了までの時間

    @pre システムアプレットとして実行中である
    @pre handle が有効なハンドルである
    @pre timeout >= 0

    @return 対象のライブラリアプレットが時間内に終了すれば true を、終了せずに強制終了した場合には false を返します。

    @details
     handle で指定されるアプリケーションから起動されているライブラリアプレット、および、そのライブラリアプレットから起動されているライブラリアプレットに対し、
     終了リクエストを送信し timeout で指定した時間だけ待機します。
     この待機中に対象のライブラリアプレットが終了すれば true を返します。
     終了しなかった場合には、強制終了し、false を返します。
*/
bool RequestExitLibraryAppletOfApplicationOrTerminate(ApplicationHandle handle, TimeSpan timeout) NN_NOEXCEPT;

/**
    @brief AllUser のためのタグ型です。
*/
struct AllUserTag {};

/**
    @brief SetApplicationUsers() に設定するための定数型です。
*/
const AllUserTag AllUser{};

/**
    @brief 指定したアプリケーションを全ユーザが使用できることを宣言します。

    @param[in] handle 対象のアプリケーションハンドルを指定します。
    @param[in] allUser AllUser 定数を指定します。

    @return 処理の結果を返します。

    @details
     handle で指定されたアプリケーションを、全ユーザが使用することを宣言します。

     本関数は StartApplication() より先に呼んでください。
*/
Result SetApplicationUsers(ApplicationHandle handle, AllUserTag allUser) NN_NOEXCEPT;

/**
    @brief 指定したアプリケーションを指定したユーザだけが使用できることを宣言します。

    @param[in] handle 対象のアプリケーションハンドルを指定します。
    @param[in] uids ユーザの配列を指定します。
    @param[in] count uids の配列サイズを指定します。

    @return 処理の結果を返します。

    @details
     handle で指定されたアプリケーションを、uids[0] から uids[count - 1] に指定されたユーザが使用することを宣言します。

     本関数は StartApplication() より先に呼んでください。
*/
Result SetApplicationUsers(ApplicationHandle handle, const account::Uid uids[], int count) NN_NOEXCEPT;

/**
    @brief 指定したアプリケーションの権利が使用できることをチェックします。

    @param[in] handle 対象のアプリケーションハンドルを指定します。

    @return 指定したアプリケーションの権利が使用できるなら true を返し、そうでなければ false を返します。

    @details
     handle で指定されたアプリケーションの権利が現在使用できることをチェックします。

     ns の権利関連 API を適切に呼び出すなどして、指定したアプリケーションの権利が使用できる状態になった際には、
     必ず本関数が true を返すことを確認してください。

     特に、一旦権利を喪失し Message_ApplicationSuspendedByRightsError が通知され、
     これに対して権利を回復する処理を行った際には、必ず本関数が true を返すことを確認してください。
     この確認を行わなかった際には、以後に再度権利を喪失した際に Message_ApplicationSuspendedByRightsError が通知されない可能性があります。

     権利を回復する処理が完了しているにもかかわらず本関数が false を返した場合は、
     権利回復を行い本関数が呼び出されるまでの間に、再度権利を喪失したことになります。
     この場合は Message_ApplicationSuspendedByRightsError を受け取った場合と同等の処理を行うようにしてください。
*/
bool CheckApplicationRightsAvailability(ApplicationHandle handle) NN_NOEXCEPT;

/**
    @brief アプリケーションの起動要求元のアプレット情報を取得します。

    @param[in] handle 対象のアプリケーションハンドル

    @pre システムアプレットとして実行中である
    @pre handle が有効なハンドルである

    @details
     handle で指定されたアプリケーションの起動要求元のアプレット情報を返します。

*/
applet::ApplicationLaunchRequestInfo GetApplicationLaunchRequestInfo(ApplicationHandle handle) NN_NOEXCEPT;


//! @}

//-----------------------------------------------------------------------------

}} // namespace nn::ae

