﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/account/detail/account_InternalConfig.h>
#include <nn/account/detail/account_LocalStorage.h>
#include <nn/account/detail/account_Notifiable.h>
#include <nn/account/user/account_UserReference.h>
#include <nn/account/account_Config.h>
#include <nn/account/account_Types.h>

#include <nn/os/os_SdkMutex.h>
#include <nn/os/os_SystemEventTypes.h>
#include <nn/nn_Result.h>
#include <nn/util/util_Uuid.h>

namespace nn { namespace account { namespace user {

enum UserRegistryTag
{
    UserRegistryTag_Registration,
};

class UserRegistry
    : detail::Notifiable<UserRegistryTag, detail::UserRegistryNotifierCountMax>
{
    NN_DISALLOW_COPY(UserRegistry);
    NN_DISALLOW_MOVE(UserRegistry);

private:
    mutable os::SdkMutex m_Lock;
    bool m_Initialized {false};
    const detail::AbstractLocalStorage* m_pStorage {nullptr};

    util::Uuid m_InstanceId {util::InvalidUuid};

    UserRef m_Users[UserCountMax]; // ユーザー情報のストレージ
    detail::Counter m_Counters[UserCountMax * 2]; // 参照カウンタ
    int m_RegisteredCount {0}; // 登録済みのユーザー数
    Uid m_LastOpenedUser {InvalidUid};

    Uid m_PendingUsers[1]; // 作成中ユーザーのストレージ
    int m_PendingCount {0}; // 作成中ユーザー数

    // 作成中ユーザーのストレージから指定の Uid を削除する
    bool DeletePendingUserUnsafe(const Uid& uid) NN_NOEXCEPT;
    // プレイヤーの並び替え
    void RelocateUserUnsafe(int fromIndex, int toIndex) NN_NOEXCEPT;

    void StoreSaveData() const NN_NOEXCEPT;

public:
    using Notifiable::ReleaseSystemEvent;

    UserRegistry() NN_NOEXCEPT = default;
    Result Initialize(const detail::AbstractLocalStorage& storage) NN_NOEXCEPT;
    void ListPendingUsers(nn::account::Uid* outUsers, int count) const NN_NOEXCEPT;
    Result GetUserRef(UserRef* pOut, const Uid& uid) const NN_NOEXCEPT;

    // 一般公開 API
    int GetUserCount() const NN_NOEXCEPT;
    nn::Result GetUserExistence(bool* pOutExistence, const nn::account::Uid& uid) const NN_NOEXCEPT;
    nn::Result ListAllUsers(nn::account::Uid* outUsers, int count) const NN_NOEXCEPT;

    // 本体システム向け
    nn::Result GetUserRegistrationEvent(os::SystemEventType** pOutEvent) NN_NOEXCEPT;

    // アカウント管理向け
    nn::Result BeginUserRegistration(Uid* pOutUid) NN_NOEXCEPT;
    nn::Result BeginUserRegistrationAs(const Uid& uid) NN_NOEXCEPT;
    nn::Result CompleteUserRegistration(const nn::account::Uid& uid) NN_NOEXCEPT;
    nn::Result CancelUserRegistration(const nn::account::Uid& uid) NN_NOEXCEPT;
    nn::Result DeleteUser(const nn::account::Uid& uid) NN_NOEXCEPT;
    nn::Result SetUserPosition(const nn::account::Uid& uid, int position) NN_NOEXCEPT;
};

}}} // ~namespace nn::account::user
