﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/account/detail/account_AsyncContextImpl.h>
#include <nn/account/detail/account_IAsyncContext.sfdl.h>
#include <nn/account/nas/account_NasOperator.h>
#include <nn/account/nas/account_NasSessionPool.h>
#include <nn/account/nas/account_NasTypes.h>
#include <nn/account/user/account_UserReference.h>
#include <nn/account/account_Types.h>
#include <nn/account/account_TypesForSystemServices.h>

#include <type_traits>

#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_IServiceObject.h>
#include <nn/sf/sf_ObjectFactory.h>

namespace nn { namespace account {
class Executor;
}} // ~namespace nn::account

namespace nn { namespace account { namespace nas {

template <typename Allocator>
class NintendoAccountLinkageProcedureImpl
    : public sf::ISharedObject
{
    NN_DISALLOW_COPY(NintendoAccountLinkageProcedureImpl);

private:
    typedef sf::ObjectFactory<typename Allocator::Policy> Factory;

    const user::UserRef m_User;
    Allocator& m_Allocator;
    Executor& m_Executor;

    NasLinkageProcedure m_Procedure;

    class PostprocessTask
        : public detail::Executable<ExecutionResource>
    {
    public:
        typedef sf::SharedPointer<NintendoAccountLinkageProcedureImpl<Allocator>> ParentPtr;
    private:
        ParentPtr m_Parent;
    protected:
        virtual Result ExecuteImpl(const ExecutionResource& resource) NN_NOEXCEPT
        {
            return m_Parent->m_Procedure.ExecutePostprocess(resource, this);
        }
    public:
        explicit PostprocessTask(ParentPtr&& parent) NN_NOEXCEPT
            : m_Parent(std::move(parent))
        {
        }
    };

public:
    NintendoAccountLinkageProcedureImpl(Allocator& allocator, Executor& executor, const user::UserRef& user, NasOperator& nasOp) NN_NOEXCEPT
        : m_User(user)
        , m_Allocator(allocator)
        , m_Executor(executor)
        , m_Procedure(user, nasOp)
    {
    }
    Result IsNetworkServiceAccountReplaced(sf::Out<bool> pOut) const NN_NOEXCEPT
    {
        *pOut = m_Procedure.IsNetworkServiceAccountReplaced();
        NN_RESULT_SUCCESS;
    }
    Result PrepareAsync(sf::Out<sf::SharedPointer<detail::IAsyncContext>>) NN_NOEXCEPT
    {
        NN_RESULT_THROW(ResultNotSupported());
    }
    Result GetRequest(sf::Out<RequestUrl> pOutRequestUrl, sf::Out<CallbackUri> pOutCallbackUri) NN_NOEXCEPT
    {
        return m_Procedure.GetRequest(pOutRequestUrl.GetPointer(), pOutCallbackUri.GetPointer(), NintendoAccountAuthorizationPageTheme_Intro);
    }
    Result GetRequestWithTheme(sf::Out<RequestUrl> pOutRequestUrl, sf::Out<CallbackUri> pOutCallbackUri, int32_t themeRaw) NN_NOEXCEPT
    {
        NintendoAccountAuthorizationPageTheme theme;
        switch (themeRaw)
        {
        case NintendoAccountAuthorizationPageTheme_Register:
        case NintendoAccountAuthorizationPageTheme_Intro:
        case NintendoAccountAuthorizationPageTheme_EmailAuthentication:
        case NintendoAccountAuthorizationPageTheme_SimpleAuthentication:
            theme = static_cast<NintendoAccountAuthorizationPageTheme>(themeRaw);
            break;
        default:
            NN_RESULT_THROW(ResultValueOutOfRange());
        }
        return m_Procedure.GetRequest(pOutRequestUrl.GetPointer(), pOutCallbackUri.GetPointer(), theme);
    }
    Result ApplyResponse(const sf::InArray<char>&) NN_NOEXCEPT
    {
        NN_RESULT_THROW(ResultNotSupported());
    }
    Result ApplyResponseAsync(sf::Out<sf::SharedPointer<detail::IAsyncContext>> pOut, const sf::InArray<char>& response) NN_NOEXCEPT
    {
        NN_RESULT_DO(m_Procedure.ApplyResponse(response.GetData(), response.GetLength()));

        auto p = Factory::template CreateSharedEmplaced<detail::IAsyncContext, detail::AsyncTask<PostprocessTask, ExecutionResource>>(
            &m_Allocator, typename PostprocessTask::ParentPtr(this, true));
        NN_RESULT_THROW_UNLESS(p, ResultOutOfSessionObject());
        NN_RESULT_DO(p.GetImpl().Initialize(&m_Executor));
        *pOut = std::move(p);
        NN_RESULT_SUCCESS;
    }
    Result Suspend(sf::Out<detail::Uuid> pOutSessionId) NN_NOEXCEPT
    {
        return m_Procedure.Suspend(pOutSessionId.GetPointer());
    }
    Result Resume(const detail::Uuid& sessionId) NN_NOEXCEPT
    {
        return m_Procedure.Resume(sessionId);
    }
};

template <typename ProcedureType, typename Allocator>
class SimpleUserOAuthProcedureImplBase
    : public sf::ISharedObject
{
    NN_DISALLOW_COPY(SimpleUserOAuthProcedureImplBase);

private:
    typedef sf::ObjectFactory<typename Allocator::Policy> Factory;

    const user::UserRef m_User;
    Allocator& m_Allocator;
    Executor& m_Executor;

    ProcedureType m_Procedure;

    class PostprocessTask
        : public detail::Executable<ExecutionResource>
    {
    public:
        typedef sf::SharedPointer<SimpleUserOAuthProcedureImplBase<ProcedureType, Allocator>> ParentPtr;
    private:
        ParentPtr m_Parent;
    protected:
        virtual Result ExecuteImpl(const ExecutionResource& resource) NN_NOEXCEPT
        {
            return m_Parent->m_Procedure.ExecutePostprocess(resource, this);
        }
    public:
        explicit PostprocessTask(ParentPtr&& parent) NN_NOEXCEPT
            : m_Parent(std::move(parent))
        {
        }
    };

public:
    template <typename T0, typename T1>
    SimpleUserOAuthProcedureImplBase(Allocator& allocator, Executor& executor, const user::UserRef& user, T0&& t0, T1&& t1) NN_NOEXCEPT
        : m_User(user)
        , m_Allocator(allocator)
        , m_Executor(executor)
        , m_Procedure(std::forward<T0>(t0), std::forward<T1>(t1))
    {
    }
    Result PrepareAsync(sf::Out<sf::SharedPointer<detail::IAsyncContext>>) NN_NOEXCEPT
    {
        NN_RESULT_THROW(ResultNotSupported());
    }
    Result GetRequest(sf::Out<RequestUrl> pOutRequestUrl, sf::Out<CallbackUri> pOutCallbackUri) NN_NOEXCEPT
    {
        return m_Procedure.GetRequest(pOutRequestUrl.GetPointer(), pOutCallbackUri.GetPointer());
    }
    Result ApplyResponse(const sf::InArray<char>&) NN_NOEXCEPT
    {
        NN_RESULT_THROW(ResultNotSupported());
    }
    Result ApplyResponseAsync(sf::Out<sf::SharedPointer<detail::IAsyncContext>> pOut, const sf::InArray<char>& response) NN_NOEXCEPT
    {
        NN_RESULT_DO(m_Procedure.ApplyResponse(response.GetData(), response.GetLength()));

        auto p = Factory::template CreateSharedEmplaced<detail::IAsyncContext, detail::AsyncTask<PostprocessTask, ExecutionResource>>(
            &m_Allocator, typename PostprocessTask::ParentPtr(this, true));
        NN_RESULT_THROW_UNLESS(p, ResultOutOfSessionObject());
        NN_RESULT_DO(p.GetImpl().Initialize(&m_Executor));
        *pOut = std::move(p);
        NN_RESULT_SUCCESS;
    }
    Result Suspend(sf::Out<detail::Uuid> pOutSessionId) NN_NOEXCEPT
    {
        return m_Procedure.Suspend(pOutSessionId.GetPointer());
    }
    Result Resume(const detail::Uuid& sessionId) NN_NOEXCEPT
    {
        return m_Procedure.Resume(sessionId);
    }
};

template <typename Allocator>
using NintendoAccountCredentialUpdateProcedureImpl = SimpleUserOAuthProcedureImplBase<NasCredentialUpdateProcedure, Allocator>;
template <typename Allocator>
using NintendoAccountNnidLinkageProcedureImpl = SimpleUserOAuthProcedureImplBase<NasNnidLinkageProcedure, Allocator>;

template <typename Allocator>
class NintendoAccountApplicationAuthorizationProcedureImpl
    : public sf::ISharedObject
{
    NN_DISALLOW_COPY(NintendoAccountApplicationAuthorizationProcedureImpl);

private:
    typedef sf::ObjectFactory<typename Allocator::Policy> Factory;

    Allocator& m_Allocator;
    NasSessionPool<Allocator>& m_NasSessionPool;
    Executor& m_Executor;

    const user::UserRef m_User;

    detail::Uuid m_SessionId;
    NasApplicationAuthorizationProcedure* m_pProcedure;

    class PreprocessTask
        : public detail::Executable<ExecutionResource>
    {
    public:
        typedef sf::SharedPointer<NintendoAccountApplicationAuthorizationProcedureImpl<Allocator>> ParentPtr;
    private:
        ParentPtr m_Parent;
    protected:
        virtual Result ExecuteImpl(const ExecutionResource& resource) NN_NOEXCEPT
        {
            return m_Parent->m_pProcedure->ExecutePreprocess(resource, this);
        }
    public:
        explicit PreprocessTask(ParentPtr&& parent) NN_NOEXCEPT
            : m_Parent(std::move(parent))
        {
        }
    };

public:
    NintendoAccountApplicationAuthorizationProcedureImpl(
        Allocator& allocator, Executor& executor,
        const user::UserRef& user, NasSessionPool<Allocator>& nasSessionPool) NN_NOEXCEPT
        : m_Allocator(allocator)
        , m_NasSessionPool(nasSessionPool)
        , m_Executor(executor)
        , m_User(user)
        , m_SessionId(detail::InvalidUuid)
        , m_pProcedure(nullptr)
    {
    }
    ~NintendoAccountApplicationAuthorizationProcedureImpl() NN_NOEXCEPT
    {
        if (m_SessionId)
        {
            m_NasSessionPool.ReleaseNasApplicationAuthorizationSession(m_User, m_SessionId);
        }
    }
    Result Initialize(const detail::Uuid& sessionId)  NN_NOEXCEPT
    {
        NN_SDK_ASSERT(sessionId);
        NN_SDK_ASSERT(!m_SessionId);

        NN_RESULT_DO(m_NasSessionPool.AcquireNasApplicationAuthorizationSession(&m_pProcedure, m_User, sessionId));
        m_SessionId = sessionId;
        NN_RESULT_SUCCESS;
    }
    Result PrepareAsync(sf::Out<sf::SharedPointer<detail::IAsyncContext>> pOut) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(m_SessionId, ResultInvalidProtocolAccess());

        auto p = Factory::template CreateSharedEmplaced<detail::IAsyncContext, detail::AsyncTask<PreprocessTask, ExecutionResource>>(
            &m_Allocator, typename PreprocessTask::ParentPtr(this, true));
        NN_RESULT_THROW_UNLESS(p, ResultOutOfSessionObject());
        NN_RESULT_DO(p.GetImpl().Initialize(&m_Executor));
        *pOut = std::move(p);
        NN_RESULT_SUCCESS;
    }
    Result GetRequest(sf::Out<RequestUrl> pOutRequestUrl, sf::Out<CallbackUri> pOutCallbackUri) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(m_SessionId, ResultInvalidProtocolAccess());

        return m_pProcedure->GetRequest(pOutRequestUrl.GetPointer(), pOutCallbackUri.GetPointer());
    }
    Result ApplyResponse(const sf::InArray<char>& response) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(m_SessionId, ResultInvalidProtocolAccess());

        return m_pProcedure->ApplyResponse(response.GetData(), response.GetLength());
    }
    Result ApplyResponseAsync(sf::Out<sf::SharedPointer<detail::IAsyncContext>>, const sf::InArray<char>&) NN_NOEXCEPT
    {
        NN_RESULT_THROW(ResultNotSupported());
    }
    Result Suspend(sf::Out<detail::Uuid> pOutSessionId) NN_NOEXCEPT
    {
        NN_RESULT_THROW(ResultNotSupported());
    }
};

template <typename SessionControlPolicy, typename Allocator>
class NintendoAccountGuestLoginProcedureImplBase
    : public sf::ISharedObject
{
    NN_DISALLOW_COPY(NintendoAccountGuestLoginProcedureImplBase);

private:
    typedef sf::ObjectFactory<typename Allocator::Policy> Factory;

    Allocator& m_Allocator;
    NasSessionPool<Allocator>& m_NasSessionPool;
    Executor& m_Executor;

    detail::Uuid m_SessionId;
    typename SessionControlPolicy::ProcedureType* m_pProcedure;

    class PreprocessTask
        : public detail::Executable<ExecutionResource>
    {
    public:
        typedef sf::SharedPointer<NintendoAccountGuestLoginProcedureImplBase<SessionControlPolicy, Allocator>> ParentPtr;
    private:
        ParentPtr m_Parent;
    protected:
        virtual Result ExecuteImpl(const ExecutionResource& resource) NN_NOEXCEPT
        {
            return m_Parent->m_pProcedure->ExecutePreprocess(resource, this);
        }
    public:
        explicit PreprocessTask(ParentPtr&& parent) NN_NOEXCEPT
            : m_Parent(std::move(parent))
        {
        }
    };

    class PostprocessTask
        : public detail::Executable<ExecutionResource>
    {
    public:
        typedef sf::SharedPointer<NintendoAccountGuestLoginProcedureImplBase<SessionControlPolicy, Allocator>> ParentPtr;
    private:
        ParentPtr m_Parent;
    protected:
        virtual Result ExecuteImpl(const ExecutionResource& resource) NN_NOEXCEPT
        {
            return m_Parent->m_pProcedure->ExecutePostprocess(resource, this);
        }
    public:
        explicit PostprocessTask(ParentPtr&& parent) NN_NOEXCEPT
            : m_Parent(std::move(parent))
        {
        }
    };

public:
    NintendoAccountGuestLoginProcedureImplBase(
        Allocator& allocator, Executor& executor,
        NasSessionPool<Allocator>& nasSessionPool) NN_NOEXCEPT
        : m_Allocator(allocator)
        , m_NasSessionPool(nasSessionPool)
        , m_Executor(executor)
        , m_SessionId(detail::InvalidUuid)
        , m_pProcedure(nullptr)
    {
    }
    ~NintendoAccountGuestLoginProcedureImplBase() NN_NOEXCEPT
    {
        if (m_SessionId)
        {
            SessionControlPolicy::template Release<Allocator>(m_SessionId, m_NasSessionPool);
        }
    }
    Result Initialize(const detail::Uuid& sessionId)  NN_NOEXCEPT
    {
        NN_SDK_ASSERT(sessionId);
        NN_SDK_ASSERT(!m_SessionId);

        NN_RESULT_DO(SessionControlPolicy::template Acquire<Allocator>(&m_pProcedure, sessionId, m_NasSessionPool));
        m_SessionId = sessionId;
        NN_RESULT_SUCCESS;
    }
    Result PrepareAsync(sf::Out<sf::SharedPointer<detail::IAsyncContext>> pOut) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(m_SessionId, ResultInvalidProtocolAccess());

        auto p = Factory::template CreateSharedEmplaced<detail::IAsyncContext, detail::AsyncTask<PreprocessTask, ExecutionResource>>(
            &m_Allocator, typename PreprocessTask::ParentPtr(this, true));
        NN_RESULT_THROW_UNLESS(p, ResultOutOfSessionObject());
        NN_RESULT_DO(p.GetImpl().Initialize(&m_Executor));
        *pOut = std::move(p);
        NN_RESULT_SUCCESS;
    }
    Result GetRequest(sf::Out<RequestUrl> pOutRequestUrl, sf::Out<CallbackUri> pOutCallbackUri) NN_NOEXCEPT
    {
        return m_pProcedure->GetRequest(pOutRequestUrl.GetPointer(), pOutCallbackUri.GetPointer());
    }
    Result ApplyResponse(const sf::InArray<char>&) NN_NOEXCEPT
    {
        NN_RESULT_THROW(ResultNotSupported());
    }
    Result ApplyResponseAsync(sf::Out<sf::SharedPointer<detail::IAsyncContext>> pOut, const sf::InArray<char>& response) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(m_SessionId, ResultInvalidProtocolAccess());

        NN_RESULT_DO(m_pProcedure->ApplyResponse(response.GetData(), response.GetLength()));

        auto p = Factory::template CreateSharedEmplaced<detail::IAsyncContext, detail::AsyncTask<PostprocessTask, ExecutionResource>>(
            &m_Allocator, typename PostprocessTask::ParentPtr(this, true));
        NN_RESULT_THROW_UNLESS(p, ResultOutOfSessionObject());
        NN_RESULT_DO(p.GetImpl().Initialize(&m_Executor));
        *pOut = std::move(p);
        NN_RESULT_SUCCESS;
    }
    Result Suspend(sf::Out<detail::Uuid> pOutSessionId) NN_NOEXCEPT
    {
        NN_RESULT_THROW(ResultNotSupported());
    }

    // 固有の機能
    Result GetAccountId(sf::Out<NetworkServiceAccountId> pOutId) NN_NOEXCEPT
    {
        return m_pProcedure->GetNetworkServiceAccountId(pOutId.GetPointer());
    }
    Result GetLinkedNintendoAccountId(sf::Out<NintendoAccountId> pOut) NN_NOEXCEPT
    {
        return m_pProcedure->GetLinkedNintendoAccountId(pOut.GetPointer());
    }
    Result GetNickname(const sf::OutArray<char>& nickname) NN_NOEXCEPT
    {
        return m_pProcedure->GetNickname(nickname.GetData(), nickname.GetLength());
    }
    Result GetProfileImage(sf::Out<uint32_t> pOutActualSize, const sf::OutBuffer& pOutImage) NN_NOEXCEPT
    {
        size_t sizeActual;
        NN_RESULT_DO(m_pProcedure->GetProfileImage(&sizeActual, pOutImage.GetPointerUnsafe(), pOutImage.GetSize()));
        NN_SDK_ASSERT(sizeActual <= ProfileImageBytesMax);
        *pOutActualSize = static_cast<uint32_t>(sizeActual);
        NN_RESULT_SUCCESS;
    }
};

struct NasGuestLoginSessionControlPolicy
{
    typedef NasGuestLoginProcedure ProcedureType;

    template <typename Allocator>
    static Result Acquire(NasGuestLoginProcedure** ppOut, const detail::Uuid& sessionId, NasSessionPool<Allocator>& pool) NN_NOEXCEPT
    {
        return pool.AcquireNasGuestLoginSession(ppOut, sessionId);
    }
    template <typename Allocator>
    static void Release(const detail::Uuid& sessionId, NasSessionPool<Allocator>& pool) NN_NOEXCEPT
    {
        pool.ReleaseNasGuestLoginSession(sessionId);
    }
};
template <typename Allocator>
using NintendoAccountGuestLoginProcedureImpl = NintendoAccountGuestLoginProcedureImplBase<NasGuestLoginSessionControlPolicy, Allocator>;

struct NasFloatingRegistrationSessionControlPolicy
{
    typedef NasFloatingRegistrationProcedure ProcedureType;

    template <typename Allocator>
    static Result Acquire(NasFloatingRegistrationProcedure** ppOut, const detail::Uuid& sessionId, NasSessionPool<Allocator>& pool) NN_NOEXCEPT
    {
        return pool.AcquireNasFloatingRegistrationSession(ppOut, sessionId);
    }
    template <typename Allocator>
    static void Release(const detail::Uuid& sessionId, NasSessionPool<Allocator>& pool) NN_NOEXCEPT
    {
        pool.ReleaseNasFloatingRegistrationSession(sessionId);
    }
};
template <typename Allocator>
using NintendoAccountFloatingRegistrationProcedureImpl = NintendoAccountGuestLoginProcedureImplBase<NasFloatingRegistrationSessionControlPolicy, Allocator>;

}}} // namespace nn::account::nas
